<?php
/**
 *
 * @category        admin
 * @package         admintools
 * @author          WebsiteBaker Project
 * @copyright       Ryan Djurovich
 * @copyright       WebsiteBaker Org. e.V., Jacobi22
 * @link            http://websitebaker.org/
 * @license         http://www.gnu.org/licenses/gpl.html
 * @platform        WebsiteBaker 2.13.1
 * @requirements    PHP 8.0.0 and higher
 * @version         $Id: thumb.php 4 2025-08-08 07:53:04Z Uwe $
 * @filesource      $HeadURL: file:///G:/SVN_Projekte/WB_Entwicklung/branches/WB_Neuentwicklung/admin/media/thumb.php $
 * @lastmodified    $Date: 2025-08-08 09:53:04 +0200 (Fr, 08 Aug 2025) $
 *
 */

// Vereinfachter und sicherer Bootstrap
if (!defined('SYSTEM_RUN')) {
    require(dirname(__DIR__, 2) . '/config.php');
}

if (isset($_GET['img']) && isset($_GET['t'])) {
    $image = $_GET['img']; // addslashes ist hier nicht nötig/sinnvoll
    $type = intval($_GET['t']);

    // Sicherheitsprüfung: Verhindert Path Traversal und stellt sicher, dass wir im MEDIA-Verzeichnis bleiben
    $sFilename = WB_PATH . MEDIA_DIRECTORY . '/' . $image;
    if (!file_exists($sFilename) || strpos(realpath($sFilename), realpath(WB_PATH . MEDIA_DIRECTORY)) !== 0) {
        http_response_code(404);
        die("Datei existiert nicht oder Zugriff verweigert.");
    }

    $extension = strtolower(pathinfo($image, PATHINFO_EXTENSION));

    // Thumbnail-Parameter definieren
    $imageMaxWidth = ($type == 1) ? 22 : 300;
    $imageMaxHeight = ($type == 1) ? 22 : 300;
    $borderWidth = ($type == 2) ? 5 : 0; // Nur für die große Vorschau
    $borderColor = '#89bbe0';

    try {
        // Bevorzuge Imagick, da es mehr Formate (AVIF, SVG, PDF etc.) unterstützt
        if (class_exists('Imagick')) {
            $imagick = new \Imagick();
            // Einstellungen für Vektorformate und PDF
            $imagick->setOption('svg:include-elements', 'all');
            $imagick->setBackgroundColor(new \ImagickPixel('transparent'));
            
            // Lese das Bild ein
            if ($extension === 'pdf') {
                $imagick->setResolution(150, 150); // Auflösung für die PDF-Rasterung
                $imagick->readImage($sFilename . '[0]'); // Nur die erste Seite
            } else {
                $imagick->readImage($sFilename);
            }
            
            // Originalmaße aus dem geladenen Bild holen
            $width_orig = $imagick->getImageWidth();
            $height_orig = $imagick->getImageHeight();
            
            if (!$width_orig || !$height_orig) {
                 throw new Exception("Bildmaße konnten nicht mit Imagick gelesen werden.");
            }
            
            // --- Seitenverhältnis-Logik ---
            $ratio = $width_orig / $height_orig;
            if (($imageMaxWidth / $imageMaxHeight) > $ratio) {
                $newHeight = $imageMaxHeight;
                $newWidth = $newHeight * $ratio;
            } else {
                $newWidth = $imageMaxWidth;
                $newHeight = $newWidth / $ratio;
            }
            $newWidth = round($newWidth);
            $newHeight = round($newHeight);

            // Bild auf die neue Größe skalieren (Seitenverhältnis bleibt erhalten)
            $imagick->resizeImage($newWidth, $newHeight, \Imagick::FILTER_LANCZOS, 1);
            
            // Rahmen nur für die große Vorschau hinzufügen
            if ($type == 2 && $borderWidth > 0) {
                $imagick->borderImage(new \ImagickPixel($borderColor), $borderWidth, $borderWidth);
            }

            // Für konsistente Ausgabe und Transparenzunterstützung als PNG ausgeben
            $imagick->setImageFormat('png');

            header('Content-Type: image/png');
            echo $imagick->getImageBlob();
            
            $imagick->clear();
            $imagick->destroy();

        } else {
            // Fallback auf GD-Bibliothek
            
            // Originalmaße mit getimagesize holen
            // @ unterdrückt Warnungen für nicht unterstützte Formate
            list($width_orig, $height_orig) = @getimagesize($sFilename);
            if (!$width_orig || !$height_orig) {
                // AVIF wird hier wahrscheinlich fehlschlagen, wenn GD es nicht unterstützt
                throw new Exception("Bildmaße konnten nicht gelesen werden oder Format wird von GD nicht unterstützt.");
            }

            // --- Seitenverhältnis-Logik (identisch zu oben) ---
            $ratio = $width_orig / $height_orig;
            if (($imageMaxWidth / $imageMaxHeight) > $ratio) {
                $newHeight = $imageMaxHeight;
                $newWidth = $newHeight * $ratio;
            } else {
                $newWidth = $imageMaxWidth;
                $newHeight = $newWidth / $ratio;
            }
            $newWidth = round($newWidth);
            $newHeight = round($newHeight);

            // Die Gesamtgröße der Leinwand inkl. Rahmen
            $canvasWidth = $newWidth + ($borderWidth * 2);
            $canvasHeight = $newHeight + ($borderWidth * 2);

            // Quellbild laden
            $src_image = null;
            switch ($extension) {
                case 'jpg': case 'jpeg': $src_image = @imagecreatefromjpeg($sFilename); break;
                case 'png': $src_image = @imagecreatefrompng($sFilename); break;
                case 'gif': $src_image = @imagecreatefromgif($sFilename); break;
                case 'webp': $src_image = @imagecreatefromwebp($sFilename); break;
                case 'bmp': $src_image = @imagecreatefrombmp($sFilename); break;
                // AVIF wird von Standard-GD nicht unterstützt, only bei Imagick
                default: throw new Exception("Format wird von der GD-Bibliothek nicht unterstützt.");
            }

            if (!$src_image) {
                throw new Exception("Bildressource konnte nicht erstellt werden.");
            }

            // Leinwand (Canvas) erstellen
            $canvas = imagecreatetruecolor($canvasWidth, $canvasHeight);
            
            // Leinwand füllen (entweder mit Rahmenfarbe oder transparent)
            if ($type == 2 && $borderWidth > 0) {
                $rgb = sscanf($borderColor, "#%02x%02x%02x");
                $bgColor = imagecolorallocate($canvas, $rgb[0], $rgb[1], $rgb[2]);
                imagefill($canvas, 0, 0, $bgColor);
            } else {
                // Transparenz für PNGs sicherstellen
                imagealphablending($canvas, false);
                imagesavealpha($canvas, true);
                $transparent = imagecolorallocatealpha($canvas, 255, 255, 255, 127);
                imagefilledrectangle($canvas, 0, 0, $canvasWidth, $canvasHeight, $transparent);
            }

            // Das skalierte Bild auf die Leinwand kopieren (mit Versatz für den Rahmen, aktuell 5px)
            imagecopyresampled(
                $canvas, $src_image,
                $borderWidth, $borderWidth, // Ziel-X, Ziel-Y (auf der Leinwand)
                0, 0,                      // Quell-X, Quell-Y
                $newWidth, $newHeight,     // Neue Breite, Neue Höhe
                $width_orig, $height_orig  // Originale Breite, Höhe
            );
            
            header('Content-Type: image/png');
            imagepng($canvas);
            
            imagedestroy($src_image);
            imagedestroy($canvas);
        }

    } catch (Exception $e) {
        http_response_code(500);
    }
}