<?php
/**
 * This source file is part of the open source project
 * ExpressionEngine (https://expressionengine.com)
 *
 * @link      https://expressionengine.com/
 * @copyright Copyright (c) 2003-2023, Packet Tide, LLC (https://www.packettide.com)
 * @license   https://expressionengine.com/license Licensed under Apache License, Version 2.0
 */

/**
 * Benchmark Class
 *
 * This class enables you to mark points and calculate the time difference
 * between them.  Memory consumption can also be displayed.
 */
class EE_Benchmark
{
    public $marker = array();

    /**
     * Set a benchmark marker
     *
     * Multiple calls to this function can be made so that several
     * execution points can be timed
     *
     * @access	public
     * @param	string	$name	name of the marker
     * @return	void
     */
    public function mark($name)
    {
        $this->marker[$name] = microtime();
    }

    /**
     * Calculates the time difference between two marked points.
     *
     * If the first parameter is empty this function instead returns the
     * {elapsed_time} pseudo-variable. This permits the full system
     * execution time to be shown in a template. The output class will
     * swap the real value for this variable.
     *
     * @access	public
     * @param	string	a particular marked point
     * @param	string	a particular marked point
     * @param	integer	the number of decimal places
     * @return	mixed
     */
    public function elapsed_time($point1 = '', $point2 = '', $decimals = 4)
    {
        if ($point1 == '') {
            return '{elapsed_time}';
        }

        if (! isset($this->marker[$point1])) {
            return '';
        }

        if (! isset($this->marker[$point2])) {
            $this->marker[$point2] = microtime();
        }

        list($sm, $ss) = explode(' ', $this->marker[$point1]);
        list($em, $es) = explode(' ', $this->marker[$point2]);

        return number_format(($em + $es) - ($sm + $ss), $decimals);
    }

    /**
     * Memory Usage
     *
     * This function returns the {memory_usage} pseudo-variable.
     * This permits it to be put it anywhere in a template
     * without the memory being calculated until the end.
     * The output class will swap the real value for this variable.
     *
     * @access	public
     * @return	string
     */
    public function memory_usage()
    {
        return '{memory_usage}';
    }

    /**
     * Get an array of current Benchmark timing data
     *
     * @return array  Benchmark timing array: [label => microtime, ...]
     **/
    public function getBenchmarkTimings()
    {
        $timings = array();
        foreach ($this->marker as $key => $val) {
            // We match the "end" marker so that the list ends
            // up in the order that it was defined
            if (preg_match("/(.+?)_end/i", $key, $match)) {
                if (isset($this->marker[$match[1] . '_end']) and isset($this->marker[$match[1] . '_start'])) {
                    $timings[ucwords(str_replace(array('_', '-'), ' ', $match[1]))] = $this->elapsed_time($match[1] . '_start', $key);
                }
            }
        }

        return $timings;
    }
}

// END CI_Benchmark class

// EOF
