<?php defined('FLATBOARD') or die('Flatboard Community.');
/*
 * Project name: Flatboard
 * Project URL: https://flatboard.org
 * Author: Frédéric Kaplon and contributors
 * All Flatboard code is released under the MIT license.
*/

class Util
{
    // Constructeur protégé, rien à initialiser ici
    protected function __construct()
    {
        // Nothing here
    }

    /**
     * Vérifie si une variable GET est définie et est une chaîne.
     *
     * @param string $name Le nom de la variable GET à vérifier.
     * @return bool Retourne vrai si la variable GET est définie et est une chaîne, sinon faux.
     */
    public static function isGET($name)
    {
        return isset($_GET[$name]) && is_string($_GET[$name]);
    }

    /**
     * Vérifie si une variable POST est définie et est une chaîne.
     *
     * @param string $name Le nom de la variable POST à vérifier.
     * @return bool Retourne vrai si la variable POST est définie et est une chaîne, sinon faux.
     */
    public static function isPOST($name)
    {
        return isset($_POST[$name]) && is_string($_POST[$name]);
    }

    /**
     * Vérifie si une entrée GET est valide pour un type donné.
     *
     * @param string $type Le type d'entrée à valider.
     * @param string $name Le nom de la variable GET à vérifier.
     * @return bool Retourne vrai si l'entrée GET est valide, sinon faux.
     */
    public static function isGETValidEntry($type, $name)
    {
        return self::isGET($name) && flatDB::isValidEntry($type, $_GET[$name]);
    }

    /**
     * Vérifie si un hook GET est valide.
     *
     * @param string $hook Le nom du hook à valider.
     * @param string $name Le nom de la variable GET à vérifier.
     * @return bool Retourne vrai si le hook GET est valide, sinon faux.
     */
    public static function isGETValidHook($hook, $name)
    {
        return self::isGET($name) && Plugin::isValidHook($hook, $_GET[$name]);
    }
	
	/**
	 * Fonction qui extrait les paramètres de l'URL à partir de PATH_INFO.
	 *
	 * @return array Un tableau associatif contenant les paramètres extraits de l'URL.
	 */
	public static function fURL()
	{
	    $out = array();
	
	    // Vérifie si PATH_INFO est défini
	    if (isset($_SERVER['PATH_INFO'])) {
	        // Extrait les segments de l'URL
	        $info = explode('/', trim($_SERVER['PATH_INFO'], '/'));
	        $infoNum = count($info);
	
	        // Parcourt les segments de l'URL par paires
	        for ($i = 0; $i < $infoNum; $i += 2) {
	            // Vérifie que l'index est valide et que le segment n'est pas vide
	            if (isset($info[$i]) && $info[$i] !== '') {
	                // Assigne la valeur suivante ou une chaîne vide si elle n'existe pas
	                $out[$info[$i]] = isset($info[$i + 1]) ? $info[$i + 1] : '';
	            }
	        }
	    }
	
	    return $out;
	}

	/**
	 * Gets the base URL
	 *
	 * <code>
	 *     echo Util::baseURL();
	 * </code>
	 *
	 * @return string La base URL du site.
	 */
	public static function baseURL()
	{
	    // Liste des fichiers à exclure de l'URL
	    $excludedFiles = array(
	        'add.php', 'auth.php', 'config.php', 'delete.php', 
	        'edit.php', 'feed.php', 'index.php', 'install.php', 
	        'search.php', 'view.php', 'download.php', 'blog.php', 
	        'forum.php'
	    );
	
	    // Remplace les fichiers exclus par une chaîne vide
	    $siteUrl = str_replace($excludedFiles, '', $_SERVER['SCRIPT_NAME']);
	
	    // Détermine le schéma (http ou https)
	    $https = (isset($_SERVER['HTTPS']) && strtolower($_SERVER['HTTPS']) === 'on') ? 'https://' : 'http://';    
	
	    // Construit l'URL complète
	    $siteUrl = $https . $_SERVER['HTTP_HOST'] . $siteUrl;
	
	    return $siteUrl;
	}

	/**
	 * Gets the current URL.
	 *
	 * <code>
	 *     echo Util::getCurrent();
	 * </code>
	 *
	 * @return string L'URL actuelle.
	 */
	public static function getCurrent()
	{
	    // Détermine le schéma (http ou https)
	    $https = !empty($_SERVER['HTTPS']) && strtolower($_SERVER['HTTPS']) === 'on' ? 'https://' : 'http://';
	
	    // Construit l'URL complète
	    $currentUrl = $https . $_SERVER['SERVER_NAME'] . $_SERVER['REQUEST_URI'];
	
	    return $currentUrl;
	}
		
	/**
	 * Redirige vers une nouvelle URL.
	 *
	 * @param string $loc Le chemin relatif ou absolu vers lequel rediriger.
	 */
	public static function redirect($loc)
	{
	    // Envoie l'en-tête de redirection
	    header('Location: ' . Util::baseURL() . $loc);
	    exit; // Termine le script après la redirection
	}
	
	/**
	 * Calcule le numéro de page d'un élément dans une liste.
	 *
	 * @param mixed $item L'élément à rechercher.
	 * @param array $items La liste des éléments.
	 * @return int Le numéro de page de l'élément.
	 */
	public static function onPage($item, $items)
	{
	    // Recherche l'index de l'élément et calcule le numéro de page
	    return (int) (array_search($item, array_values($items), true) / 8) + 1;
	}
	
	/**
	 * Formate un nombre entier en notation abrégée.
	 *
	 * @param int $int Le nombre à formater.
	 * @return string Le nombre formaté.
	 */
	public static function shortNum($int)
	{
	    if ($int < 1000) {
	        return intval($int); // Retourne le nombre entier si inférieur à 1000
	    } else {
	        return round($int / 1000, 1) . 'K'; // Retourne le nombre en milliers avec un 'K'
	    }
	}
	
	/**
	 * Convertit un identifiant de date en une chaîne de date formatée.
	 *
	 * @param string $id L'identifiant contenant la date.
	 * @param string $pattern Le format de date à utiliser (par défaut 'Y/m/d H:i').
	 * @param bool $cooldate Indique si le format "cool" doit être utilisé pour les dates récentes.
	 * @return string La date formatée.
	 */
	public static function toDate($id, $pattern = 'Y/m/d H:i', $cooldate = true)
	{
		global $lang, $config;

		// Convertit la sous-chaîne de l'identifiant en timestamp
		$timestamp = strtotime(substr($id, 0, 16));
		$diff = time() - $timestamp;

		// Vérifie si le format de date "cool" doit être utilisé
		if ($pattern === $config['date_format'] && $cooldate && $diff < 604800) { // 1 semaine
			$periods = array(86400 => $lang['day'], 3600 => $lang['hour'], 60 => $lang['minute'], 1 => $lang['second']);
			foreach ($periods as $key => $value) {
				if ($diff >= $key) {
					$num = (int) ($diff / $key);
					if (defined('TIMESTAMP') && TIMESTAMP) {
						$date = $num . ' ' . $value . ($num > 1 ? $lang['plural'] : '') . ' ' . $lang['ago'];
					} else {
						$date = $lang['ago'] . ' ' . $num . ' ' . $value . ($num > 1 ? $lang['plural'] : '');
					}
					break; // Sortir de la boucle après avoir trouvé la première période
				}
			}
		} else {
			// Pour les dates en français, on utilise IntlDateFormatter si disponible
			if ($config['lang'] === 'fr-FR') {
				$dateID = substr($id, 0, 16);
				$dateTimeObj = new DateTime($dateID, new DateTimeZone('Europe/Paris'));
				if (class_exists('IntlDateFormatter')) {
					// Formate la date selon les préférences
					$dateFormatted = IntlDateFormatter::formatObject(
						$dateTimeObj,
						'eee d MMMM y à HH:mm',
						'fr'
					);
					$date = ucwords($dateFormatted);
				} else {
					// Si IntlDateFormatter n'est pas disponible, utiliser un format de date par défaut
					$date = date($pattern, $timestamp);
				}
			} else {
				// Format de date par défaut
				$date = date($pattern, $timestamp);
			}
		}

		return $date;
	}
	
	/**
	 * Vérifie la présence de la mention de copyright dans le fichier de langue.
	 *
	 * <code>
	 *     echo Util::checkCopy($langFile);
	 * </code>
	 *
	 * @param string $langFile Le chemin vers le fichier de langue à vérifier.
	 * @return void
	 */
	public static function checkCopy($langFile)
	{
	    global $lang, $config;
	
	    // Vérifie si le contenu du fichier de langue contient la mention de copyright
	    if (strpos(file_get_contents($langFile), 'false;">Flatboard') === false) {
	        // Si la mention n'est pas trouvée, affiche un message d'erreur
	        die('<h1><a href="https://github.com/Fred89/flatboard/blob/master/LICENSE">MIT License!</a></h1>' .
	            'Veuillez écrire dans le fichier "<i>' . LANG_DIR . $config['lang'] . '.php</i>" cette chaîne : ' .
	            base64_decode('PGRpdiBzdHlsZT0idGV4dC1hbGlnbjpsZWZ0O2NvbG9yOmJsYWNrOyBiYWNrZ3JvdW5kLWNvbG9yOndoaXRlOyBwYWRkaW5nOjAuNWVtOyBvdmVyZmxvdzphdXRvO2ZvbnQtc2l6ZTpzbWFsbDsgZm9udC1mYW1pbHk6bW9ub3NwYWNlOyAiPjxzcGFuIHN0eWxlPSJjb2xvcjojZGQyNDAwOyI+J3Bvd2VyZWQnPC9zcGFuPiA8c3BhbiBzdHlsZT0iY29sb3I6IzAwNmZmODsiPj0mZ3Q7PC9zcGFuPiA8c3BhbiBzdHlsZT0iY29sb3I6I2RkMjQwMDsiPidDcmVhdGVkIHdpdGggJmx0O2EgaHJlZj0mcXVvdDtodHRwOi8vZmxhdGJvYXJkLmZyZWUuZnImcXVvdDsgb25jbGljaz0mcXVvdDt3aW5kb3cub3Blbih0aGlzLmhyZWYpOyByZXR1cm4gZmFsc2U7JnF1b3Q7Jmd0O0ZsYXRib2FyZCc8L3NwYW4+IDxzcGFuIHN0eWxlPSJjb2xvcjojMDA2ZmY4OyI+Ljwvc3Bhbj4oPHNwYW4gc3R5bGU9ImNvbG9yOiM0MDAwODA7Ij5kZWZpbmVkPC9zcGFuPig8c3BhbiBzdHlsZT0iY29sb3I6I2RkMjQwMDsiPidGTEFUQk9BUkRfUFJPJzwvc3Bhbj4pPzxzcGFuIHN0eWxlPSJjb2xvcjojZGQyNDAwOyI+JyBQcm9uPC9zcGFuPjo8c3BhbiBzdHlsZT0iY29sb3I6I2RkMjQwMDsiPicnPC9zcGFuPik8c3BhbiBzdHlsZT0iY29sb3I6IzAwNmZmODsiPi48L3NwYW4+IDxzcGFuIHN0eWxlPSJjb2xvcjojZGQyNDAwOyI+JyZsdDsvYSZndDsgYW5kICZsdDtpIGNsYXNzPSZxdW90O2ZhIGZhLWhlYXJ0JnF1b3Q7Jmd0OyZsdDsvaSZndDsuJzwvc3Bhbj4sPC9kaXY+')
	            . '<p>Si vous avez des questions, veuillez contacter :<br/>' .
	            '- E-mail : <a href="mailto:stradfred@gmail.com">stradfred@gmail.com</a><br/>' .
	            '- Forum : <a href="https://flatboard.org/view.php/forum/general-questions">support</a></p>');
	    }
	}
			
	/**
	 * Obtient la taille d'un fichier en format lisible (B, KB, MB, GB, TB).
	 *
	 * @param string $file Chemin du fichier
	 * @param int $digits Chiffres à afficher
	 * @return string|bool Taille (KB, MB, GB, TB) ou false si le fichier n'existe pas
	 */
	public static function getFilesize($file, $digits = 2) {
	    if (!file_exists($file)) {
	        return false; // Retourne false si le fichier n'existe pas
	    }
	
	    $bytes = filesize($file);
	    if ($bytes < 1024) return $bytes . ' B';
	    elseif ($bytes < 1048576) return round($bytes / 1024, $digits) . ' KB';
	    elseif ($bytes < 1073741824) return round($bytes / 1048576, $digits) . ' MB';
	    elseif ($bytes < 1099511627776) return round($bytes / 1073741824, $digits) . ' GB';
	    else return round($bytes / 1099511627776, $digits) . ' TB';    
	}
	
	/**
	 * Convertit une taille en octets en une chaîne lisible (B, KB, MB, GB, TB).
	 *
	 * @param int $size Taille en octets
	 * @return string Taille formatée
	 */
	public static function sizeConversion($size) {
	    if ($size < 0) {
	        return '0 B'; // Retourne '0 B' si la taille est négative
	    }
	
	    $unit = array('B', 'KB', 'MB', 'GB', 'TB', 'PB');
	    $i = floor(log($size, 1024)); // Calcule l'indice de l'unité
	    return round($size / pow(1024, $i), 2) . ' ' . $unit[$i];
	}
	
	/**
	 * Méthode qui contrôle la mémoire utilisée.
	 *
	 * @param string $type Type de contrôle de la mémoire (available, peak ou current usage)
	 * @return string Taille de la mémoire formatée
	 */
	public static function getMemory($type = 'usage') {
	    switch ($type) {
	        case 'available':
	            $memoryAvailable = filter_var(ini_get("memory_limit"), FILTER_SANITIZE_NUMBER_INT);
	            $memoryAvailable = $memoryAvailable * 1024 * 1024; // Convertit en octets
	            $size = (int) $memoryAvailable;
	            break;
	        case 'peak':
	            $size = (int) memory_get_peak_usage(true);
	            break;
	        case 'usage':
	            $size = (int) memory_get_usage(true);
	            break;
	        default:
	            $size = 0;
	            break;
	    }
	
	    return Util::sizeConversion($size);
	}
	
	/**
	 * Méthode qui vérifie les modules PHP installés sur le serveur (installation).
	 *
	 * @param array $modules Modules à contrôler
	 * @return string|null Message d'erreur ou null si tous les modules sont installés
	 */
	public static function checkModules($modules = array('mbstring', 'json', 'gd', 'dom')) {
	    $missing = [];
	    foreach ($modules as $module) {
	        if (!extension_loaded($module)) {
	            $errorText = 'PHP module <b>' . $module . '</b> is not installed.';
	            error_log('[ERROR] ' . $errorText, 0);
	            $missing[] = $errorText; // Ajoute le message d'erreur à la liste
	        }
	    }
	
	    if (!empty($missing)) {
	        return 'PHP modules missing:<br>' . implode('<br>', $missing);
	    }
	
	    return null; // Retourne null si tous les modules sont installés
	}
					
	/**
	 * Récupère une chaîne de langue formatée à partir des arguments fournis.
	 *
	 * @param string $format Le format de la chaîne, contenant des mots séparés par des espaces.
	 * @return string La chaîne formatée.
	 */
	public static function lang($format)
	{
	    global $lang;
	
	    // Récupère tous les arguments passés à la fonction
	    $argList = func_get_args();
	    $wordList = array();
	
	    // Explose le format en mots et remplace chaque mot par sa traduction
	    foreach (explode(' ', $format) as $word) {
	        $wordList[] = isset($lang[$word]) ? $lang[$word] : $word; // Utilise la traduction si elle existe
	    }
	
	    // Formate la chaîne avec les arguments supplémentaires
	    return vsprintf(implode($lang['useSpace'] ? ' ' : '', $wordList), array_slice($argList, 1));
	}
		
	/**
	 * Méthode qui teste si la fonction php mail est disponible
	 *
	 * @param	io			affiche à l'écran le résultat du test si à VRAI
	 * @param	format		format d'affichage
	 * @return	boolean		retourne vrai si la fonction php mail est disponible
	 **/
	public static function testMail($io=true, $format="<div class=\"alert alert-#color\" role=\"alert\">#symbol #message</div>\n") 
	{
		global $lang;
		if($return=function_exists('mail')) {
			if($io==true) {
				$output = str_replace('#color', 'success', $format);
				$output = str_replace('#symbol', '&#10004;', $output);
				$output = str_replace('#message', $lang['mail_available'], $output);
				return $output;
			}
		} else {
			if($io==true) {
				$output = str_replace('#color', 'danger', $format);
				$output = str_replace('#symbol', '&#10007;', $output);
				$output = str_replace('#message', $lang['mail_not_available'], $output);
				return $output;
			}
		}
		return $return;
	}
	
	/**
	 * Méthode d'envoi de mail.
	 *
	 * @param string $name Nom de l'expéditeur
	 * @param string $from Email de l'expéditeur
	 * @param array|string $to Adresse(s) du(des) destinataire(s)
	 * @param string $subject Objet du mail
	 * @param string $body Contenu du mail
	 * @param string $contentType Type de contenu (text ou html)
	 * @param array|string|false $cc Adresse(s) en copie (CC)
	 * @param array|string|false $bcc Adresse(s) en copie cachée (BCC)
	 * @return boolean Renvoie FAUX en cas d'erreur d'envoi
	 */
	public static function sendMail($name, $from, $to, $subject, $body, $contentType = "text", $cc = false, $bcc = false) 
	{
	    // Convertit les destinataires, CC et BCC en chaînes si ce sont des tableaux
	    if (is_array($to)) {
	        $to = implode(', ', $to);
	    }
	    if (is_array($cc)) {
	        $cc = implode(', ', $cc);
	    }
	    if (is_array($bcc)) {
	        $bcc = implode(', ', $bcc);
	    }
	
	    // Prépare les en-têtes du mail
	    $headers  = "From: " . $name . " <" . $from . ">\r\n";
	    $headers .= "Reply-To: " . $from . "\r\n";
	    $headers .= 'MIME-Version: 1.0' . "\r\n";
	
	    // Détermine le type de contenu
	    if ($contentType === 'html') {
	        $headers .= 'Content-Type: text/html; charset="' . CHARSET . '"' . "\r\n";
	    } else {
	        $headers .= 'Content-Type: text/plain; charset="' . CHARSET . '"' . "\r\n";
	    }
	
	    $headers .= 'Content-Transfer-Encoding: 8bit' . "\r\n";
	    $headers .= 'Date: ' . date("D, j M Y G:i:s O") . "\r\n"; // Format de date
	
	    // Ajoute les en-têtes CC et BCC si présents
	    if (!empty($cc)) {
	        $headers .= 'Cc: ' . $cc . "\r\n";
	    }
	    if (!empty($bcc)) {
	        $headers .= 'Bcc: ' . $bcc . "\r\n";
	    }
	
	    // Envoie le mail et retourne le résultat
	    return mail($to, $subject, $body, $headers);
	}
	
	/**
	 * Renvoie la forme plurielle ou singulière d'un mot en fonction du nombre.
	 * $num = 6;
	 * $texte = 'Vous avez acheté ' . $num . ' journ' . Util::pluralize($num, 'aux', 'al') . ' ce jour.<br />';
	 * $texte .= 'Vous avez ' . $num . ' commentaire' . Util::pluralize($num) . '<br />';	 
	 *
	 * @param int $num Le nombre à évaluer.
	 * @param string $plural La forme plurielle du mot.
	 * @param string $single La forme singulière du mot.
	 * @return string La forme appropriée du mot.
	 */
	public static function pluralize($num, $plural = 's', $single = '') 
	{
	    return ($num === 0 || $num === 1) ? $single : $plural; 
	}
	
	/**
	 * Renvoie la description en fonction de la page actuelle.
	 *
	 * @return string La description de la page.
	 */
	public static function Description()
	{
	    global $config, $cur, $out;
	
	    if ($cur === 'home') { 
	        return $config['description']; 
	    } else {
	        return $out['subtitle']; 
	    }
	}
	
	/**
	 * Génère un bouton d'aide avec une info-bulle.
	 *
	 * @param string $phrase La phrase à afficher dans l'info-bulle.
	 * @return string Le code HTML du bouton d'aide.
	 */
	public static function Help($phrase)
	{
	    global $config, $lang;
	
	    // Fix pour éviter les erreurs de type null dans PHP 7.4
	    $lang['more_info'] = isset($lang['more_info']) ? $lang['more_info'] : null;
	
	    return '&nbsp;<a class="text-primary" data-toggle="tooltip" data-placement="top" title="' . $lang['more_info'] . '">
	                <i class="fa fa-info-circle fa-lg" aria-hidden="true" data-container="body" data-toggle="popover" data-placement="right" data-content="' . $phrase . '"></i>
	            </a>';
	}
	
	/**
	 * Méthode qui supprime tous les fichiers d'un dossier.
	 *
	 * @param string $dir Le chemin du dossier dont les fichiers doivent être supprimés.
	 * @return void
	 * <code>
	 *     echo Util::deletecache($dir);
	 * </code>
	 */
	public static function deletecache($dir)
	{
	    global $lang;
	
	    // Récupère tous les fichiers dans le répertoire, y compris les fichiers cachés
	    $files = glob($dir . '{,.}*', GLOB_BRACE); 
	
	    // Vérifie si des fichiers ont été trouvés
	    if (empty($files)) {
	        Plugin::redirectMsg($lang['no_files_found'], 'config.php', $lang['config'], 'alert alert-warning');
	        return; // Sort de la fonction si aucun fichier n'est trouvé
	    }
	
	    // Parcourt chaque fichier trouvé
	    foreach ($files as $file) {
	        if (is_file($file)) {
	            // Tente de supprimer le fichier
	            if (@unlink($file)) {
	                Plugin::redirectMsg($lang['cache_clean'], 'config.php', $lang['config'], 'alert alert-success');
	            } else {
	                Plugin::redirectMsg($lang['folder_error'], 'config.php', $lang['config'], 'alert alert-danger'); 
	            }
	        }
	    }
	}

	/**
	 * Méthode qui retourne le type de compression disponible.
	 *
	 * @return void
	 * <code>
	 *     echo Util::httpEncoding();
	 * </code>
	 */
	public static function httpEncoding() {
	    global $HTTP_ACCEPT_ENCODING;
	
	    // Vérifie si les en-têtes ont déjà été envoyés
	    if (headers_sent()) {
	        return; // Sort de la fonction si les en-têtes sont déjà envoyés
	    }
	
	    // Détermine le type de compression disponible
	    if (strpos($HTTP_ACCEPT_ENCODING, 'x-gzip') !== false) {
	        $encoding = 'x-gzip';
	    } elseif (strpos($HTTP_ACCEPT_ENCODING, 'gzip') !== false) {
	        $encoding = 'gzip';
	    } else {
	        $encoding = false;
	    }
	
	    // Si un encodage est trouvé, compresse le contenu
	    if ($encoding) {
	        $contents = ob_get_contents(); // Récupère le contenu du tampon de sortie
	        ob_end_clean(); // Nettoie le tampon de sortie
	
	        // Envoie l'en-tête de compression
	        header('Content-Encoding: ' . $encoding);
	        print("\x1f\x8b\x08\x00\x00\x00\x00\x00"); // En-tête GZIP
	
	        // Compresse le contenu
	        $compressedContents = gzcompress($contents, 9);
	        print($compressedContents); // Affiche le contenu compressé
	        exit(); // Termine le script
	    } else {
	        ob_end_flush(); // Envoie le contenu non compressé
	        exit(); // Termine le script
	    }
	}
		
	/**
	* Méthode de journalisation
	*
	* @param	text	string
	* @param	type	string
    *  <code>
    *      echo Util::journal($text);
    *  </code>	
	**/	
	public static function journal($text, $type=0)
	{
		if (is_array($text) ) {
			error_log('------------------------', $type);
			error_log('Array', $type);
			error_log('------------------------', $type);
			foreach ($text as $key=>$value) {
				error_log($key.'=>'.$value, $type);
			}
			error_log('------------------------', $type);
		}
		error_log('(' .VERSION. ') (' .$_SERVER['REQUEST_URI']. ') ' .$text, $type);
	}					
}