<?php defined('FLATBOARD') or die('Flatboard Community.');
/**
 * Identicon
 *
 * @author 		Frédéric K.
 * @copyright	(c) 2020-2025
 * @license		http://opensource.org/licenses/MIT
 * @package		FlatBoard
 * @version		3.5.0
 * @update		2025-02-08
 */	

define('PATH_IDENTICON_AVATAR', UPLOADS_DIR . 'avatars' . DS);
define('PATH_GEN_AVATAR', 	 	PATH_IDENTICON_AVATAR . 'generates' . DS);
define('HTML_AVATAR', 		 	HTML_BASEPATH . DS . 'uploads' . DS . 'avatars' . DS);
define('HTML_GEN_AVATAR', 	 	HTML_AVATAR . 'generates' . DS);  
define('HTML_PLUGIN_AVATAR', 	HTML_BASEPATH . DS . 'plugin' . DS . 'identicon' . DS);  
define('IDENTICON_ONLINE', 		TRUE);                 

/**
 * Pré-installe les paramètres par défaut.
 */
function identicon_install()
{
    $plugin = 'identicon';
    
    // Vérifier si le plugin est déjà installé
    if (flatDB::isValidEntry('plugin', $plugin)) {
        return; // Si le plugin est déjà installé, ne rien faire
    }
	
    // Création des dossiers pour les avatars
    $folders = [PATH_IDENTICON_AVATAR, PATH_GEN_AVATAR];
    foreach ($folders as $folder) {    
        if (!is_dir($folder)) {
            mkdir($folder, 0777, true); // Créer le dossier avec les permissions appropriées
        }	
        // Créer un fichier index.html s'il n'existe pas
        if (!file_exists($folder . 'index.html')) {
            $f = fopen($folder . 'index.html', 'a+');
            fwrite($f, ''); // Écrire un contenu vide
            fclose($f);	
        }
    }

    // Initialiser les données du plugin
    $data = [
        $plugin . 'state' => true, 
        'taille' => '50', 
        'forme' => 'rounded-circle', 
        'dots' => true
    ];
    
    // Sauvegarder les données du plugin
    flatDB::saveEntry('plugin', $plugin, $data);         
}

/**
 * Vérifie si le plugin est installé.
 *
 * @param string $plugin Le nom du plugin à vérifier
 * @return bool Vrai si le plugin est installé, faux sinon
 */
function pluginIsHere($plugin) 
{
    return flatDB::isValidEntry('plugin', $plugin);
}

/**
 * Configuration de l'admin pour le plugin Identicon
 */
function identicon_config()
{    
    global $lang, $token; 
    $plugin = 'identicon';
    $out = '';
           
    // Traitement des données POST
    if (!empty($_POST) && CSRF::check($token)) {
        $data[$plugin . 'state'] = Util::isPOST('state') ? $_POST['state'] : '';  
        $data['taille'] = HTMLForm::clean($_POST['taille']);    
        $data['forme'] = HTMLForm::clean($_POST['forme']);
        $data['dots'] = Util::isPOST('dots') ? $_POST['dots'] : '';  
               
        // Sauvegarder les données du plugin
        flatDB::saveEntry('plugin', $plugin, $data);
        
        // Afficher un message de succès
        $out .= Plugin::redirectMsg(
            $lang['data_save'],
            'config.php' . DS . 'plugin' . DS . $plugin,
            $lang['plugin'] . '&nbsp;<b>' . $lang[$plugin . 'name'] . '</b>'
        );   
    } else {
        // Lire les données du plugin si elles existent
        if (flatDB::isValidEntry('plugin', $plugin)) {
            $data = flatDB::readEntry('plugin', $plugin);
        }

        // Générer le formulaire de configuration
        $out .= HTMLForm::form('config.php' . DS . 'plugin' . DS . $plugin, '
            <div class="row"> 
                <div class="col">' .
                    HTMLForm::checkBox('state', $data[$plugin . 'state']) . '
                </div>				              	
                <div class="col">' .
                    HTMLForm::checkBox('dots', $data['dots']) . '
                </div>
            </div>						               	
            <div class="row">				
                <div class="col">' .
                    HTMLForm::text('taille', isset($data) ? $data['taille'] : '', 'number', '', '', 'pixel') .
                '</div>
                <div class="col">' .
                    HTMLForm::select('forme', [
                        'rounded-circle' => $lang['rounded-circle'], 
                        'rounded' => $lang['rounded'], 
                        'img-thumbnail' => $lang['img-thumbnail']
                    ], $data['forme']) . '
                </div>				
            </div>' .       
            HTMLForm::simple_submit()
        );
    }

    return $out; // Retourner le contenu généré
}

/**
 * Ajoute un peu de CSS pour le plugin Identicon
 */
function identicon_head()
{
    $plugin = 'identicon';
    
    // Lecture des données du plugin
    $data = flatDB::readEntry('plugin', $plugin);
    
    if ($data[$plugin . 'state']) {
        return '
        <style>
            .avatar-status {
                border-radius: 50%; 
                width: 15px; 
                height: 15px; 
                margin-left: -15px; 
                position: absolute; 
                border: 2px solid #fff;
            }
            .avatar-status.green {
                background: rgba(51, 217, 178, 1);
                box-shadow: 0 0 0 0 rgba(51, 217, 178, 1);
                animation: pulse-green 2s infinite;
            }
            @keyframes pulse-green {
                0% { transform: scale(0.95); box-shadow: 0 0 0 0 rgba(51, 217, 178, 0.7); }
                70% { transform: scale(1); box-shadow: 0 0 0 10px rgba(51, 217, 178, 0); }
                100% { transform: scale(0.95); box-shadow: 0 0 0 0 rgba(51, 217, 178, 0); }
            }
            .avatar-status.red {
                background: rgba(255, 82, 82, 1);
                box-shadow: 0 0 0 0 rgba(255, 82, 82, 1);
                animation: pulse-red 2s infinite;
            }
            @keyframes pulse-red {
                0% { transform: scale(0.95); box-shadow: 0 0 0 0 rgba(255, 82, 82, 0.7); }
                70% { transform: scale(1); box-shadow: 0 0 0 10px rgba(255, 82, 82, 0); }
                100% { transform: scale(0.95); box-shadow: 0 0 0 0 rgba(255, 82, 82, 0); }
            }
            figure:hover img {
                box-shadow: #e3e3e3 0 0 0 0.5rem;
            }
            figure img {
                transition: box-shadow 300ms ease-out;
            }
        </style>' . PHP_EOL;
    }
}

/**
 * Vérifie les membres connectés via le plugin online
 *
 * @param string $username Le nom d'utilisateur à vérifier
 * @return string Statut des utilisateurs en ligne
 */
function checkOnline($username)
{
    global $sessionTrip, $cur;
    $plugin = 'online';

    // Vérifier si le plugin "online" est installé
    if (pluginIsHere($plugin)) {
        $users = [];
        // Lecture des données du plugin
        $data = flatDB::readEntry('plugin', $plugin);
        
        if ($data[$plugin . 'state']) {
            $online = flatDB::readEntry('plugin', $plugin . 'DB');
            foreach ((array)$online as $ip => $time) {
                // Vérifier si c'est une IP valide
                $validIP = filter_var($ip, FILTER_VALIDATE_IP);
                // Ne filtrer que les membres
                $users[] = $validIP ? '' : $ip;			
                
                // Mode en ligne
                $trip = HTMLForm::trip($sessionTrip, $username);	
                if ($username === $users || $username === $trip) {
                    $usersStat = '<span class="avatar-status green"></span>'; // Utilisateur en ligne
                } else {
                    $usersStat = '<span class="avatar-status red"></span>'; // Utilisateur hors ligne
                }									
            }	
            
            // Afficher le statut des utilisateurs en fonction de la page actuelle
            $currentUsers = ($cur !== 'home' && IDENTICON_ONLINE) ? $usersStat : '';	
        }
    } else {
        $currentUsers = ''; // Si le plugin n'est pas installé, retourner une chaîne vide
    }

    return $currentUsers; // Retourner le statut des utilisateurs
}

/**
 * Affiche l'avatar de l'utilisateur
 *
 * @param string $username Le nom d'utilisateur pour lequel afficher l'avatar
 * @return string HTML de l'avatar
 */
function identicon_profile($username)
{
    global $sessionTrip, $cur;
    $plugin = 'identicon';

    // Lecture des données du plugin
    $data = flatDB::readEntry('plugin', $plugin);
    
    // Récupération du pseudo sans le hash
    $user = strstr($username, '@', true);
    
    // Taille de l'avatar en configuration
    $size = $data['taille']; 
    
    // Conversion du nom d'utilisateur
    $h1 = hash('crc32', $username);
    $h2 = hash('crc32b', $username);
    
    // Points ou carrés ?
    $d = $data['dots'] ? 'dot' : '';
    
    // Génération du nom d'utilisateur
    $s = $h1 . $h2[0];
    $file_path = PATH_GEN_AVATAR . $s . 'x' . $size . $d;
    $file_html = HTML_GEN_AVATAR . $s . 'x' . $size . $d;
    
    // Lien de génération de l'avatar avec paramètres
    $dots = $data['dots'] ? '&d=dot' : '';
    $pic = HTML_PLUGIN_AVATAR . 'avatar.php?f=' . urlencode(PATH_GEN_AVATAR) . '&u=' . urlencode($username) . '&sz=' . $size . $dots;

    /**
     * INDICATEUR DE STATUT EN LIGNE / HORS LIGNE
     */	
    // Vérifier si le plugin "online" est installé
    if (pluginIsHere($plugin) && IDENTICON_ONLINE) {	
        $trip = HTMLForm::trip($sessionTrip, $username);
        $usersStat = ($username === $trip) ? '<span class="avatar-status green"></span>' : '<span class="avatar-status red"></span>';	
        $online = (IDENTICON_ONLINE) ? $usersStat : '';
    } else {
        $online = '';
    }

    /**
     * GÉNÉRATION ET AFFICHAGE DE L'AVATAR
     */					  	
    $identity = '';
    if ($data[$plugin . 'state']) {	
        // Avatar personnalisé placé dans uploads/avatars/ sans l'arobase
        // Exemple: pseudo@motdepasse = pseudo_motdepasse.png
        $avatar = str_replace(['#', '@'], '_', $username);	  	  	
        
        if (file_exists(PATH_IDENTICON_AVATAR . $avatar . '.png')) {
            // Avatar personnalisé trouvé
            $identity .= '<img class="' . $data['forme'] . '" src="' . HTML_AVATAR . $avatar . '.png" width="' . $size . '" height="' . $size . '" alt="' . $user . '">' . $online . PHP_EOL; 
        } elseif (is_numeric($username)) {
            // Si le pseudo est numérique, utiliser un avatar anonyme
            $identity .= '<img class="' . $data['forme'] . '" src="' . HTML_PLUGIN_AVATAR . 'anonymous.png" width="' . $size . '" height="' . $size . '" alt="anonymous">' . $online . PHP_EOL;
        } elseif (file_exists($file_path . '.png')) {
            // Si l'avatar est déjà généré, le prendre en cache
            $identity .= '<img class="' . $data['forme'] . '" src="' . $file_html . '.png" width="' . $size . '" height="' . $size . '" alt="' . $user . '">' . $online . PHP_EOL; 
        } else {
            // Sinon, donner l'URL de génération
            $identity .= '<img class="' . $data['forme'] . '" src="' . $pic . '" width="' . $size . '" height="' . $size . '" alt="' . $user . '">' . $online . PHP_EOL;    
        }			    
    }   	
    
    return $identity; // Retourner l'HTML de l'avatar
}
?>