<?php
/*
 * Project name: Flatboard
 * Project URL: https://flatboard.org
 * Author: Frédéric Kaplon and contributors
 * All Flatboard code is released under the MIT license.
 * 
 * Rate Limiter Global Ultra Simple
 * Limite TOUTES les requêtes POST/GET à 2 par seconde maximum
 */
class GlobalRequestLimiter {
    
    private $dataPath;
    private $maxRequests;
    private $timeWindow;
    
    public function __construct($dataPath = './rate_limits/', $maxRequests = 2, $timeWindow = 1) {
        $this->dataPath = rtrim($dataPath, '/') . '/';
        $this->maxRequests = $maxRequests;
        $this->timeWindow = $timeWindow;
        
        // Créer le dossier de stockage
        if (!is_dir($this->dataPath)) {
            mkdir($this->dataPath, 0755, true);
        }
        
        // Sécuriser le dossier
        $htaccessFile = $this->dataPath . '.htaccess';
        if (!file_exists($htaccessFile)) {
            file_put_contents($htaccessFile, "Deny from all\n");
        }
    }
    
    /**
     * Obtenir l'IP du client
     */
    private function getClientIp(): string {
        $headers = ['HTTP_CF_CONNECTING_IP', 'HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'REMOTE_ADDR'];
        
        foreach ($headers as $header) {
            if (!empty($_SERVER[$header])) {
                $ip = $_SERVER[$header];
                if (strpos($ip, ',') !== false) {
                    $ip = trim(explode(',', $ip)[0]);
                }
                if (filter_var($ip, FILTER_VALIDATE_IP)) {
                    return $ip;
                }
            }
        }
        
        return $_SERVER['REMOTE_ADDR'] ?? '127.0.0.1';
    }
    
    /**
     * Vérifier et appliquer la limitation globale
     */
    public function checkGlobalLimit(): void {
        $ip = $this->getClientIp();
        $filename = $this->dataPath . 'global_' . md5($ip) . '.json';
        $currentTime = microtime(true); // Utiliser microtime pour plus de précision
        
        // Charger les données existantes
        $data = [];
        if (file_exists($filename)) {
            $jsonData = file_get_contents($filename);
            $data = json_decode($jsonData, true) ?: [];
        }
        
        // Initialiser si vide
        if (!isset($data['requests'])) {
            $data['requests'] = [];
        }
        
        // Filtrer les requêtes dans la fenêtre de temps (dernière seconde)
        $validRequests = array_filter($data['requests'], function($timestamp) use ($currentTime) {
            return ($currentTime - $timestamp) <= $this->timeWindow;
        });
        
        // Vérifier si la limite est atteinte
        if (count($validRequests) >= $this->maxRequests) {
            $this->blockRequest($validRequests);
        }
        
        // Ajouter la nouvelle requête
        $validRequests[] = $currentTime;
        
        // Garder seulement les requêtes récentes pour économiser l'espace
        $data['requests'] = array_slice($validRequests, -10); // Garder max 10 entrées
        $data['last_request'] = $currentTime;
        $data['total_requests'] = ($data['total_requests'] ?? 0) + 1;
        
        // Sauvegarder
        file_put_contents($filename, json_encode($data), LOCK_EX);
        
        // Nettoyage occasionnel (1% de chance)
        if (mt_rand(1, 100) === 1) {
            $this->cleanup();
        }
    }
    
    /**
     * Bloquer la requête et renvoyer une erreur 429
     */
    private function blockRequest(array $requests): void {
        $oldestRequest = min($requests);
        $retryAfter = max(0.1, $this->timeWindow - (microtime(true) - $oldestRequest));
        
        // Headers HTTP
        http_response_code(429);
        header('Content-Type: application/json; charset=utf-8');
        header('X-RateLimit-Limit: ' . $this->maxRequests);
        header('X-RateLimit-Window: ' . $this->timeWindow . 's');
        header('X-RateLimit-Remaining: 0');
        header('Retry-After: ' . ceil($retryAfter));
        
        // Réponse JSON
        $response = [
            'error' => 'Rate limit exceeded',
            'message' => 'Trop de requêtes. Maximum ' . $this->maxRequests . ' requêtes par seconde.',
            'retry_after' => ceil($retryAfter),
            'limit' => $this->maxRequests,
            'window' => $this->timeWindow . 's',
            'timestamp' => date('Y-m-d H:i:s')
        ];
        
        echo json_encode($response, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }
    
    /**
     * Nettoyer les anciens fichiers
     */
    private function cleanup(): void {
        $files = glob($this->dataPath . 'global_*.json');
        $currentTime = time();
        $maxAge = 300; // 5 minutes
        
        foreach ($files as $file) {
            if (file_exists($file) && ($currentTime - filemtime($file)) > $maxAge) {
                unlink($file);
            }
        }
    }
    
    /**
     * Obtenir les statistiques actuelles (pour debug)
     */
    public function getStats(): array {
        $ip = $this->getClientIp();
        $filename = $this->dataPath . 'global_' . md5($ip) . '.json';
        
        if (!file_exists($filename)) {
            return [
                'current_requests' => 0,
                'max_requests' => $this->maxRequests,
                'remaining' => $this->maxRequests,
                'window' => $this->timeWindow . 's'
            ];
        }
        
        $data = json_decode(file_get_contents($filename), true);
        $currentTime = microtime(true);
        
        $validRequests = array_filter($data['requests'] ?? [], function($timestamp) use ($currentTime) {
            return ($currentTime - $timestamp) <= $this->timeWindow;
        });
        
        $current = count($validRequests);
        
        return [
            'current_requests' => $current,
            'max_requests' => $this->maxRequests,
            'remaining' => max(0, $this->maxRequests - $current),
            'window' => $this->timeWindow . 's',
            'total_requests' => $data['total_requests'] ?? 0
        ];
    }
}