<?php
defined('FLATBOARD') or die('Flatboard Community.');

/**
 * Online Plugin
 *
 * Tracks and displays online users and bots on a Flatboard website, with a modal to show visitor details.
 * Simplified modal for non-authenticated users (total visitors, staff presence, bots).
 * Full modal for authenticated users (admin/worker), with IPs (full for admins, masked for workers) and user-agents (bots).
 * Includes scrollbar and text visibility fixes.
 *
 * @author      Frédéric K.
 * @copyright   (c) 2015-2025
 * @license     http://opensource.org/licenses/MIT
 * @package     FlatBoard
 * @version     3.2
 * @update      2025-06-10
 */

const PLUGIN_NAME = 'online';

/**
 * Masks a portion of an IP address for non-admin users.
 *
 * @param string $ip The IP address (IPv4 or IPv6)
 * @return string The masked or original IP
 */
function mask_ip($ip)
{
    if (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_INET)) {
        // For IPv4: mask the last two octets (e.g., 192.168.*.*)
        $parts = explode('.', $ip);
        if (count($parts) === 4) {
            $parts[2] = '*';
            $parts[3] = '*';
            return implode('.', $parts);
        }
    } elseif (filter_var($ip, FILTER_VALIDATE_IP, FILTER_FLAG_INET6)) {
        // For IPv6: mask the last four segments (e.g., 2001:db8::****:****)
        $parts = explode(':', $ip);
        $count = count($parts);
        if ($count >= 4) {
            for ($i = $count - 4; $i < $count; $i++) {
                $parts[$i] = '****';
            }
            return implode(':', $parts);
        }
    }
    return $ip; // Return original IP if invalid or unable to mask
}

/**
 * Installs default plugin settings.
 */
function online_install()
{
    // Check if plugin is already installed
    if (flatDB::isValidEntry('plugin', PLUGIN_NAME)) {
        return;
    }

    // Default configuration
    $data = [
        PLUGIN_NAME . 'state' => false,
        'color' => 'primary',
        PLUGIN_NAME . '_display' => 'icon'
    ];

    // Save plugin configuration
    flatDB::saveEntry('plugin', PLUGIN_NAME, $data);

    // Initialize hits tracking
    $online = [$_SERVER['REMOTE_ADDR'] => ['time' => time(), 'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown']];
    flatDB::saveEntry('plugin', PLUGIN_NAME . '_hits', $online);

    // Initialize authenticated users tracking
    flatDB::saveEntry('plugin', PLUGIN_NAME . '_auth_users', []);
}

/**
 * Generates admin configuration form for the plugin.
 *
 * @return string HTML form or success message
 */
function online_config()
{
    global $lang, $token;
    $out = '';

    // Color and display options
    $color_select = [
        'primary' => $lang['blue'],
        'secondary' => $lang['grey'],
        'success' => $lang['green'],
        'danger' => $lang['red'],
        'warning' => $lang['yellow'],
        'info' => $lang['cyan'],
        'light' => $lang['white'],
        'dark' => $lang['dark']
    ];
    $display = [
        'icon' => $lang['icon'],
        'text' => $lang['text']
    ];

    // Handle form submission
    if (!empty($_POST) && CSRF::check($token)) {
        $data = [
            PLUGIN_NAME . 'state' => filter_input(INPUT_POST, 'state', FILTER_VALIDATE_BOOLEAN) ?? false,
            'color' => filter_input(INPUT_POST, 'color', FILTER_VALIDATE_REGEXP, [
                'options' => ['regexp' => '/^(primary|secondary|success|danger|warning|info|light|dark)$/']
            ]) ?? 'primary',
            PLUGIN_NAME . '_display' => filter_input(INPUT_POST, PLUGIN_NAME . '_display', FILTER_VALIDATE_REGEXP, [
                'options' => ['regexp' => '/^(icon|text)$/']
            ]) ?? 'icon'
        ];

        // Save updated configuration
        flatDB::saveEntry('plugin', PLUGIN_NAME, $data);

        // Display success message
        $out .= Plugin::redirectMsg(
            $lang['data_save'],
            'config.php' . DS . 'plugin' . DS . PLUGIN_NAME,
            $lang['plugin'] . ' <b>' . $lang[PLUGIN_NAME . 'name'] . '</b>'
        );
    } else {
        // Load existing configuration
        $data = flatDB::readEntry('plugin', PLUGIN_NAME) ?: [];

        // Generate configuration form
        $out .= HTMLForm::form(
            'config.php' . DS . 'plugin' . DS . PLUGIN_NAME,
            HTMLForm::checkBox('state', $data[PLUGIN_NAME . 'state'] ?? false) .
            '<div class="form-row">
                <div class="col-3">' . HTMLForm::select('color', $color_select, $data['color'] ?? 'primary') . '</div>
                <div class="col-3">' . HTMLForm::select(PLUGIN_NAME . '_display', $display, $data[PLUGIN_NAME . '_display'] ?? 'icon') . '</div>
            </div>' .
            HTMLForm::simple_submit()
        );
    }

    return $out;
}

/**
 * Displays online users and bots in the footer with a modal for visitor details.
 * Simplified modal for non-authenticated users; full modal for authenticated users (admin/worker).
 * Admins see all IPs and user-agents; workers see masked IPs; non-authenticated see total visitors and staff presence.
 *
 * @return string HTML output for footer
 */
function online_footer()
{
    global $lang;
    $out = '';

    // Load plugin configuration
    $data = flatDB::readEntry('plugin', PLUGIN_NAME);
    if (!$data || !($data[PLUGIN_NAME . 'state'] ?? false)) {
        return $out;
    }

    // Detect crawlers
    $crawlers = smart_ip_detect_crawler($_SERVER['HTTP_USER_AGENT'] ?? '');
    $online = flatDB::readEntry('plugin', PLUGIN_NAME . '_hits') ?: [];
    $authUsers = flatDB::readEntry('plugin', PLUGIN_NAME . '_auth_users') ?: [];

    // Clean inactive IPs (older than 60 seconds)
    foreach ($online as $ip => $info) {
        if (time() - $info['time'] > 60) {
            unset($online[$ip]);
        }
    }

    // Clean inactive authenticated users
    foreach ($authUsers as $ip => $info) {
        if (time() - $info['time'] > 60) {
            unset($authUsers[$ip]);
        }
    }

    // Save current user's IP and user-agent
    $ip = User::getRealIpAddr();
    $online[$ip] = [
        'time' => time(),
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown'
    ];

    // Update authenticated users if logged in
    if (isset($_SESSION['role'])) {
        $user = isset($_SESSION['trip']) && !empty($_SESSION['trip']) ? $_SESSION['trip'] : ($_SESSION['role'] ?? '');
        if (!empty($user) && $user !== 'Unknown') {
            $displayUser = strpos($user, '@') !== false ? explode('@', $user)[0] : $user;
            $authUsers[$ip] = [
                'time' => time(),
                'username' => $displayUser,
                'role' => is_array($_SESSION['role']) ? implode(',', array_values($_SESSION['role'])) : $_SESSION['role']
            ];
        }
    }

    // Save updated data
    flatDB::saveEntry('plugin', PLUGIN_NAME . '_hits', $online);
    flatDB::saveEntry('plugin', PLUGIN_NAME . '_auth_users', $authUsers);

    // Check user roles
    $isAdmin = User::isAdmin();
    $isAuthenticated = User::isWorker(); // Includes admin and worker

    // Prepare modal content
    $modalContent = '';

    // Authenticated users (from flatDB)
    if (!empty($authUsers)) {
        $modalContent .= '<h6>' . ($lang['authenticated_users'] ?? 'Authenticated Users') . '</h6>';
        $modalContent .= '<ul class="list-group list-group-flush">';
        foreach ($authUsers as $ip => $info) {
            $username = htmlspecialchars($info['username']);
            if ($isAdmin) {
                $modalContent .= '<li class="list-group-item">' . $username . ' (' . ($lang['staff'] ?? 'Staff') . ', IP: ' . htmlspecialchars($ip) . ')</li>';
            } else {
                $modalContent .= '<li class="list-group-item">' . $username . ' (' . ($lang['staff'] ?? 'Staff') . ')</li>';
            }
        }
        $modalContent .= '</ul>';
    }

    // Connected visitors (IPs or total count)
    $modalContent .= '<h6 class="mt-3">' . ($lang['connected_users'] ?? 'Connected Visitors') . '</h6>';
    if ($isAuthenticated) {
        // Full details for authenticated users (admin/worker)
        $modalContent .= '<ul class="list-group list-group-flush">';
        if (!empty($online)) {
            foreach ($online as $ip => $info) {
                $displayIp = $isAdmin ? htmlspecialchars($ip) : mask_ip($ip);
                $userAgent = $isAdmin ? htmlspecialchars($info['user_agent']) : '';
                $suspect = $isAdmin && ($info['user_agent'] === 'Unknown' || empty($info['user_agent'])) ? '<span class="text-warning" title="Suspicious: No user-agent">⚠</span>' : '';
                $modalContent .= '<li class="list-group-item">' . $displayIp . ' (' . date('H:i:s', $info['time']) . ')' . $suspect;
                if ($isAdmin && !empty($userAgent)) {
                    $modalContent .= '<br><small>User-Agent: ' . $userAgent . '</small>';
                }
                $modalContent .= '</li>';
            }
        } else {
            $modalContent .= '<li class="list-group-item">' . ($lang['no_visitors'] ?? 'No visitors online') . '</li>';
        }
        $modalContent .= '</ul>';
    } else {
        // Simplified for non-authenticated users
        $visitorCount = count($online);
        $modalContent .= '<p>' . sprintf($lang['visitor_count'] ?? '%d visitors online', $visitorCount) . '</p>';
    }

    // Detected bots
    if (!empty($crawlers)) {
        $modalContent .= '<h6 class="mt-3">' . ($lang['bots_detected'] ?? 'Detected Bots') . '</h6>';
        $modalContent .= '<ul class="list-group list-group-flush">';
        foreach ($crawlers as $bot) {
            $modalContent .= '<li class="list-group-item">' . htmlspecialchars($bot) . ' (User-Agent: ' . htmlspecialchars($_SERVER['HTTP_USER_AGENT'] ?? 'Unknown') . ')</li>';
        }
        $modalContent .= '</ul>';
    }

    // Modal HTML (Bootstrap 4.5.3) with custom styles and scrollbar
    $modal = '
    <style>
        #onlineVisitorsModal .modal-content {
            background-color: #fff;
            color: #333;
        }
        #onlineVisitorsModal .modal-body {
            color: #333;
            max-height: 400px;
            overflow-y: auto;
        }
        #onlineVisitorsModal .list-group-item,
        #onlineVisitorsModal .h6 {
            color: #333;
        }
        #onlineVisitorsModal .modal-header,
        #onlineVisitorsModal .modal-footer {
            background-color: #f8f9fa;
            color: #333;
        }
        #onlineVisitorsModal .close,
        #onlineVisitorsModal .close span {
            color: #333;
        }
    </style>
    <div class="modal fade" id="onlineVisitorsModal" tabindex="-1" role="dialog" aria-labelledby="onlineVisitorsModalLabel" aria-hidden="true">
        <div class="modal-dialog" role="document">
            <div class="modal-content">
                <div class="modal-header">
                    <h5 class="modal-title" id="onlineVisitorsModalLabel">' . ($lang['online_visitors'] ?? 'Online Visitors') . '</h5>
                    <button type="button" class="close" data-dismiss="modal" aria-label="Close">
                        <span aria-hidden="true">×</span>
                    </button>
                </div>
                <div class="modal-body">
                    ' . $modalContent . '
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-dismiss="modal">' . ($lang['close'] ?? 'Close') . '</button>
                </div>
            </div>
        </div>
    </div>';

    // Display badge for online users and bots
    $out .= ' <span class="badge badge-' . htmlspecialchars($data['color'] ?? 'primary') . '" data-toggle="modal" data-target="#onlineVisitorsModal" style="cursor: pointer;">' .
            ($data[PLUGIN_NAME . '_display'] === 'icon' ? '<i class="fa fa-user"></i> ' : $lang['online']) .
            count($online);

    if (!empty($authUsers)) {
        $out .= ' - <a data-toggle="tooltip" data-placement="top" title="' . ($lang['staff_online'] ?? 'Staff online') . ': ' . count($authUsers) . '"><i class="fa fa-user-plus"></i></a>';
    }

    if (!empty($crawlers)) {
        $botNames = implode(', ', $crawlers);
        $out .= ' - <a data-toggle="modal" data-target="#onlineVisitorsModal" title="' . ($lang['bots_detected'] ?? 'Detected bots') . ': ' . htmlspecialchars($botNames) . '"><i class="fa fa-android"></i></a>';
    }

    $out .= '</span>' . $modal;

    return $out;
}

/**
 * Detects crawlers, spiders, or bots based on user agent.
 *
 * @param string $user_agent User agent string
 * @return array List of detected bot names
 */
function smart_ip_detect_crawler($user_agent)
{
    // Comprehensive list of known bots (2025)
    $crawlers = [
        // Search Engines
        ['Googlebot', 'Google'],
        ['Bingbot', 'Bing'],
        ['Slurp', 'Yahoo! Slurp'],
        ['DuckDuckBot', 'DuckDuckGo'],
        ['YandexBot', 'Yandex'],
        ['Baiduspider', 'Baidu'],
        ['Sogou', 'Sogou Spider'],
        ['Exabot', 'Exabot'],
        ['NaverBot', 'Naver'],
        ['CoccocBot', 'Coc Coc'],
        ['YandexImages', 'Yandex Images'],
        ['YandexVideo', 'Yandex Video'],
        ['YandexNews', 'Yandex News'],
        ['Googlebot-Image', 'Google Image'],
        ['Googlebot-Video', 'Google Video'],
        ['Googlebot-News', 'Google News'],

        // Social Media
        ['Twitterbot', 'Twitter'],
        ['facebot', 'Facebook'],
        ['LinkedInBot', 'LinkedIn'],
        ['SlackBot', 'Slack'],
        ['Pinterest', 'Pinterest'],
        ['TelegramBot', 'Telegram'],
        ['WhatsApp', 'WhatsApp'],
        ['Discordbot', 'Discord'],

        // SEO and Analytics Tools
        ['AhrefsBot', 'Ahrefs'],
        ['SemrushBot', 'Semrush'],
        ['MJ12bot', 'Majestic'],
        ['DotBot', 'DotBot'],
        ['Screaming Frog', 'Screaming Frog'],
        ['SiteAnalyzer', 'Site Analyzer'],
        ['Netcraft', 'Netcraft'],
        ['Grapeshot', 'Grapeshot'],

        // Monitoring and Archival
        ['ia_archiver', 'Alexa'],
        ['Archive.org', 'Archive.org'],
        ['UptimeRobot', 'UptimeRobot'],
        ['W3C_Validator', 'W3C Validator'],

        // AI and Research Bots
        ['PerplexityBot', 'Perplexity'],
        ['ClaudeBot', 'Claude'],
        ['GrokBot', 'Grok (xAI)'],
        ['xAI-Bot', 'xAI'],
        ['AnthropicBot', 'Anthropic'],

        // Others
        ['Feedly', 'Feedly'],
        ['Mail.RU', 'Mail.Ru'],
        ['BLEXBot', 'BLEXBot'],
        ['PiplBot', 'Pipl'],
        ['MegaIndex', 'MegaIndex'],
        ['SeznamBot', 'Seznam'],
        ['PetalBot', 'Petal']
    ];

    $detectedBots = [];
    foreach ($crawlers as $crawler) {
        if (stripos($user_agent, $crawler[0]) !== false) {
            $detectedBots[] = $crawler[1];
        }
    }

    return $detectedBots;
}
?>