<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @package Core Event Listeners
 * @copyright 2012 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Verify Module items
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrCore_verify_module_listener($_data, $_user, $_conf, $_args, $event)
{
    // Switch to NEW temp value table from OLD temp table
    if (jrCore_db_table_exists('jrCore', 'temp') && jrCore_db_table_exists('jrCore', 'tempvalue')) {

        // We have not been migrated yet - insert values and drop old table
        $cnt = 0;
        $tbl = jrCore_db_table_name('jrCore', 'temp');
        while (true) {
            $req = "SELECT * FROM {$tbl} GROUP BY temp_module, temp_key ORDER BY temp_updated DESC LIMIT 100";
            $_rt = jrCore_db_query($req, 'NUMERIC');
            if ($_rt && is_array($_rt)) {
                $_dl = array();
                $_in = array();
                foreach ($_rt as $_t) {
                    $mod   = jrCore_db_escape($_t['temp_module']);
                    $key   = jrCore_db_escape($_t['temp_key']);
                    $_in[] = "('{$mod}','{$key}','" . $_t['temp_updated'] . "','" . jrCore_db_escape($_t['temp_value']) . "')";
                    $_dl[] = "(temp_module = '{$mod}' AND temp_key = '{$key}')";
                }
                $tbv = jrCore_db_table_name('jrCore', 'tempvalue');
                $req = "INSERT INTO {$tbv} (temp_module,temp_key,temp_updated,temp_value) VALUES " . implode(',', $_in) . ' ON DUPLICATE KEY UPDATE temp_updated = UNIX_TIMESTAMP()';
                $cnt += jrCore_db_query($req, 'COUNT');

                // Cleanup
                $req = "DELETE FROM {$tbl} WHERE " . implode(' OR ', $_dl);
                jrCore_db_query($req);

            }
            else {
                break;
            }
        }
        if ($cnt && $cnt > 0) {
            jrCore_logger('INF', "core: migrated " . jrCore_number_format($cnt) . " unique temp keys to new format");
        }
        $req = "DROP TABLE {$tbl}";
        jrCore_db_query($req);
    }

    // Migrate BBCode settings from jrForum to jrCore
    $tbl = jrCore_db_table_name('jrProfile', 'quota_value');
    $req = "UPDATE {$tbl} SET `value` = REPLACE(`value`, 'jrForum_format_string_bbcode', 'jrCore_format_string_bbcode') WHERE `name` = 'active_formatters' AND `value` LIKE '%jrForum_format_string_bbcode%'";
    jrCore_db_query($req);

    // Delete form designer fields for live search that have been entered as the MD5
    $tbl = jrCore_db_table_name('jrCore', 'form');
    $req = "SELECT * FROM {$tbl} WHERE `type` = 'live_search'";
    $_rt = jrCore_db_query($req, 'NUMERIC');
    if ($_rt && is_array($_rt)) {
        $_dl = array();
        foreach ($_rt as $f) {
            if (strpos($f['name'], 'ls') === 0 && jrCore_checktype(substr($f['name'], 2), 'md5')) {
                $_dl[] = "(`module` = '" . jrCore_db_escape($f['module']) . "' AND `view` = '" . jrCore_db_escape($f['view']) . "' AND `name` = '{$f['name']}')";
            }
        }
        if (count($_dl) > 0) {
            $req = "DELETE FROM {$tbl} WHERE " . implode(' OR ', $_dl);
            jrCore_db_query($req);
        }
    }

    return $_data;
}

/**
 * Repair Modules
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrCore_repair_module_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_mods;
    foreach ($_mods as $mod_dir => $_inf) {
        if (jrCore_is_datastore_module($mod_dir)) {
            jrCore_db_sync_datastore_profile_ids($mod_dir);
        }
    }

    // Clean up our pending table of any items that should no longer be
    // pending or are for modules that are no longer in the system
    jrCore_verify_pending_items();

    return $_data;
}

/**
 * Reset Marketplace cache
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrCore_run_view_function_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post;
    // If we are doing a marketplace update - reset cache
    if (isset($_post['module']) && $_post['module'] == 'jrMarket' && isset($_post['option']) && $_post['option'] == 'validate_modules' && jrUser_is_admin()) {
        jrCore_delete_all_cache_entries();
    }
    return $_data;
}

/**
 * Process when viewing results
 * @param $_data string incoming HTML
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return string
 */
function jrCore_view_results_listener($_data, $_user, $_conf, $_args, $event)
{
    jrCore_set_user_session_key('session_updated', time());

    // Replace emoji
    $_data = jrCore_replace_emoji($_data);

    // Add in admin javascript if we were unable to add it earlier
    if ($src = jrCore_get_flag('admin_javascript_src')) {
        $_data  = str_replace('</head>', "<script type=\"text/javascript\" src=\"{$src}\"></script>\n</head>", $_data);
    }
    return $_data;
}

/**
 * Run on process exit and used for cleanup/inserting
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrCore_process_exit_listener($_data, $_user, $_conf, $_args, $event)
{
    // Our core process exit listener handles core level cleanup
    // and tasks that should happen after a process shutdown
    // NOTE: client has disconnected at this point!
    if (jrCore_can_be_queue_worker()) {

        // Set flag used to tell Activity Log to show [system] instead of [user_name]
        jrCore_set_flag('jrcore_logger_system_user_active', 1);

        // Run maintenance check
        jrCore_maintenance_check();

        // And are queue workers enabled?
        if (jrCore_queues_are_active()) {

            // Get modules that have registered Queue Workers ...
            // jrCore_register_queue_worker('Module', 'QueueName', 'WorkerFunction', int:Queue Entries to Process, int:Max Simultaneous Workers, int:Worker Queue Timeout, int(1-9):Worker Priority);
            if ($_workers = jrCore_get_flag('jrcore_register_queue_worker')) {

                // And see if we have any queue entries
                if ($_qn = jrCore_get_ready_queues()) {

                    // Do we have a MAXIMUM number of worker processes that can be running at any one time, system wide?
                    $max = jrCore_get_config_value('jrCore', 'max_system_queue_workers', 12);
                    if ($max > 0) {
                        $worker_count = 0;
                        // How many queues are being worked RIGHT NOW?
                        foreach ($_qn as $q) {
                            if (isset($q['queue_workers']) && $q['queue_workers'] > 0) {
                                $worker_count += intval($q['queue_workers']);
                            }
                        }
                        if ($worker_count >= $max) {
                            // We are AT or OVER the number of queues allowed to be worked system wide - EXIT
                            if (jrCore_create_local_lock('jrCore', 'core_max_queue_workers_reached', 120)) {
                                jrCore_logger('CRI', "core: max queue workers ({$max}) reached - unable to work queue entries", $_qn);
                            }
                            return $_data;
                        }
                    }

                    // We have queue entries to work
                    // Conversions and other queue-based work can take a long time to run
                    set_time_limit(0);

                    foreach ($_workers as $_modules) {
                        foreach ($_modules as $mod => $_queue) {
                            foreach ($_queue as $qname => $qdat) {

                                // Only process entries for queue we actually have
                                if (!isset($_qn["{$mod}_{$qname}"])) {
                                    continue;
                                }

                                // Queue Function that is going to be run
                                $func = $qdat[0];
                                if (!function_exists($func)) {
                                    // Only alarm once every 2 minutes on any missing registered work function
                                    if (jrCore_create_local_lock('jrCore', "core_missing_worker_{$func}", 120)) {
                                        jrCore_logger('MAJ', "core: registered queue worker function: {$func} for module: {$mod} does not exist");
                                    }
                                    continue;
                                }

                                // Maximum number of workers that can be running at one time on this queue
                                $maxw = (isset($qdat[2])) ? intval($qdat[2]) : 1;

                                // Worker Timeout (in seconds) - how long this worker can work a single queue entry
                                $tout = (isset($qdat[3]) && is_numeric($qdat[3])) ? intval($qdat[3]) : 3600;

                                // Can this process become a Queue Worker?
                                if ($wpid = jrCore_queue_worker_can_work($mod, $qname, $tout, $maxw)) {

                                    // Yes - how many Queue Entries can this worker work before exiting?
                                    // NOTE: queue_worker slot for this queue has increased by 1 now
                                    $rlws = false;
                                    $qcnt = intval($qdat[1]);
                                    if ($qcnt === 0) {
                                        $qcnt = 1000000; // "0" means "all" - set this to a high number here
                                    }
                                    if ($_qn["{$mod}_{$qname}"] <= $qcnt) {
                                        // No need to run more queue checks than we have queue entries
                                        $qcnt = intval($_qn["{$mod}_{$qname}"]);
                                    }
                                    while ($qcnt > 0) {

                                        // Are we still active?
                                        if (jrCore_queues_are_active()) {

                                            // We are under the max allowed workers for this queue
                                            // and have been assigned a worker queue slot - get a queue entry
                                            $_tmp = jrCore_queue_get($mod, $qname, $wpid, null, $tout, $maxw);
                                            if ($_tmp && isset($_tmp['queue_id'])) {

                                                // Pass the queue entry to the registered queue function
                                                $ret = $func($_tmp['queue_data']);

                                                // Our queue function can return:
                                                // 1) TRUE - everything is good, delete queue entry
                                                // 3) FALSE - an issue was encountered processing the queue
                                                // 2) # - indicates we should sleep the queue entry for # number of seconds - default is an additional 30 seconds for EACH retry
                                                // 4) EXIT - force exit of queue worker loop
                                                if ($ret === true) {

                                                    // We are done working - decrement queue depth
                                                    // NOTE: worker moves on to the next queue here here so we do not decrement worker count
                                                    jrCore_queue_decrement_queue_depth($mod, $qname);

                                                    // We successfully processed our queue entry - delete it
                                                    jrCore_queue_delete($_tmp['queue_id']);
                                                    $rlws = true;

                                                }
                                                elseif ($ret === 'EXIT') {

                                                    // Forced exit by worker - NO CHANGE to queue depth
                                                    jrCore_queue_release($_tmp['queue_id']);
                                                    jrCore_queue_release_worker_slot($mod, $qname);
                                                    break 4;

                                                }
                                                elseif ($ret === 'THROTTLED') {

                                                    // "THROTTLED" is a special return condition from the Mail worker that
                                                    // tells the core queue system to sleep the queue entry for an additional
                                                    // 60 seconds, but to NOT increment the queue_count
                                                    jrCore_queue_release($_tmp['queue_id'], 60, null, false);
                                                    $rlws = true;  // "release worker slot"

                                                }
                                                elseif (jrCore_checktype($ret, 'number_nn')) {

                                                    // "SLEEP" - if we get a NUMBER returned from our queue worker it means
                                                    // that we have been told to "go back to sleep" for $ret seconds
                                                    $sec = (int) $ret;
                                                    jrCore_queue_release($_tmp['queue_id'], $sec);
                                                    $rlws = true;  // "release worker slot"

                                                }
                                                else {

                                                    // If this queue has failed 10 times, we have a problem - delete and exit
                                                    if ($_tmp['queue_count'] >= 9) {

                                                        // We are done working - decrement queue depth
                                                        // NOTE: worker moves on to the next queue here so we do not decrement worker count
                                                        jrCore_queue_decrement_queue_depth($mod, $qname);

                                                        // Delete this queue entry
                                                        jrCore_queue_delete($_tmp['queue_id']);

                                                        // Let someone know
                                                        jrCore_logger('MAJ', "core: deleted queue_id {$_tmp['queue_id']} in {$mod}/{$qname} queue - failed 10 times", $_tmp);
                                                        $rlws = true;  // "release worker slot"

                                                    }
                                                    else {
                                                        // Failed to work queue - set sleep for retry
                                                        // Depending on the number of times this queue has
                                                        // been worked, we increment by 30 seconds each time
                                                        $sec = (($_tmp['queue_count'] + 1) * 30);
                                                        jrCore_queue_release($_tmp['queue_id'], $sec);
                                                        $rlws = true;  // "release worker slot"
                                                    }

                                                }
                                                $qcnt--;

                                            }
                                            else {

                                                // We did NOT get a queue - release slot and move to next queue
                                                jrCore_queue_release_worker_slot($mod, $qname);
                                                $rlws = false;
                                                break;

                                            }
                                        }
                                        else {

                                            // Queues are NOT active - do not start another worker - release slot and exit
                                            jrCore_queue_release_worker_slot($mod, $qname);
                                            break 4;

                                        }
                                    }
                                    // END while()
                                    // If we worked a queue release our worker slot
                                    if ($rlws) {
                                        jrCore_queue_release_worker_slot($mod, $qname);
                                    }
                                }
                            }
                        }
                    }

                    // Delete profile caches that were reset during queue work
                    jrCore_process_exit_delete_profile_cache();

                }
            }
        }
    }
    return $_data;
}

/**
 * Run daily maintenance events for the Core
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return bool
 */
function jrCore_daily_maintenance_listener($_data, $_user, $_conf, $_args, $event)
{
    // Cleanup Recycle Bin
    $rb_expire = jrCore_get_config_value('jrCore', 'recycle_bin_expire', 3);
    if (!empty($rb_expire)) {

        // Delete expired recycle bin
        $old = ($rb_expire * 86400);
        $tbl = jrCore_db_table_name('jrCore', 'recycle');
        $req = "SELECT r_id, r_module AS module, r_profile_id AS profile_id, r_item_id AS item_id, r_data AS data FROM {$tbl} WHERE r_time < (UNIX_TIMESTAMP() - {$old})";
        $_rt = jrCore_db_query($req, 'r_id');
        if ($_rt && is_array($_rt)) {

            // Delete RB entries
            $req = "DELETE FROM {$tbl} WHERE r_id IN(" . implode(',', array_keys($_rt)) . ')';
            jrCore_db_query($req);

            // Remove media
            foreach ($_rt as $_i) {
                if (!empty($_i['pid']) && strpos($_i['data'], 'rb_item_media')) {
                    if (!$_fl = jrCore_get_flag("jrCore_maintenance_recycle_bin_{$_i['pid']}")) {
                        $_fl = jrCore_get_media_files($_i['pid'], 'rb_*');
                        jrCore_set_flag("jrCore_empty_recycle_bin_{$_i['pid']}", $_fl);
                    }
                    if ($_fl && is_array($_fl)) {
                        foreach ($_fl as $_file) {
                            $name = basename($_file['name']);
                            if (strpos($name, "rb_{$_i['module']}_{$_i['item_id']}_") === 0) {
                                jrCore_delete_media_file($_i['profile_id'], $name);
                            }
                        }
                    }
                }
            }

            // Trigger event for any modules that may need to manually clean up
            $_args = array(
                '_items' => $_rt
            );
            jrCore_trigger_event('jrCore', 'expire_recycle_bin', $_args);
        }
    }

    // Cleanup stats table if configured
    jrCore_delete_expired_stat_entries();

    return true;
}

/**
 * Cleanup from files uploaded via {jrCore_upload_button}
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrCore_db_create_item_exit_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post;
    if (jrCore_is_automatic_upload_handling_enabled() && isset($_post['upload_module']) && $_post['upload_module'] == $_args['module'] && isset($_post['upload_token']) && isset($_post['upload_field'])) {
        // Save any uploaded media file
        if (jrCore_is_uploaded_media_file($_post['upload_module'], $_post['upload_field'], $_data['_profile_id'])) {
            $_data['_item_id'] = $_args['_item_id'];
            if (jrCore_save_media_file($_post['upload_module'], $_post['upload_field'], $_data['_profile_id'], $_args['_item_id'], $_post['upload_field'], $_data)) {
                // Clean up any file uploads
                $dir = jrCore_get_upload_temp_directory($_post['upload_token']);
                if (is_dir($dir)) {
                    jrCore_delete_dir_contents($dir);
                    rmdir($dir);
                }
            }
        }
    }
    return $_data;
}

/**
 * Log 404 not found if configured
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrCore_404_not_found_listener($_data, $_user, $_conf, $_args, $event)
{
    global $_post;
    if (jrCore_get_config_value('jrCore', 'log_404', 'off') == 'on') {
        $uri = '/';
        if (isset($_data['_uri'])) {
            $uri = jrCore_strip_html($_data['_uri']);
        }
        // If this is a request for an old JS or CSS cache page, ignore
        if (strpos($uri, 'cart?fcsid=') || (strpos($uri, '/data/cache/') === 0 && (strpos($uri, '.js') || strpos($uri, '.css') || strpos(' ' . $uri, '.well-known')))) {
            return $_data;
        }
        $_er = array(
            '_post'    => $_post,
            'referrer' => (isset($_SERVER['HTTP_REFERER'])) ? $_SERVER['HTTP_REFERER'] : 'none',
            'client'   => (isset($_SERVER['HTTP_USER_AGENT'])) ? $_SERVER['HTTP_USER_AGENT'] : 'unknown'
        );
        jrCore_logger('MIN', 'core: 404 Page not found: ' . $uri, $_er);
    }
    return $_data;
}

/**
 * Updated module - ensure latest repair.php is setup
 * @param array $_data incoming data array
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return array
 */
function jrCore_updated_module_listener($_data, $_user, $_conf, $_args, $event)
{
    if (isset($_data['module_directory']) && $_data['module_directory'] == 'jrCore') {
        // Make sure we are running the latest copy of the repair.php.html script
        @copy(APP_DIR . '/modules/jrCore/root/repair.php.html', APP_DIR . '/repair.php.html');
        jrCore_validate_module_schema('jrCore');
        jrCore_reset_template_cache();
        jrCore_create_master_css($_conf['jrCore_active_skin']);
        jrCore_create_master_javascript($_conf['jrCore_active_skin']);
        jrCore_delete_all_cache_entries();
    }
    return $_data;
}
