<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @package Media and File
 * @copyright 2012 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

// FORCE_LOCAL - flag used to indicate that the media function should reference
// the local file system regardless of the configured active media system plugin
const FORCE_LOCAL = true;

/**
 * Return TRUE if HEIC images are supported
 * @return bool
 */
function jrCore_is_heic_supported()
{
    // Is HEIC/HEIF supported?
    // Check Image Magick First
    $pass = false;
    if (class_exists('Imagick')) {
        $_formats = \Imagick::queryformats('HEI*');
        if (is_array($_formats)) {
            $pass = 'imagick';
        }
    }
    // Fall through - check for tifig in System Tool
    if (!$pass) {
        if (jrCore_get_tool_path('tifig', 'jrSystemTools')) {
            $pass = 'tifig';
        }
    }
    return $pass;
}

/**
 * Return TRUE if module is a "media" module (accepts file uploads)
 * @param $module string module to check
 * @return bool
 */
function jrCore_is_media_module($module)
{
    return jrCore_is_datastore_module($module) && is_file(APP_DIR . "/modules/{$module}/templates/urlscan_player.tpl");
}

/**
 * Generate a unique string of X length
 * @param $length int Length of code to generate
 * @return string
 */
function jrCore_create_unique_string($length)
{
    return implode('', array_map(function () {
        return chr(mt_rand(0, 1) ? mt_rand(48, 57) : mt_rand(97, 122));
    }, range(0, ($length - 1))));
}

/**
 * Delete old upload directories
 * @param int $old_hours hours without update to be deleted
 * @return int returns number of directories removed
 */
function jrCore_delete_old_upload_directories($old_hours = 4)
{
    $rem = 0;
    $old = (int) $old_hours;
    if ($old < 1) {
        $old = 1;
    }
    $dir = jrCore_get_module_cache_dir('jrCore');
    jrCore_start_timer('filesystem');
    if ($h = opendir(realpath($dir))) {
        while (false !== ($file = readdir($h))) {
            if (is_dir("{$dir}/{$file}") && strlen($file) === 32 && jrCore_checktype($file, 'md5')) {
                // This is an upload temp directory - delete it if it has been here for over $old hours
                if (filemtime("{$dir}/{$file}") < (time() - ($old * 3600))) {
                    jrCore_delete_dir_contents("{$dir}/{$file}", true, 0, true);
                    $rem++;
                }
            }
        }
        closedir($h);
    }
    jrCore_stop_timer('filesystem');
    return $rem;
}

/**
 * Cleanup old Maintenance lock files
 * @return bool
 */
function jrCore_delete_old_maintenance_lock_files()
{
    $dir = jrCore_get_module_cache_dir('jrCore');
    jrCore_start_timer('filesystem');
    if ($_lk = glob("{$dir}/minute_maintenance*.lock", GLOB_NOSORT)) {
        $old = (time() - 180);  // 3 minutes or older
        foreach ($_lk as $file) {
            if (filemtime($file) < $old) {
                jrCore_unlink($file);  // OK
            }
        }
    }
    if ($_lk = glob("{$dir}/hourly_maintenance*.lock", GLOB_NOSORT)) {
        $old = (time() - 10800); // 3 hours or older
        foreach ($_lk as $file) {
            if (filemtime($file) < $old) {
                jrCore_unlink($file);  // OK
            }
        }
    }
    if ($_lk = glob("{$dir}/daily_maintenance*.lock", GLOB_NOSORT)) {
        $old = (time() - (3 * 86400));  // 3 days or older
        foreach ($_lk as $file) {
            if (filemtime($file) < $old) {
                jrCore_unlink($file);  // OK
            }
        }
    }
    jrCore_stop_timer('filesystem');
    return true;
}

/**
 * Delete abandoned live search results entries in the Temp table
 * @return int
 */
function jrCore_delete_old_livesearch_temp_entries()
{
    $tbl = jrCore_db_table_name('jrCore', 'tempvalue');
    $req = "DELETE FROM {$tbl} WHERE temp_updated < (UNIX_TIMESTAMP() - 28800) AND temp_key LIKE '%_live_search'";
    return jrCore_db_query($req, 'COUNT');
}

/**
 * Delete debug and error logs not written to in 3 days
 */
function jrCore_delete_old_error_and_debug_logs()
{
    $old = (time() - 259200);
    foreach (array('error_log', 'debug_log') as $log) {
        $log_file = APP_DIR . "/data/logs/{$log}";
        if (is_file($log_file) && filemtime($log_file) < $old) {
            jrCore_unlink($log_file);
        }
    }
    return true;
}

/**
 * Delete test templates that are over 30 seconds old
 */
function jrCore_delete_old_test_templates()
{
    $dir = jrCore_get_module_cache_dir('jrCore');
    if ($_tpl = glob("{$dir}/test_template_*.tpl", GLOB_NOSORT)) {
        $old = (time() - 30);  // 30 seconds
        foreach ($_tpl as $file) {
            if (filemtime($file) < $old) {
                jrCore_unlink($file);
            }
        }
    }
    return true;
}

/**
 * Delete master JS files over an hour old
 */
function jrCore_delete_old_js_files()
{
    $skn = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    if ($sum = jrCore_get_config_value('jrCore', "{$skn}_javascript_version", false)) {
        $dir = jrCore_get_module_cache_dir($skn);
        if ($_files = glob("{$dir}/*.js", GLOB_NOSORT)) {
            if (count($_files) > 30) {
                $_fl = array();
                foreach ($_files as $file) {
                    if (!strpos(' ' . $file, $sum)) {
                        $time       = filemtime($file);
                        $_fl[$file] = $time;
                    }
                }
                if (!empty($_fl)) {
                    $cnt = count($_fl);
                    if ($cnt > 30) {
                        asort($_fl);
                        $_fl = array_slice($_fl, 0, ($cnt - 30), true);
                        foreach ($_fl as $file => $time) {
                            jrCore_unlink($file);
                        }
                    }
                }
            }
        }
    }
    return true;
}

/**
 * Delete master CSS files over an hour old
 */
function jrCore_delete_old_css_files()
{
    $skn = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    if ($sum = jrCore_get_config_value('jrCore', "{$skn}_css_version", false)) {
        $dir = jrCore_get_module_cache_dir($skn);
        if ($_files = glob("{$dir}/*.css", GLOB_NOSORT)) {
            if (count($_files) > 30) {
                $_fl = array();
                foreach ($_files as $file) {
                    if (!strpos(' ' . $file, $sum)) {
                        $time       = filemtime($file);
                        $_fl[$file] = $time;
                    }
                }
                if (!empty($_fl)) {
                    $cnt = count($_fl);
                    if ($cnt > 30) {
                        asort($_fl);
                        $_fl = array_slice($_fl, 0, ($cnt - 30), true);
                        foreach ($_fl as $file => $time) {
                            jrCore_unlink($file);
                        }
                    }
                }
            }
        }
    }
    return true;
}

/**
 * Delete old datastore export CSV files
 */
function jrCore_delete_old_datastore_csv_files()
{
    $dir = jrCore_get_module_cache_dir('jrCore');
    $_ex = glob("{$dir}/*_datastore_*.csv", GLOB_NOSORT);
    if ($_ex && is_array($_ex)) {
        $old = (time() - 7200);  // 2 hours
        foreach ($_ex as $file) {
            if (filemtime($file) < $old) {
                jrCore_unlink($file);
            }
        }
    }
}

/**
 * Return TRUE if referrer is local or allowed
 * @return bool
 * @deprecated no longer used - always returns TRUE
 */
function jrCore_media_is_allowed_referrer_domain()
{
    return true;
}

/**
 * Get file content with timers
 * @param string $file
 * @return bool|string
 */
function jrCore_file_get_contents($file)
{
    jrCore_start_timer('filesystem');
    $temp = false;
    if (is_file($file)) {
        $temp = file_get_contents($file);
    }
    jrCore_stop_timer('filesystem');
    return $temp;
}

/**
 * Set media location cookie
 */
function jrCore_set_media_location_cookie()
{
    global $_post;
    $uri = (!empty($_post['_uri'])) ? $_post['_uri'] : '/';
    $val = jrCore_url_encode_string($uri);
    $exp = jrCore_get_config_value('jrCore', 'play_key_expire_hours', 8);
    $exp = (time() + (intval($exp) * 3600));
    if (jrCore_set_cookie('ml', $val, $exp, true)) {
        return true;
    }
    return false;
}

/**
 * Get the last download location for a user
 * @return bool
 */
function jrCore_get_media_location_cookie()
{
    if (jrCore_get_config_value('jrCore', 'disable_play_cookies', 'off') == 'on') {
        return true;
    }
    if ($val = jrCore_get_cookie('ml')) {
        return jrCore_url_decode_string($val);
    }
    return false;
}

/**
 * Get list of registered Media players by type
 * @param $type string one of: audio|video|mixed
 * @return array|bool
 */
function jrCore_get_registered_media_players($type)
{
    $_tmp = jrCore_get_registered_module_features('jrCore', 'media_player');
    if (!$_tmp || !is_array($_tmp)) {
        return false;
    }
    $_out = array();
    foreach ($_tmp as $module => $_players) {
        foreach ($_players as $pname => $ptype) {
            if ($ptype == $type || $type == 'all') {
                $_out[$pname] = $module;
            }
        }
    }
    if (count($_out) > 0) {
        return $_out;
    }
    return false;
}

/**
 * Get media player options registered by modules
 * @param string $type
 * @return array|bool
 */
function jrCore_get_media_player_options($type = 'all')
{
    if ($_pl = jrCore_get_registered_media_players($type)) {
        $_tm = array();
        foreach ($_pl as $n => $p) {
            $player       = trim(str_replace($p, '', $n), '_');
            $_tm[$player] = ucwords(str_replace('_', ' ', $player));
        }
        return $_tm;
    }
    return false;
}

/**
 * Uses FFMpeg to retrieve information about audio and video files
 * @param string $file File to get data for
 * @param string $field_prefix Prefix for return array keys
 * @return array|false
 */
function jrCore_get_media_file_metadata($file, $field_prefix)
{
    $ffmpeg = jrCore_get_tool_path('ffmpeg', 'jrCore');
    if (!is_file($ffmpeg)) {
        jrCore_logger('CRI', 'core: required ffmpeg binary not found in modules/jrCore/tools');
        return false;
    }
    if (!is_executable($ffmpeg)) {
        jrCore_logger('CRI', 'core: ffmpeg binary in modules/jrCore/tools is not executable!');
        return false;
    }

    // Audio (WMA)
    // Duration: 00:03:15.0, start: 1.579000, bitrate: 162 kb/s
    // Stream #0.0: Audio: wmav2, 44100 Hz, stereo, 160 kb/s

    // Audio (MP3)
    // Duration: 00:02:17.3, bitrate: 191 kb/s
    // Stream #0.0: Audio: mp3, 44100 Hz, stereo, 192 kb/s

    // Audio (M4A)
    // Duration: 00:00:28.61, start: 0.023220, bitrate: 117 kb/s
    // Stream #0:0(eng): Audio: aac (mp4a / 0x6134706D), 44100 Hz, stereo, s16, 116 kb/s

    // Audio (FLAC)
    // Duration: N/A, bitrate: N/A
    // Duration: 00:05:27.76, start: 0.000000, bitrate: 728 kb/s
    // Stream #0.0: Audio: flac, 44100 Hz, stereo

    // Audio (OGG)
    // Duration: 00:00:27.9, start: 0.686440, bitrate: 91 kb/s
    // Stream #0.0: Audio: vorbis, 44100 Hz, stereo, 96 kb/s

    // Video (WMV)
    // Duration: 00:02:39.9, start: 4.000000, bitrate: 913 kb/s
    // Stream #0.0: Audio: wmav2, 48000 Hz, stereo, 128 kb/s
    // Stream #0.1: Video: wmv3, yuv420p, 640x512, 766 kb/s, 25.00 fps(r)

    // Video MOV (with unsupported M4A audio)
    // Duration: 00:01:19.6, start: 0.000000, bitrate: 1288 kb/s
    // Stream #0.0(eng): Video: h264, yuv420p, 640x368 [PAR 0:1 DAR 0:1], 25.00 tb(r)
    // Stream #0.1(eng): Audio: mp4a / 0x6134706D, 11025 Hz, mono

    // Video (FLV)
    // Duration: 08:25:32.0, start: 0.000000, bitrate: 64 kb/s
    // Stream #0.0: Video: flv, yuv420p, 320x240, 25.00 fps(r)
    // Stream #0.1: Audio: mp3, 22050 Hz, stereo, 64 kb/s

    // Stream #0:0(eng): Video: mpeg4 (Simple Profile) (mp4v / 0x7634706D), yuv420p, 176x144 [SAR 1:1 DAR 11:9], 122 kb/s, 29.97 fps, 29.97 tbr, 90k tbn, 30k tbc

    // Metadata:
    //  encoder         : Audiograbber 1.81.03, LAME dll 3.92, 160 Kbit/s, Joint Stereo, Normal quality
    //  title           : Pastichio Medley
    //  artist          : Smashing Pumpkins
    //  publisher       : Hut
    //  genre           : Rock
    //  album           : The Aeroplane Flies High - Zero
    //  track           : 7
    //  album_artist    : Smashing Pumpkins
    //  composer        : Billy Corgan
    //  date            : 1996
    // Duration: 00:23:00.44, start: 0.000000, bitrate: 160 kb/s

    ob_start();
    jrCore_start_timer('metadata');
    exec("nice -n 9 {$ffmpeg} -analyzeduration 30000000 -probesize 30000000 -i " . escapeshellarg($file) . " 2>&1", $_tmp);
    jrCore_stop_timer('metadata');
    ob_end_clean();

    $_out = array();
    $meta = false;
    if ($_tmp && is_array($_tmp)) {
        foreach ($_tmp as $line) {
            $line = trim($line);
            if (strpos($line, 'Duration:') === 0) {
                $meta = false;
                // Duration: 00:07:21.18, start: 0.000000, bitrate: 128 kb/s
                $length = jrCore_string_field($line, 2);
                if (strpos($length, '.')) {
                    list($sec,) = explode('.', $length, 2);
                    if (strlen($sec) >= 8) {
                        $length = $sec;
                    }
                }
                else {
                    $length = substr($length, 0, 8);
                }
                $_out["{$field_prefix}_length"] = $length;

                // FLAC's bitrate will only be found on the duration line
                // Duration: 00:05:27.76, bitrate: 728 kb/s
                // Get bitrate
                $bitrate = false;
                $prv     = false;
                foreach (explode(' ', $line) as $prt) {
                    if (strtolower($prt) == 'kb/s') {
                        $bitrate = (int) $prv;
                        break;
                    }
                    $prv = $prt;
                }
                if (jrCore_checktype($bitrate, 'number_nz')) {
                    $_out["{$field_prefix}_bitrate"] = (int) $bitrate;
                }
            }
            elseif (strpos($line, 'Stream') === 0 && strpos($line, 'Audio') && !isset($save)) {
                $meta = false;
                $line = trim(str_replace(array('(default)', ','), '', $line));
                // Stream #0:0: Audio: mp3, 44100 Hz, stereo, s16, 128 kb/s
                // Stream #0:0(eng): Audio: aac (mp4a / 0x6134706D), 44100 Hz, stereo, fltp, 116 kb/s (default)

                // Get bitrate
                $bitrate = false;
                $prv     = false;
                foreach (explode(' ', $line) as $prt) {
                    if (strtolower($prt) == 'kb/s') {
                        $bitrate = (int) $prv;
                        break;
                    }
                    $prv = $prt;
                }
                if (jrCore_checktype($bitrate, 'number_nz')) {
                    $_out["{$field_prefix}_bitrate"] = (int) $bitrate;
                }
                // Get smprate
                $smprate = false;
                $prv     = false;
                foreach (explode(' ', $line) as $prt) {
                    if (strtolower($prt) == 'hz') {
                        $smprate = (int) $prv;
                        break;
                    }
                    $prv = $prt;
                }
                if (jrCore_checktype($smprate, 'number_nz')) {
                    $_out["{$field_prefix}_smprate"] = (int) $smprate;
                }
            }
            elseif (strpos($line, 'Audio:') === 0 && !isset($save)) {
                $meta = false;
                // Stream #0:0: Audio: mp3, 44100 Hz, stereo, s16, 128 kb/s
                $_out["{$field_prefix}_bitrate"] = (int) jrCore_string_field($line, -2);
                $_out["{$field_prefix}_smprate"] = (int) jrCore_string_field($line, 5);
            }
            elseif (strpos($line, 'Video:') && !strpos($line, 'Video: png')) {
                $meta = false;
                $save = false;
                // This is a video file - get our details
                foreach (explode(' ', $line) as $word) {
                    if (strtolower($word) == 'kb/s,') {
                        $_out["{$field_prefix}_bitrate"] = $save;
                    }
                    elseif (strpos($word, 'x')) {
                        $_wrd = explode('x', $word);
                        if (count($_wrd) === 2 && (strlen($_wrd[0]) > 1 && strlen($_wrd[0]) < 5) && (strlen($_wrd[1]) > 1 && strlen($_wrd[1]) < 5)) {
                            $_out["{$field_prefix}_resolution"] = trim(str_replace(',', '', $word));
                        }
                    }
                    $save = $word;
                }
            }
            elseif (strpos($line, 'Metadata:') === 0) {
                $meta = true;
            }
            elseif ($meta && strpos($line, ':')) {
                list($tag, $val) = explode(':', $line, 2);
                $tag = strip_tags(trim($tag));
                switch ($tag) {
                    case 'title':
                    case 'artist':
                    case 'composer':
                    case 'publisher':
                    case 'album':
                    case 'genre':
                    case 'date':
                        if (!isset($_out["{$field_prefix}_{$tag}"])) {
                            // Take the first one
                            $_out["{$field_prefix}_{$tag}"] = jrCore_strip_html(trim($val));
                        }
                        break;
                    case 'track':
                        // Our "track" becomes our order field used for
                        // ordering of items in albums.  Note that some track fields
                        // can contain a '/' - i.e. 5/12 - we only want the first
                        if (!isset($_out["{$field_prefix}_track"])) {
                            $val = trim(strip_tags($val));
                            if (strpos($val, '/')) {
                                list($val,) = explode('/', $val, 2);
                            }
                            $_out["{$field_prefix}_track"] = intval($val);
                        }
                        break;
                }
            }
        }
    }
    return $_out;
}

/**
 * Get full path to a media file
 * @param $module string Module Name to save file for
 * @param $field_name string Unique File Name field
 * @param $_item array Array of item information from jrCore_db_get_item()
 * @return string
 */
function jrCore_get_media_file_path($module, $field_name, $_item)
{
    // We don't need a PLUGIN for this function - it works
    // because $dir will change depending on the media plugin!
    if (!isset($_item["{$field_name}_size"])) {
        return false;
    }
    $dir = jrCore_get_media_directory($_item['_profile_id']);
    return "{$dir}/{$module}_{$_item['_item_id']}_{$field_name}." . $_item["{$field_name}_extension"];
}

/**
 * Get full URL to a media file
 * @param $module string Module Name to save file for
 * @param $file_name string Unique File Name field
 * @param $_item array Array of item information from jrCore_db_get_item()
 * @return string
 */
function jrCore_get_media_file_url($module, $file_name, $_item)
{
    // NOTE: This function simply returns the URL - it is up to the
    // calling module to ensure the URL is actually accessible, as
    // the core places restrictions on direct access to media items
    if (!isset($_item["{$file_name}_size"])) {
        return false;
    }
    $url = jrCore_get_media_url($_item['_profile_id']);
    return "{$url}/{$module}_{$_item['_item_id']}_{$file_name}." . $_item["{$file_name}_extension"];
}

/**
 * Download and save a media file for a profile/item_id
 * @param string $url URL to download file
 * @param string $module Module to save file for
 * @param int $profile_id Profile ID
 * @param int $item_id Unique DS Item ID
 * @param string $field Field name
 * @param bool $overwrite Set to TRUE to overwrite an existing field/file
 * @return bool
 */
function jrCore_download_and_save_profile_media_file($url, $module, $profile_id, $item_id, $field, $overwrite = true)
{
    if (jrCore_is_datastore_module($module)) {
        $iid = (int) $item_id;
        $_ex = null;
        if (!$overwrite) {
            if (!$_ex = jrCore_db_get_item($module, $iid, true)) {
                $_ex = null;
            }
        }
        $pid = (int) $profile_id;
        $cdr = jrCore_get_module_cache_dir('jrCore');
        $tmp = "{$cdr}/{$module}_{$pid}_{$iid}_{$field}";
        if (jrCore_download_file($url, $tmp)) {
            if (!$_tmp = jrCore_save_media_file($module, $tmp, $pid, $iid, $field, $_ex)) {
                jrCore_unlink($tmp);
                return false;
            }
            jrCore_unlink($tmp);
            return $_tmp;
        }
    }
    return false;
}

/**
 * Delete attached media file(s) for a given item_id
 * @note Despite the name, this function will remove ALL files for the given profile_id/item_id/file_name
 * @param string $module Module Name to save file for
 * @param string $file_name Name of file field in form
 * @param int $profile_id the Profile ID to save the media file for
 * @param int $unique_id Unique Item ID from DataStore
 * @param bool $item_check set to FALSE to not check that the DS item is deleted as well
 * @return bool
 */
function jrCore_delete_item_media_file($module, $file_name, $profile_id, $unique_id, $item_check = true)
{
    if (!jrCore_checktype($unique_id, 'number_nz')) {
        return false;
    }
    $type = jrCore_get_active_media_system();
    $func = "_{$type}_media_delete_file";
    if (function_exists($func)) {

        // Delete media file keys from DataStore
        if ($item_check) {
            $_it = jrCore_db_get_item($module, $unique_id, SKIP_TRIGGERS);
            if ($_it && is_array($_it)) {
                $_ky = array();
                foreach ($_it as $k => $v) {
                    if (preg_match("/^{$file_name}_[^_]+$/", $k)) {
                        $_ky[] = $k;
                    }
                }
                if (count($_ky) > 0) {
                    jrCore_db_delete_multiple_item_keys($module, $unique_id, $_ky, false);
                }
            }
        }
        return $func($module, $file_name, $profile_id, $unique_id);

    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Disable the automatic upload handling by the Core
 * @return bool
 */
function jrCore_disable_automatic_upload_handling()
{
    return jrCore_set_flag('jrcore_disable_automatic_upload_handling', 1);
}

/**
 * Check if automatic upload handling is enabled
 * @return bool
 */
function jrCore_is_automatic_upload_handling_enabled()
{
    return !(jrCore_get_flag('jrcore_disable_automatic_upload_handling') === 1);
}

/**
 * Delete the temp directory used for file uploads in a form
 * @param null $upload_token string MD5 hash
 * @return bool
 */
function jrCore_delete_upload_temp_directory($upload_token = null)
{
    global $_post;
    if (is_null($upload_token)) {
        if (isset($_post['upload_token']) && strlen($_post['upload_token']) === 32 && jrCore_checktype($_post['upload_token'], 'md5')) {
            $upload_token = $_post['upload_token'];
        }
    }
    if (!is_null($upload_token)) {
        $dir = jrCore_get_upload_temp_directory($upload_token);
        jrCore_delete_dir_contents($dir, true, 0, true);
        return true;
    }
    return false;
}

/**
 * Get the upload temp directory used for file uploads
 * @param $upload_token string MD5 upload token
 * @return string|false
 */
function jrCore_get_upload_temp_directory($upload_token)
{
    if (jrCore_checktype($upload_token, 'md5')) {
        $dir = jrCore_get_module_cache_dir('jrCore');
        return "{$dir}/{$upload_token}";
    }
    return false;
}

/**
 * Get uploaded media files
 * @param string $module
 * @param string $file_name
 * @return array|false
 * @noinspection PhpReturnDocTypeMismatchInspection
 */
function jrCore_get_uploaded_media_files($module = null, $file_name = null)
{
    global $_post;
    if (!empty($_post['upload_token']) && jrCore_checktype($_post['upload_token'], 'md5')) {
        $type = jrCore_get_active_media_system();
        $func = jrCore_get_active_media_function($type, 'get_uploaded_media_files', func_get_args());
        if (function_exists($func)) {
            return $func($_post['upload_token'], $module, $file_name);
        }
        jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    }
    return false;
}

/**
 * Saves all uploaded media files for a given Item ID
 * @note: $view is not used in this function but is there for backwards compatibility
 * @param string $module Module Name to save file for
 * @param string $view View to save files for
 * @param int $profile_id the Profile ID to save the media file for
 * @param int $unique_id Unique Item ID from DataStore
 * @param array $_existing Item Array (for update checking)
 * @return bool
 */
function jrCore_save_all_media_files($module, $view, $profile_id, $unique_id, $_existing = null)
{
    global $_post;
    if (!empty($_post['jr_html_form_token'])) {
        $_form = jrCore_form_get_session($_post['jr_html_form_token']);
        if (!isset($_form['form_fields']) || !is_array($_form['form_fields'])) {
            // Nothing submitted
            return true;
        }
        foreach ($_form['form_fields'] as $_field) {
            switch ($_field['type']) {
                case 'file':
                case 'image':
                case 'audio':
                case 'video':
                    // @note: jrCore_is_uploaded_media_file will return the
                    // NUMBER of uploaded files for the given field (could be > 1)
                    if (jrCore_is_uploaded_media_file($module, $_field['name'], $profile_id)) {
                        jrCore_save_media_file($module, $_field['name'], $profile_id, $unique_id, null, $_existing);
                    }
                    break;
            }
        }
    }
    return true;
}

/**
 * Save an Uploaded media file
 * @param string $module
 * @param string $file_name
 * @param int $profile_id
 * @param int $unique_id
 * @param null $field
 * @param null $_existing
 * @return bool
 */
function jrCore_save_media_file($module, $file_name, $profile_id, $unique_id, $field = null, $_existing = null)
{
    if (!$unique_id || !jrCore_checktype($unique_id, 'number_nz')) {
        return false;
    }
    // make sure this module is using a DataStore
    if (jrCore_db_get_prefix($module) === false) {
        jrCore_logger('CRI', "core: {$module} is not a datastore module - unable to handle uploads automatically");
        return false;
    }
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'save_media_file', func_get_args());
    if (function_exists($func)) {
        return $func($module, $file_name, $profile_id, $unique_id, $field, $_existing);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Get uploaded files for a given upload token
 * @param string $upload_token MD5 upload token
 * @return array|false
 * @noinspection PhpReturnDocTypeMismatchInspection
 */
function jrCore_get_uploaded_meter_files($upload_token)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'get_uploaded_meter_files', func_get_args());
    if (function_exists($func)) {
        return $func($upload_token);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * @return array
 * @ignore
 * jrCore_get_media_system_plugins
 */
function jrCore_get_media_system_plugins()
{
    return jrCore_get_system_plugins('media');
}

/**
 * jrCore_get_active_media_system
 * @return string
 */
function jrCore_get_active_media_system()
{
    return jrCore_get_config_value('jrCore', 'active_media_system', 'jrCore_local');
}

/**
 * Get the active media function
 * @param string $type active media system prefix
 * @param string $func function to get
 * @param array $_args function arguments
 * @return mixed
 */
function jrCore_get_active_media_function($type, $func, $_args = null)
{
    $temp = "_{$type}_{$func}";
    if (!function_exists($temp)) {
        $temp = "_jrCore_local_{$func}";
    }
    $_tmp = array('type' => $type, 'func' => $func, 'active_function' => $temp);
    $_tmp = jrCore_trigger_event('jrCore', 'get_media_function', $_tmp, $_args);
    if (isset($_tmp['active_function'])) {
        return $_tmp['active_function'];
    }
    return $temp;
}

/**
 * Get the media directory for a Profile ID
 * @param $profile_id int Profile ID to get media directory for
 * @param $force_local bool set to TRUE to force local file system
 * @return bool
 */
function jrCore_get_media_directory($profile_id, $force_local = false)
{
    if ($profile_id !== 'system' && !jrCore_checktype($profile_id, 'number_nn')) {
        return false;
    }
    $type = ($force_local) ? 'jrCore_local' : jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_get_directory', func_get_args());
    if (function_exists($func)) {
        return $func($profile_id);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Get the media URL for a Profile ID
 * @param $profile_id int Profile ID to get media directory for
 * @param $force_local bool set to TRUE to force local file system
 * @return bool
 */
function jrCore_get_media_url($profile_id, $force_local = false)
{
    $type = ($force_local) ? 'jrCore_local' : jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_get_url', func_get_args());
    if (function_exists($func)) {
        return $func($profile_id);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Create a new Media Directory for a Profile ID
 * @param $profile_id int Profile ID to get media directory for
 * @param $force_local bool set to TRUE to force local file system
 * @return bool
 */
function jrCore_create_media_directory($profile_id, $force_local = false)
{
    $type = ($force_local) ? 'jrCore_local' : jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_create_directory', func_get_args());
    if (function_exists($func)) {
        return $func($profile_id);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Delete a Media Directory for a Profile ID
 * @param $profile_id int Profile ID to get media directory for
 * @param $force_local bool set to TRUE to force local file system
 * @return bool
 */
function jrCore_delete_media_directory($profile_id, $force_local = false)
{
    $type = ($force_local) ? 'jrCore_local' : jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_delete_directory', func_get_args());
    if (function_exists($func)) {
        return $func($profile_id);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Gets the number of uploaded media files
 * @param string $module Module Name to check file for
 * @param string $file_name Name of file field in form
 * @param int $profile_id the Profile ID the file(s) were uploaded under
 * @return bool
 */
function jrCore_is_uploaded_media_file($module, $file_name, $profile_id)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'is_uploaded_media_file', func_get_args());
    if (function_exists($func)) {
        return $func($module, $file_name, $profile_id);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Gets the number of uploaded media files
 * @param mixed $source
 * @param string $target
 * @return bool
 */
function jrCore_move_uploaded_media_file($source, $target)
{
    @ini_set('max_execution_time', 120);  // shouldn't need longer than 2 minutes to copy over data
    if ($source == 'php://input') {
        return jrCore_chunked_copy($source, $target);
    }
    jrCore_start_timer('filesystem');
    if (is_uploaded_file($source)) {
        if (move_uploaded_file($source, $target)) {
            jrCore_stop_timer('filesystem');
            return true;
        }
        jrCore_stop_timer('filesystem');
    }
    return false;
}

/**
 * Clean up "old" (replaced) media items in a profile directory
 * @param $profile_id int Profile ID
 * @param $module string Module Name
 * @param $unique_id int Item_ID
 * @param $field string Unique item field name
 * @param null $_exclude option array of file names to exclude
 * @return bool
 */
function jrCore_delete_old_media_items($profile_id, $module, $unique_id, $field, $_exclude = null)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_delete_old_items', func_get_args());
    if (function_exists($func)) {
        return $func($profile_id, $module, $unique_id, $field, $_exclude);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Get size of a Profile media directory
 * @param int $profile_id Profile ID to create media directory for
 * @return bool
 */
function jrCore_get_media_directory_size($profile_id)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_get_directory_size', func_get_args());
    if (function_exists($func)) {
        return $func($profile_id);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Get an array of all media files in a profile directory
 * @param int $profile_id Profile ID
 * @param string $pattern optional glob pattern
 * @return array|bool Returns array of files
 * @noinspection PhpReturnDocTypeMismatchInspection
 */
function jrCore_get_media_files($profile_id, $pattern = null)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_get_files', func_get_args());
    if (function_exists($func)) {
        return $func($profile_id, $pattern);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Read a file from a profile directory
 * @param int $profile_id Profile ID
 * @param string $file File name to read
 * @param string $save_as Save AS to alternate path
 * @return bool Returns True/False
 */
function jrCore_read_media_file($profile_id, $file, $save_as = null)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_read', func_get_args());
    if (function_exists($func)) {
        return $func($profile_id, $file, $save_as);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Write a file to a profile directory
 * @param int $profile_id Profile ID
 * @param string $file File to write data to
 * @param string $data Data to write to file
 * @param string $access Access permissions
 * @return bool
 */
function jrCore_write_media_file($profile_id, $file, $data, $access = null)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_write', func_get_args());
    if (function_exists($func)) {
        return $func($profile_id, $file, $data, $access);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Delete a file from a profile directory
 * @param int $profile_id Profile ID
 * @param string $file File name to delete
 * @return bool
 */
function jrCore_delete_media_file($profile_id, $file)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_delete', func_get_args());
    if (function_exists($func)) {
        return $func($profile_id, $file);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * wrapper function to check if a file of the specified file type exists.
 * @param int $profile_id Profile ID
 * @param string $file_name File name to check
 * @return bool
 */
function jrCore_media_file_exists($profile_id, $file_name)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_exists', func_get_args());
    if (function_exists($func)) {
        return $func($profile_id, basename($file_name));
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Confirm a media file is on the local server
 * @param int $profile_id Profile ID
 * @param string $file Unique File Name field
 * @param string $local_save save to this file instead of the media directory
 * @param bool $force set to TRUE to force overwrite of existing local file
 * @param int $size set to expected master file size
 * @return string|bool
 * @noinspection PhpReturnDocTypeMismatchInspection
 */
function jrCore_confirm_media_file_is_local($profile_id, $file, $local_save = null, $force = false, $size = 0)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_confirm_is_local', func_get_args());
    if (function_exists($func)) {
        return $func($profile_id, $file, $local_save, $force, $size);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * jrCore_media_file_stream
 * @param int $profile_id Profile ID
 * @param string $file File name to Download
 * @param string $send_name File name to use in download dialog
 * @return bool
 */
function jrCore_media_file_stream($profile_id, $file, $send_name)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_stream', func_get_args());
    if (function_exists($func)) {
        jrCore_db_close();
        return $func($profile_id, $file, $send_name);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * The jrCore_media_file_download function will download a media file
 * @param string $profile_id Profile ID
 * @param string $file File name to Download
 * @param string $send_name File name to use in download dialog
 * @return bool
 */
function jrCore_media_file_download($profile_id, $file, $send_name)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_download', func_get_args());
    if (function_exists($func)) {
        jrCore_db_close();
        return $func($profile_id, $file, $send_name);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Copy an existing file to a media file in a profile directory
 * @param int $target_profile_id target profile_id
 * @param string $source_file file to copy - full path to file OR file name in profile media directory
 * @param string $target_file_name file to copy to - file name to copy TO in profile media directory
 * @return bool
 */
function jrCore_copy_media_file($target_profile_id, $source_file, $target_file_name)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_copy', func_get_args());
    if (function_exists($func)) {
        return $func($target_profile_id, $source_file, $target_file_name);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * The jrCore_rename_media_file function is a media wrapper
 * @param string $profile_id Directory file is located in
 * @param string $file File old (existing) name
 * @param string $new_name File new name
 *
 * @return bool Returns True/False
 */
function jrCore_rename_media_file($profile_id, $file, $new_name)
{
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'media_rename', func_get_args());
    if (function_exists($func)) {
        return $func($profile_id, $file, $new_name);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Copy an uploaded file for a given field to a new location
 * @param array $_uploads output of jrCore_get_uploaded_meter_files()
 * @param string $field upload field name
 * @param string $target target to save to
 * @return bool
 * @since 6.5.0b9
 */
function jrCore_copy_uploaded_file($_uploads, $field, $target)
{
    if (!is_array($_uploads)) {
        return false;
    }
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'copy_uploaded_file', func_get_args());
    if (function_exists($func)) {
        return $func($_uploads, $field, $target);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

/**
 * Delete a single uploaded file
 * @param array $_uploads output of jrCore_get_uploaded_meter_files()
 * @param string $field upload field name
 * @return bool
 * @since 6.5.0b9
 */
function jrCore_delete_uploaded_file($_uploads, $field)
{
    if (!is_array($_uploads)) {
        return false;
    }
    $type = jrCore_get_active_media_system();
    $func = jrCore_get_active_media_function($type, 'delete_uploaded_file', func_get_args());
    if (function_exists($func)) {
        return $func($_uploads, $field);
    }
    jrCore_logger('CRI', "core: required plugin media function: {$func} does not exist");
    return false;
}

//--------------------------------------
// Local FileSystem media plugins
//--------------------------------------

/**
 * Get media files that have been uploaded from a form
 * NOTE: $module is no longer used in this function, but was
 * at one time so is left there for backwards compatibility
 * @param string $upload_token MD5 upload token
 * @param string $module Module Name to check file for
 * @param string $file_name Name of file field in form
 * @return array|false
 */
function _jrCore_local_get_uploaded_media_files($upload_token, $module = null, $file_name = null)
{
    $dir = jrCore_get_upload_temp_directory($upload_token);
    if (is_dir($dir)) {
        jrCore_start_timer('filesystem');
        if (is_null($file_name)) {
            $_tmp = glob("{$dir}/*.tmp", GLOB_NOSORT);
        }
        else {
            $_tmp = glob("{$dir}/*_{$file_name}.tmp", GLOB_NOSORT);
        }
        jrCore_stop_timer('filesystem');
        if ($_tmp && is_array($_tmp) && count($_tmp) > 0) {
            foreach ($_tmp as $k => $v) {
                $_tmp[$k] = substr($v, 0, strlen($v) - 4);
            }
            natcasesort($_tmp);
            return $_tmp;
        }
    }
    return false;
}

/**
 * Saves an uploaded media file to the proper profile directory
 * @param string $module Module Name to save file for
 * @param string $file_name Name of file field in form
 * @param int $profile_id the Profile ID to save the media file for
 * @param int $unique_id Unique Item ID from DataStore
 * @param string $field Field to save as (defaults to field name from file)
 * @param array $_existing Item Array (for update checking)
 * @return mixed array|bool
 */
function _jrCore_local_save_media_file($module, $file_name, $profile_id, $unique_id, $field = null, $_existing = null)
{
    global $_post;
    $_up = array();

    // See if we have been given a FULL PATH FILE - if so, that's the one we use,
    // otherwise we figure it out from the current post.
    if (is_file($file_name)) {
        // 1_audio_file_2
        if (is_null($field)) {
            list(, $field) = explode('_', basename($file_name), 2);
        }
        if (is_file("{$file_name}.tmp")) {
            $fname = trim(jrCore_file_get_contents("{$file_name}.tmp"));
        }
        else {
            $fname = basename($file_name);
        }
        $_up[$field] = array(
            'tmp_name' => $file_name,
            'name'     => $fname,
            'size'     => filesize($file_name),
            'type'     => jrCore_mime_type($fname),
            'error'    => 0
        );
    }
    else {
        // Don't mess with this structure or multiple uploads will fail!
        if (!empty($_post['upload_token'])) {
            if ($_up = jrCore_get_uploaded_meter_files($_post['upload_token'])) {
                if ($file_name !== false) {
                    foreach ($_up as $fname => $_file) {
                        $ofs = preg_replace('/_[0-9]+$/', '', $fname);
                        if ($ofs != $file_name) {
                            unset($_up[$fname]);
                        }
                    }
                }
            }
        }
    }

    // Check that we have something
    if (!is_array($_up) || count($_up) === 0) {
        return false;
    }

    // Let other modules manipulate the files if needed
    $_up = jrCore_trigger_event('jrCore', 'save_media_file_uploads', $_up);

    // If we get an $_existing item it means we are ADDING files to
    // an existing item and NOT replacing what is already there
    if (is_array($_existing)) {
        $fnd = false;
        $num = 0;
        $pfx = jrCore_db_get_prefix($module);
        foreach ($_existing as $k => $v) {
            if (strpos($k, $pfx) === 0 && strpos($k, '_extension')) {
                // This is a FILE field - get index
                $ofs = str_replace('_extension', '', $k);
                $ofs = (int) preg_replace('/[^0-9]/', '', $ofs);
                if ($ofs > $num) {
                    $num = $ofs;
                }
                $fnd = true;
            }
        }
        if ($fnd) {
            $num++;
        }
        $_nw = array();
        foreach ($_up as $f => $_inf) {
            $f = preg_replace('/_[0-9]+/', '', $f);
            if ($num > 0) {
                $_nw["{$f}_{$num}"] = $_inf;
            }
            else {
                $_nw[$f] = $_inf;
            }
            $num++;
        }
        $_up = $_nw;
        unset($_nw);
    }

    $_data = false;
    // Save off each media file that was uploaded
    foreach ($_up as $fname => $_file) {

        $ext = jrCore_file_extension($_file['name']);
        // If we do NOT have a file extension, we need to grab the mime type and add the file extension on
        if (!$ext || strlen($ext) === 0 || strlen($ext) > 4) {
            $typ = jrCore_mime_type($_file['tmp_name']);
            $ext = jrCore_file_extension_from_mime_type($typ);
        }

        $nam = "{$module}_{$unique_id}_{$fname}.{$ext}";
        if (!jrCore_write_media_file($profile_id, $nam, $_file['tmp_name'])) {
            jrCore_logger('CRI', "core: error saving media file: {$profile_id}/{$nam}");
            return false;
        }

        // We need to cleanup old files if we are adding to an item
        if (is_array($_existing)) {
            jrCore_delete_old_media_items($profile_id, $module, $unique_id, $fname, array($nam));
        }

        // Okay we've saved it.  Next, we need to update the datastore
        // entry with the info from the file
        $pdir      = jrCore_get_media_directory($profile_id, FORCE_LOCAL);
        $save_name = $_file['name'];
        if (!stripos($save_name, ".{$ext}")) {
            $save_name = "{$save_name}.{$ext}";
        }
        $_data = array(
            "{$fname}_time"      => 'UNIX_TIMESTAMP()',
            "{$fname}_name"      => $save_name,
            "{$fname}_size"      => $_file['size'],
            "{$fname}_type"      => jrCore_mime_type($nam),
            "{$fname}_extension" => $ext
        );

        // We have some extra info we want to make available to our listeners,
        // but we don't want it to be part of the data
        $_args = array(
            'module'     => $module,
            'file_name'  => $fname,
            'profile_id' => $profile_id,
            'unique_id'  => $unique_id,
            'saved_file' => "{$pdir}/{$nam}",
            '_file'      => $_file,
            '_existing'  => $_existing
        );

        // Trigger our save media file event
        $_data = jrCore_trigger_event('jrCore', 'save_media_file', $_data, $_args);
        jrCore_set_flag("jrcore_created_pending_item_{$module}_{$unique_id}", 1);
        jrCore_db_update_item($module, $unique_id, $_data);
    }
    return $_data;
}

/**
 * Move all files uploaded by the meter into an array
 * @param string $upload_token Form Token
 * @return array
 */
function _jrCore_local_get_uploaded_meter_files($upload_token)
{
    $_up = false;
    $dir = jrCore_get_upload_temp_directory($upload_token);
    if (is_dir($dir)) {
        // We've got uploaded files via the progress meter
        jrCore_start_timer('filesystem');
        // @note do NOT use GLOB_NOSORT here!
        $_tmp = glob("{$dir}/*");
        // [0] => data/cache/jrCore/12046f3177d5079e5528aa7a34175c73/1_audio_file       <- contains actual file
        // [1] => data/cache/jrCore/12046f3177d5079e5528aa7a34175c73/1_audio_file.tmp   <- contains file name
        if ($_tmp && is_array($_tmp)) {
            $_up = array();
            $_nm = array();
            foreach ($_tmp as $file) {
                if (is_file($file)) {
                    $ext = jrCore_file_extension($file);
                    if ($ext == 'tmp') {
                        list($f_num, $field) = explode('_', basename($file), 2);
                        $field = str_replace('.tmp', '', $field);
                        $fname = jrCore_file_get_contents($file);
                        $fdata = "{$dir}/{$f_num}_{$field}";
                        if (is_file($fdata)) {
                            $key = $field;
                            if (!isset($_nm[$field])) {
                                $_nm[$field] = 0;
                            }
                            if ($_nm[$field] > 0) {
                                $key = "{$field}_" . $_nm[$field];
                            }
                            $_up[$key] = array(
                                'tmp_name' => $fdata,
                                'name'     => $fname,
                                'size'     => filesize($fdata),
                                'type'     => jrCore_mime_type($fname),
                                'error'    => 0
                            );
                            $_nm[$field]++;
                        }
                    }
                }
            }
        }
        jrCore_stop_timer('filesystem');
    }
    return $_up;
}

/**
 * The _local_media_get_directory_group function will return the
 * directory "group" that a given profile_id belongs to.  This is
 * to overcome ext3 limitations on dirs in dirs.
 * @param int $profile_id Profile ID
 * @return false|int|string Returns string on success, bool false on failure
 */
function _jrCore_local_media_get_directory_group($profile_id)
{
    if (jrCore_checktype($profile_id, 'number_nn')) {
        return intval(ceil($profile_id / 1000));
    }
    elseif ($profile_id === 'system') {
        return '1';
    }
    return false;
}

/**
 * Local FileSystem Get Media Directory function
 * @param int $profile_id Profile ID
 * @return string
 */
function _jrCore_local_media_get_directory($profile_id)
{
    if ($profile_id === 'system') {
        $group_dir  = 1;
        $profile_id = 0;
    }
    else {
        $group_dir = _jrCore_local_media_get_directory_group($profile_id);
    }
    $media_dir = APP_DIR . "/data/media/{$group_dir}/{$profile_id}";
    if (!is_dir($media_dir)) {
        jrCore_create_directory($media_dir, true);
    }
    return $media_dir;
}

/**
 * Local FileSystem Get Media URL function
 * @param int $profile_id Profile ID
 * @return string
 */
function _jrCore_local_media_get_url($profile_id)
{
    if ($profile_id === 'system') {
        $group_dir  = 1;
        $profile_id = 0;
    }
    else {
        $group_dir = _jrCore_local_media_get_directory_group($profile_id);
    }
    return jrCore_get_base_url() . "/data/media/{$group_dir}/{$profile_id}";
}

/**
 * Local FileSystem Get Media Directory function
 * @param int $profile_id Profile ID
 * @return bool
 */
function _jrCore_local_media_create_directory($profile_id)
{
    global $_conf;
    // First our media directory
    $media_dir = _jrCore_local_media_get_directory($profile_id);

    if (!is_dir($media_dir)) {
        if (!jrCore_create_directory($media_dir, true)) {
            jrCore_logger('CRI', 'core: unable to create local profile media directory: ' . str_replace(APP_DIR . '/', '', $media_dir));
            return false;
        }
    }
    if (!is_writable($media_dir)) {
        if (!chmod($media_dir, $_conf['jrCore_dir_perms'])) {
            jrCore_logger('CRI', 'core: unable to permission local profile media directory: ' . str_replace(APP_DIR . '/', '', $media_dir));
            return false;
        }
    }
    return true;
}

/**
 * Local FileSystem Delete Media Directory function
 * @param int $profile_id Profile ID
 * @return bool
 */
function _jrCore_local_media_delete_directory($profile_id)
{
    $media_dir = _jrCore_local_media_get_directory($profile_id);
    jrCore_delete_dir_contents($media_dir, false, 0, true);
    return true;
}

/**
 * Local filesystem get count of uploaded media files
 * @param string $module Module Name to check file for
 * @param string $file_name Name of file field in form
 * @param int $profile_id the Profile ID the file(s) were uploaded under
 * @return false|int
 */
function _jrCore_local_is_uploaded_media_file($module, $file_name, $profile_id)
{
    global $_post;
    if (!empty($_post['upload_token'])) {
        $dir = jrCore_get_upload_temp_directory($_post['upload_token']);
        if (is_dir($dir)) {
            jrCore_start_timer('filesystem');
            $_tmp = glob("{$dir}/*_{$file_name}.tmp", GLOB_NOSORT);
            jrCore_stop_timer('filesystem');
            if ($_tmp && is_array($_tmp) && count($_tmp) > 0) {
                return count($_tmp);
            }
        }
    }
    return false;
}

/**
 * Local filesystem copy uploaded file
 * @param array $_uploads
 * @param string $field
 * @param string $target
 * @return bool
 */
function _jrCore_local_copy_uploaded_file($_uploads, $field, $target)
{
    @ini_set('max_execution_time', 60);  // shouldn't need longer than 1 minute to copy over data
    if (!isset($_uploads[$field])) {
        return false;
    }
    if (!isset($_uploads[$field]['tmp_name'])) {
        return false;
    }
    if (!is_file($_uploads[$field]['tmp_name'])) {
        return false;
    }
    if (strpos($target, APP_DIR) !== 0 || strpos($target, '/../')) {
        return false;
    }
    $source = $_uploads[$field]['tmp_name'];
    jrCore_start_timer('filesystem');
    if (copy($source, $target)) {
        jrCore_stop_timer('filesystem');
        return true;
    }
    jrCore_stop_timer('filesystem');
    return false;
}

/**
 * Local filesystem delete uploaded file
 * @param array $_uploads
 * @param string $field
 * @return bool
 */
function _jrCore_local_delete_uploaded_file($_uploads, $field)
{
    if (!is_array($_uploads)) {
        return false;
    }
    if (!isset($_uploads[$field])) {
        return false;
    }
    if (!isset($_uploads[$field]['tmp_name'])) {
        return false;
    }
    if (jrCore_unlink($_uploads[$field]['tmp_name'])) {
        jrCore_unlink("{$_uploads[$field]['tmp_name']}.tmp");
        return true;
    }
    return false;
}

/**
 * Clean up "old" (replaced) media items in a profile directory
 * @param $profile_id int Profile ID
 * @param $module string Module Name
 * @param $unique_id int Item_ID
 * @param $field string Unique item field name
 * @param null $_exclude option array of file names to exclude
 * @return bool
 */
function _jrCore_local_media_delete_old_items($profile_id, $module, $unique_id, $field, $_exclude = null)
{
    $pdir = jrCore_get_media_directory($profile_id, FORCE_LOCAL);
    jrCore_start_timer('filesystem');
    $_old = glob("{$pdir}/{$module}_{$unique_id}_{$field}.*", GLOB_NOSORT);
    jrCore_stop_timer('filesystem');
    if ($_old && is_array($_old)) {
        foreach ($_old as $old_file) {
            if (is_array($_exclude)) {
                $old_name = basename($old_file);
                if (!in_array($old_name, $_exclude)) {
                    jrCore_unlink($old_file); // OK
                }
            }
            else {
                jrCore_unlink($old_file); // OK
            }
        }
    }
    return true;
}

/**
 * Delete matching media file(s) for a given item ID
 * @param string $module Module Name to save file for
 * @param string $file_name Name of file field in form
 * @param int $profile_id the Profile ID to save the media file for
 * @param int $unique_id Unique Item ID from DataStore
 * @return bool
 */
function _jrCore_local_media_delete_file($module, $file_name, $profile_id, $unique_id)
{
    $dir = jrCore_get_media_directory($profile_id, FORCE_LOCAL);
    $nam = "{$module}_{$unique_id}_{$file_name}.";
    jrCore_start_timer('filesystem');
    if ($h = opendir(realpath($dir))) {
        while (false !== ($file = readdir($h))) {
            if (strpos($file, $nam) === 0) {
                unlink("{$dir}/{$file}"); // OK
            }
        }
        closedir($h);
    }
    jrCore_stop_timer('filesystem');
    return true;
}

/**
 * Local FileSystem Get Media Directory Size function
 * @param int $profile_id Profile ID
 * @return int
 */
function _jrCore_local_media_get_directory_size($profile_id)
{
    $media_dir = _jrCore_local_media_get_directory($profile_id);
    $size      = 0;
    jrCore_start_timer('filesystem');
    clearstatcache();
    if ($h = opendir(realpath($media_dir))) {
        while (false !== ($file = readdir($h))) {
            if ($file == '.' || $file == '..' || $file == 'cache') {
                continue;
            }
            else {
                $size += filesize("{$media_dir}/{$file}");
            }
        }
        closedir($h);
    }
    jrCore_stop_timer('filesystem');
    return $size;
}

/**
 * Local FileSystem Get Media Files for a Profile ID
 * @param int $profile_id Profile ID
 * @param string $pattern optional glob pattern
 * @return array|false
 */
function _jrCore_local_media_get_files($profile_id, $pattern = null)
{
    $media_dir = _jrCore_local_media_get_directory($profile_id);
    if (!is_null($pattern)) {
        jrCore_start_timer('filesystem');
        $_fl = glob("{$media_dir}/{$pattern}", GLOB_NOSORT);
        jrCore_stop_timer('filesystem');
    }
    else {
        $_fl = array();
        jrCore_start_timer('filesystem');
        if ($h = opendir(realpath($media_dir))) {
            while (false !== ($file = readdir($h))) {
                if ($file == '.' || $file == '..') {
                    continue;
                }
                $_fl[] = "{$media_dir}/{$file}";
            }
            closedir($h);
        }
        jrCore_stop_timer('filesystem');
    }
    $_ot = array();
    if (count($_fl) > 0) {
        foreach ($_fl as $file) {
            if (is_file($file)) {
                $_ot[] = array(
                    'name' => $file,
                    'size' => filesize($file),
                    'time' => filemtime($file)
                );
            }
        }
    }
    return (count($_ot) > 0) ? $_ot : false;
}

/**
 * Local FileSystem Read Function
 * Can potentially use a TON of RAM - only use if needed
 * @param int $profile_id Profile ID
 * @param string $file File Name
 * @param string $save_as If given, will be saved to the given path
 * @return bool|string
 */
function _jrCore_local_media_read($profile_id, $file, $save_as = null)
{
    $media_dir = _jrCore_local_media_get_directory($profile_id);
    if (is_file("{$media_dir}/{$file}")) {
        if (!is_null($save_as) && strpos($save_as, APP_DIR) === 0) {
            return jrCore_chunked_copy("{$media_dir}/{$file}", $save_as);
        }
        else {
            return jrCore_file_get_contents("{$media_dir}/{$file}");
        }
    }
    return false;
}

/**
 * Local FileSystem Write Function
 * This function will set permissions on the created/updated file to 0644
 * @param int $profile_id Profile ID
 * @param string $file File Name to write to
 * @param string $data Data to write to file
 * @param string $access ACL access level (not used on local file system)
 * @return bool
 */
function _jrCore_local_media_write($profile_id, $file, $data, $access)
{
    $media_dir = _jrCore_local_media_get_directory($profile_id);
    if (@is_file($data)) {
        return jrCore_chunked_copy($data, "{$media_dir}/{$file}");
    }
    else {
        if (jrCore_write_to_file("{$media_dir}/{$file}", $data)) {
            return true;
        }
    }
    return false;
}

/**
 * Local FileSystem Delete Function
 * @param int $profile_id Profile ID
 * @param string $file file
 * @return bool
 * @ignore used internally
 */
function _jrCore_local_media_delete($profile_id, $file)
{
    global $_conf;
    $media_dir = _jrCore_local_media_get_directory($profile_id);
    if (is_file("{$media_dir}/{$file}")) {
        $tmp = jrCore_unlink("{$media_dir}/{$file}"); // OK
        if (!$tmp) {
            // try to change permissions and try again
            chmod("{$media_dir}/{$file}", $_conf['jrCore_file_perms']);
            $tmp = jrCore_unlink("{$media_dir}/{$file}"); // OK
        }
        return $tmp;
    }
    elseif (is_file($file) && strpos($file, $media_dir) === 0) {
        // We've been given a full path file - handle it
        $tmp = jrCore_unlink($file); // OK
        if (!$tmp) {
            // try to change permissions and try again
            chmod($file, $_conf['jrCore_file_perms']);
            $tmp = jrCore_unlink($file); // OK
        }
        return $tmp;
    }
    return false;
}

/**
 * Local FileSystem Exist function
 * @param int $profile_id Profile ID
 * @param string $file file
 * @return bool
 * @ignore used internally
 */
function _jrCore_local_media_exists($profile_id, $file)
{
    $media_dir = _jrCore_local_media_get_directory($profile_id);
    if (is_file("{$media_dir}/{$file}")) {
        return true;
    }
    // See if we were given full path
    elseif (is_file($file) && strpos($file, $media_dir) === 0) {
        return true;
    }
    return false;
}

/**
 * Local FileSystem Confirm function
 * @param int $profile_id Profile ID
 * @param string $file file
 * @param string $local_save local file name to save to
 * @param bool $force
 * @param int $size file size of master file
 * @return bool|string
 * @ignore used internally
 */
function _jrCore_local_media_confirm_is_local($profile_id, $file, $local_save = null, $force = false, $size = 0)
{
    $media_dir = _jrCore_local_media_get_directory($profile_id);
    if (is_file("{$media_dir}/{$file}")) {
        if (!is_null($local_save) && $local_save != "{$media_dir}/{$file}") {
            if (jrCore_chunked_copy("{$media_dir}/{$file}", $local_save)) {
                return $local_save;
            }
        }
        return "{$media_dir}/{$file}";
    }
    // See if we were given full path
    elseif (is_file($file) && strpos($file, $media_dir) === 0) {
        return $file;
    }
    return false;
}

/**
 * Local FileSystem Stream function - Sends HEADERS!
 * @param int $profile_id Profile ID
 * @param string $file File to Stream
 * @param string $send_name Send-As Filename
 * @return bool
 */
function _jrCore_local_media_stream($profile_id, $file, $send_name)
{
    $media_dir = _jrCore_local_media_get_directory($profile_id);
    if (!is_file("{$media_dir}/{$file}")) {
        return false;
    }
    $size = filesize("{$media_dir}/{$file}");
    $type = jrCore_mime_type("{$media_dir}/{$file}");

    if (isset($_SERVER['HTTP_RANGE']) && $_SERVER['HTTP_RANGE'] != 'bytes=0-') {
        _jrCore_local_media_stream_with_range("{$media_dir}/{$file}", $type);
        return true;
    }

    // Check for 304 not modified
    $tim = filectime("{$media_dir}/{$file}");
    if ($tim && $tim > 0) {
        $ifs = (function_exists('getenv')) ? getenv('HTTP_IF_MODIFIED_SINCE') : false;
        if (!$ifs && !empty($_SERVER['HTTP_IF_MODIFIED_SINCE'])) {
            $ifs = $_SERVER['HTTP_IF_MODIFIED_SINCE'];
        }
        if ($ifs && strtotime($ifs) == $tim) {
            $_tmp = jrCore_get_flag('jrcore_set_custom_header');
            if ($_tmp && is_array($_tmp)) {
                foreach ($_tmp as $header) {
                    header($header);
                }
            }
            header("Last-Modified: " . gmdate('D, d M Y H:i:s \G\M\T', $tim));
            header('Content-Disposition: inline; filename="' . $send_name . '"');
            header('Expires: ' . gmdate('D, d M Y H:i:s \G\M\T', time() + 86400000));
            header('HTTP/1.1 304 Not Modified');
            return true;
        }
    }

    header('Accept-Ranges: bytes');
    header('Content-Length: ' . $size);
    header('Content-Type: ' . $type);
    header('Content-Disposition: inline; filename="' . $send_name . '"');

    jrCore_start_timer('filesystem');
    $handle = fopen("{$media_dir}/{$file}", 'rb');
    if (!$handle) {
        jrCore_logger('CRI', "core: unable to create file handle for streaming: {$media_dir}/{$file}");
        jrCore_stop_timer('filesystem');
        return false;
    }
    $bytes_sent = 0;
    while (true) {
        fseek($handle, $bytes_sent);
        // Read 1 megabyte at a time...
        $buffer     = fread($handle, 1048576);
        $bytes_sent += strlen($buffer);
        echo $buffer;
        flush();
        unset($buffer);
        // Also - check that we have not sent out more data then the allowed size
        if ($bytes_sent >= $size) {
            break;
        }
    }
    fclose($handle);
    jrCore_stop_timer('filesystem');
    return true;
}

/**
 * Stream a file to the iPhone with RANGE support
 * @param string $file File to Stream
 * @param string $type Mime-Type being streamed
 */
function _jrCore_local_media_stream_with_range($file, $type)
{
    $fp     = false;
    $size   = filesize($file); // File size
    $length = $size; // Content length
    $start  = 0; // Start byte
    $end    = $size - 1; // End byte

    // Send the accept range header
    header("Accept-Ranges: 0-{$length}");
    if (!empty($_SERVER['HTTP_RANGE'])) {
        $c_end = $end;
        // Extract the range string
        list(, $range) = explode('=', $_SERVER['HTTP_RANGE'], 2);
        if (strpos($range, ',') !== false) {
            jrCore_db_close();
            header('HTTP/1.1 416 Requested Range Not Satisfiable');
            header("Content-Range: bytes {$start}-{$end}/{$size}");
            exit; // OK
        }
        // If the range starts with an '-' we start from the beginning
        // If not, we forward the file pointer
        // And make sure to get the end byte if specified
        if ($range[0] == '-') {
            // The n-number of the last bytes is requested
            $c_start = $size - substr($range, 1);
        }
        else {
            $range   = explode('-', $range);
            $c_start = $range[0];
            $c_end   = (isset($range[1]) && is_numeric($range[1])) ? $range[1] : $size;
        }
        // End bytes can not be larger than $end.
        $c_end = ($c_end > $end) ? $end : $c_end;
        // Validate the requested range and return an error if it's not correct.
        if ($c_start > $c_end || $c_start > $size - 1 || $c_end >= $size) {
            jrCore_db_close();
            header('HTTP/1.1 416 Requested Range Not Satisfiable');
            header("Content-Range: bytes {$start}-{$end}/{$size}");
            exit; // OK
        }
        $start  = $c_start;
        $end    = $c_end;
        $length = $end - $start + 1; // Calculate new content length
        jrCore_start_timer('filesystem');
        $fp = @fopen($file, 'rb');
        fseek($fp, $start);
        header('HTTP/1.1 206 Partial Content');
    }
    if (!$fp) {
        jrCore_start_timer('filesystem');
        $fp = @fopen($file, 'rb');
    }
    // Notify the client the byte range we'll be outputting
    header("Content-Range: bytes {$start}-{$end}/{$size}");
    header("Content-Length: {$length}");
    header('Content-Type: ' . $type);

    // Start buffered download
    set_time_limit(0); // Reset time limit for big files
    $buffer = 1048576;
    while (!feof($fp) && ($p = ftell($fp)) <= $end) {
        if ($p + $buffer > $end) {
            $buffer = $end - $p + 1;
        }
        echo fread($fp, $buffer);
        flush();
    }
    fclose($fp);
    jrCore_stop_timer('filesystem');
}

/**
 * Local FileSystem Download function
 * NOTE: Sends HEADERS!
 * @param int $profile_id Profile ID
 * @param string $file File to Download
 * @param string $send_name Send-As Filename
 * @return bool
 */
function _jrCore_local_media_download($profile_id, $file, $send_name)
{
    $media_dir = _jrCore_local_media_get_directory($profile_id);
    if (!is_file("{$media_dir}/{$file}") && !is_link("{$media_dir}/{$file}")) {
        return false;
    }
    // Send headers to initiate download prompt
    $size = filesize("{$media_dir}/{$file}");
    header('Content-Length: ' . $size);
    header('Connection: close');
    header('Content-Type: application/octet-stream');
    header('Content-Transfer-Encoding: binary');
    header('Content-Disposition: attachment; filename="' . $send_name . '"');

    jrCore_start_timer('filesystem');
    $handle = fopen("{$media_dir}/{$file}", 'rb');
    if (!$handle) {
        jrCore_stop_timer('filesystem');
        jrCore_logger('CRI', "core: unable to create file handle for download: {$media_dir}/{$file}");
        return false;
    }
    $bytes_sent = 0;
    while ($bytes_sent < $size) {
        fseek($handle, $bytes_sent);
        // Read 1 megabyte at a time...
        $buffer     = fread($handle, 1048576);
        $bytes_sent += strlen($buffer);
        echo $buffer;
        ob_flush();
        flush();
        unset($buffer);
        // Also - check that we have not sent out more data then the allowed size
        if ($bytes_sent >= $size) {
            fclose($handle);
            jrCore_stop_timer('filesystem');
            return true;
        }
    }
    fclose($handle);
    jrCore_stop_timer('filesystem');
    return true;
}

/**
 * Local FileSystem Copy function
 * @param int $profile_id Profile ID
 * @param string $source_file file to copy - full path to file OR file name in profile media directory
 * @param string $target_file file to copy to - file name to copy TO in profile media directory
 * @return bool
 * @ignore used internally
 */
function _jrCore_local_media_copy($profile_id, $source_file, $target_file)
{
    $media_dir = _jrCore_local_media_get_directory($profile_id);
    if (!is_file($source_file)) {
        $source_file = "{$media_dir}/{$source_file}";
    }
    if (is_file($source_file)) {
        $max = @ini_get('max_execution_time');
        if (!$max || $max < 600) {
            @ini_set('max_execution_time', 600); // 10 minutes max
        }
        if (strpos($target_file, APP_DIR) === 0) {
            $target_file = basename($target_file);
        }
        return jrCore_chunked_copy($source_file, "{$media_dir}/{$target_file}");
    }
    return false;
}

/**
 * Local FileSystem Rename function
 * @param int $profile_id Profile ID
 * @param string $file file name
 * @param string $new_name new name for file
 * @return bool
 * @ignore used internally
 */
function _jrCore_local_media_rename($profile_id, $file, $new_name)
{
    $media_dir = _jrCore_local_media_get_directory($profile_id);
    jrCore_start_timer('filesystem');
    if (!is_file($file) && is_file("{$media_dir}/{$file}")) {
        $file = "{$media_dir}/{$file}";
    }
    if (is_file($file)) {
        if (rename($file, "{$media_dir}/{$new_name}")) {
            jrCore_stop_timer('filesystem');
            return true;
        }
        if (jrCore_chunked_copy($file, "{$media_dir}/{$new_name}")) {
            unlink($file); // OK
            jrCore_stop_timer('filesystem');
            return true;
        }
    }
    jrCore_stop_timer('filesystem');
    return false;
}

/**
 * Local FileSystem Stat function
 * @param int $profile_id Profile ID
 * @param string $file file
 * @return array|false
 * @ignore used internally
 */
function _jrCore_local_media_stat($profile_id, $file)
{
    $media_dir = _jrCore_local_media_get_directory($profile_id);
    jrCore_start_timer('filesystem');
    if (is_file("{$media_dir}/{$file}")) {
        $stat = stat("{$media_dir}/{$file}");
        jrCore_stop_timer('filesystem');
        return $stat;
    }
    return false;
}

/**
 * Unlink a file with Timer
 * @param string $file
 * @return bool
 */
function jrCore_unlink($file)
{
    $tmp = false;
    jrCore_start_timer('filesystem');
    if (is_file($file)) {
        $tmp = unlink($file);
    }
    jrCore_stop_timer('filesystem');
    return $tmp;
}
