<?php
 /**
 * Jamroom System Core module
 *
 * copyright 2025 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @package Smarty Functions and Modifiers
 * @copyright 2012 Talldude Networks, LLC.
 * @author Brian Johnson <brian [at] jamroom [dot] net>
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * Get a smarty error string for a missing parameter
 * @param $param string Param name
 * @return string
 */
function jrCore_smarty_missing_error($param)
{
    $_tmp = debug_backtrace();
    if (isset($_tmp[1]['function'])) {
        $file = jrCore_smarty_get_debug_file($_tmp[1]);
        $func = '{' . str_replace('smarty_function_', '', $_tmp[1]['function']) . '} ';
        return $func . 'required parameter "' . jrCore_entity_string($param) . '" is missing (' . $file . ')';
    }
    return 'required parameter "' . jrCore_entity_string($param) . '" is missing';
}

/**
 * Get a smarty error string for a missing parameter
 * @param $param string Param name
 * @return string
 */
function jrCore_smarty_invalid_error($param)
{
    $_tmp = debug_backtrace();
    if (isset($_tmp[1]['function'])) {
        $file = jrCore_smarty_get_debug_file($_tmp[1]);
        $func = '{' . str_replace('smarty_function_', '', $_tmp[1]['function']) . '} ';
        return $func . 'invalid value received for "' . jrCore_entity_string($param) . '" parameter (' . $file . ')';
    }
    return 'invalid value received for "' . jrCore_entity_string($param) . '" parameter';
}

/**
 * Send a custom error to the template
 * @param $text string Error string
 * @return string
 */
function jrCore_smarty_custom_error($text)
{
    $_tmp = debug_backtrace();
    if (isset($_tmp[1]['function'])) {
        $file = jrCore_smarty_get_debug_file($_tmp[1]);
        $func = '{' . str_replace('smarty_function_', '', $_tmp[1]['function']) . '} ';
        return $func . jrCore_entity_string($text) . ' (' . $file . ')';
    }
    return jrCore_entity_string($text);
}

/**
 * Get file name from smarty debug dump
 * @param $_tmp array info from debug_backtrace();
 * @return string
 */
function jrCore_smarty_get_debug_file($_tmp)
{
    global $_mods;
    if (isset($_tmp['file'])) {
        // data/cache/jrElastic/da181648655af4d42437256697fbe2cd^17aad2f57c512a33c99156f88e9e73483d1aaea2_1.file.76ca6e4b048861b9d2417ac4816c7de4^jrAudio^item_list.tpl.php
        $_tm = basename($_tmp['file']);
        $_tm = explode('^', $_tm);
        if ($_tm && is_array($_tm) && count($_tm) === 4) {
            $mod = $_tm[2];
            $fil = str_replace('.php', '', $_tm[3]);
            if (isset($_mods[$mod])) {
                return "modules/{$mod}/templates/{$fil}";
            }
            else {
                return "skins/{$mod}/{$fil}";
            }
        }
    }
    return '';
}

/**
 * Get item action buttons for the given type
 * @param $type string one of "index", "list" or "detail"
 * @param $params array Smarty function parameters
 * @param $smarty object Smarty Object
 * @return string
 */
function jrCore_item_action_buttons($type, $params, $smarty)
{
    global $_user, $_conf;
    if (isset($params['profile_check']) && $params['profile_check'] === true && !jrProfile_is_profile_view()) {
        return '';
    }
    if (!isset($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (!jrCore_module_is_active($params['module'])) {
        return '';
    }
    switch ($type) {
        case 'detail':
        case 'list':
            if (!isset($params['item']) || !is_array($params['item'])) {
                return '';
            }
            $pid = (int) $params['item']['_profile_id'];

            // Let other modules check out the item
            $params['item'] = jrCore_trigger_event('jrCore', 'item_action_button_data', $params['item'], $params);

            break;
        default:
            if (!isset($params['profile_id']) || !jrCore_checktype($params['profile_id'], 'number_nz')) {
                return '';
            }
            if (!isset($params['item'])) {
                $params['item'] = array();
            }
            $pid = (int) $params['profile_id'];
            break;
    }

    // Get module's with registered features
    $_rs = array();
    $_rf = jrCore_get_registered_module_features('jrCore', "item_{$type}_button"); // 'item_list_button' 'item_detail_button' to help locate via IDE search.
    if ($_rf && is_array($_rf)) {
        foreach ($_rf as $bmod => $_ft) {
            foreach ($_ft as $func => $_inf) {
                $_inf['module']   = $bmod;
                $_inf['function'] = $func;
                $_rs[]            = $_inf;
            }
        }
    }
    else {
        return '';
    }

    // Let modules exclude buttons by trigger
    $_ex = array();
    $_ex = jrCore_trigger_event('jrCore', "exclude_item_{$type}_buttons", $_ex, $params, $params['module']);

    if (isset($params['exclude']) && strlen($params['exclude']) > 0) {
        $_px = explode(',', $params['exclude']);
        if ($_px && is_array($_px)) {
            foreach ($_px as $exclude) {
                $exclude       = trim($exclude);
                $_ex[$exclude] = true;
            }
        }
    }

    // The admin can:
    // set a specific button to not show
    // set the ORDER the buttons appear in (left to right)
    // Our config holds the info, ordered and by function => on|off
    $_bt = array();
    $_cb = array();
    $opt = "{$params['module']}_item_{$type}_buttons";
    if (!empty($_conf[$opt])) {
        $_ord = json_decode($_conf[$opt], true);
        foreach ($_ord as $_ab) {
            $_cb["{$_ab['function']}"] = 1;
        }
        // "new" modules may not be present in the order until the admin actually
        // re-orders things, so let's add any extra in at the end.
        foreach ($_rs as $_inf) {
            if (!isset($_cb["{$_inf['function']}"])) {
                $_ord[] = $_inf;
            }
        }
        unset($_cb);
    }
    else {
        $_ord = $_rs;
    }

    // We now have buttons in the order we need
    $_ord = jrCore_trigger_event('jrCore', "item_{$type}_buttons", $_ord, $params);

    $siz = null;
    if (isset($params['size']) && jrCore_checktype($params['size'], 'number_nz')) {
        $siz = (int) $params['size'];
    }
    $cls = null;
    if (isset($params['class']) && strlen($params['class']) > 0) {
        $cls = $params['class'];
    }
    $clr = null;
    if (isset($params['color']) && strlen($params['color']) === 6 && jrCore_checktype($params['color'], 'hex')) {
        $clr = $params['color'];
    }

    $_ln = jrUser_load_lang_strings();
    if ($_ord && count($_ord) > 0) {

        foreach ($_ord as $_inf) {

            $func = $_inf['function'];
            if (!function_exists($func)) {
                continue;
            }
            if (isset($_inf['active']) && $_inf['active'] == 'off') {
                // Purposefully disabled
                continue;
            }

            // Check for exclusions
            if (isset($_ex[$func]) && $_ex[$func] === true) {
                // Purposefully excluded from smarty function call
                continue;
            }

            // Each button has some info about it:
            // group  => visitor|user|multi|power|admin|master
            // quota  => #|#|#|# ...
            if (isset($_inf['group'])) {
                switch ($_inf['group']) {
                    // Special "Owner" case
                    case 'owner':
                        if (!jrUser_is_admin()) {
                            switch ($type) {
                                case 'detail':
                                case 'list':
                                    if (!jrUser_can_edit_item($params['item'])) {
                                        continue 3;
                                    }
                                    break;
                                default:
                                    if (!jrUser_is_linked_to_profile($pid)) {
                                        continue 3;
                                    }
                                    break;
                            }
                        }
                        break;

                    default:
                        // User must be part of defined group
                        if (!jrCore_user_is_part_of_group($_inf['group'])) {
                            continue 2;
                        }
                        break;
                }
                $params['group'] = $_inf['group'];
            }
            if (isset($_inf['quota']) && strlen($_inf['quota']) > 0 && $_inf['quota'] != '_' && !jrUser_is_admin()) {
                if (!jrUser_is_logged_in()) {
                    continue;
                }
                if (is_numeric($_inf['quota']) && $_user['profile_quota_id'] != $_inf['quota']) {
                    continue;
                }
                elseif (strpos($_inf['quota'], ',')) {
                    $pass = false;
                    foreach (explode(',', $_inf['quota']) as $qt) {
                        if ($_user['profile_quota_id'] == $qt) {
                            $pass = true;
                            break;
                        }
                    }
                    if (!$pass) {
                        continue;
                    }
                }
            }

            // Show our button
            if ($_tmp = $func($params['module'], $params['item'], $params, $smarty)) {
                if (is_array($_tmp)) {
                    if (!empty($_tmp['icon']) && !empty($_tmp['url'])) {
                        $alt = '';
                        if (isset($_tmp['alt']) && strlen($_tmp['alt']) > 0) {
                            if (is_numeric($_tmp['alt'])) {
                                if (isset($_ln["{$params['module']}"]["{$_tmp['alt']}"])) {
                                    $alt = $_ln["{$params['module']}"]["{$_tmp['alt']}"];
                                }
                            }
                            if (strlen($alt) === 0) {
                                $alt = $_tmp['alt'];
                            }
                        }
                        $_tmp['icon']   = jrCore_get_icon_html($_tmp['icon'], $siz, $cls, $clr);
                        $_tmp['title']  = $alt;
                        $_tmp['params'] = $params;
                        $_bt[]          = jrCore_parse_template('item_action_button.tpl', $_tmp, 'jrCore');
                    }
                }
                else {
                    $_bt[] = trim($_tmp); // raw output
                }
            }
        }
    }

    // If this is a PENDING item, show pending notice
    $pfx = jrCore_db_get_prefix($params['module']);
    if ($pfx && jrProfile_is_profile_owner($pid) && isset($params['item']["{$pfx}_pending"]) && $params['item']["{$pfx}_pending"] >= 1) {
        // If this user is an ADMIN user give them the ability to approve
        // directly from the item detail buttons
        if (jrUser_is_admin()) {
            $murl = jrCore_get_module_url('jrCore');
            $_tmp = array(
                'icon'   => jrCore_get_icon_html('ok-hilighted', $siz, $cls, $clr),
                'title'  => 'approve this item',
                'url'    => "{$_conf['jrCore_base_url']}/{$murl}/pending_item_approve/{$params['module']}/id={$params['item']['_item_id']}",
                'params' => $params
            );
        }
        else {
            $_tmp = array(
                'icon'    => jrCore_get_icon_html('lock-hilighted', $siz, $cls, $clr),
                'title'   => jrCore_entity_string($_ln['jrCore'][71]),
                'onclick' => "jrCore_alert('" . addslashes($_ln['jrCore'][71]) . "')",
                'params'  => $params
            );
        }
        $_bt[] = jrCore_parse_template('item_action_button.tpl', $_tmp, 'jrCore');
    }
    // Lastly - if this is a MASTER ADMIN viewing, let them config
    if (jrUser_is_master() && jrProfile_is_profile_view() && (!jrCore_is_mobile_device() || jrCore_is_tablet_device())) {
        $_tmp  = array(
            'icon'   => jrCore_get_icon_html('settings', $siz, $cls, $clr),
            'title'  => 'Configure these buttons and the order they appear in',
            'url'    => jrCore_get_base_url() . '/' . jrCore_get_module_url('jrCore') . '/item_action_buttons/' . $type . '/m=' . $params['module'],
            'params' => $params
        );
        $_bt[] = jrCore_parse_template('item_action_button.tpl', $_tmp, 'jrCore');
    }
    if (!isset($params['separator'])) {
        $params['separator'] = ' ';
    }
    return implode($params['separator'], $_bt);
}

/**
 * Add module "action" buttons to an item index
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_item_index_buttons($params, $smarty)
{
    if (!isset($params['profile_id'])) {
        return jrCore_smarty_missing_error('profile_id');
    }
    if (!jrCore_checktype($params['profile_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('profile_id');
    }
    $out = jrCore_item_action_buttons('index', $params, $smarty);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Add module "action" buttons to an item listing
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_item_list_buttons($params, $smarty)
{
    if (!isset($params['item'])) {
        return jrCore_smarty_missing_error('item');
    }
    if (!is_array($params['item'])) {
        return jrCore_smarty_invalid_error('item');
    }
    $out = jrCore_item_action_buttons('list', $params, $smarty);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Add module "action" buttons to an item detail page
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_item_detail_buttons($params, $smarty)
{
    if (!isset($params['item'])) {
        return jrCore_smarty_missing_error('item');
    }
    if (!is_array($params['item'])) {
        return jrCore_smarty_invalid_error('item');
    }
    $out = jrCore_item_action_buttons('detail', $params, $smarty);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Add "bundle" buttons to a bundle index
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_bundle_index_buttons($params, $smarty)
{
    if (!isset($params['profile_id'])) {
        return jrCore_smarty_missing_error('profile_id');
    }
    if (!jrCore_checktype($params['profile_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('profile_id');
    }
    $out = jrCore_item_action_buttons('bundle_index', $params, $smarty);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Add "bundle" buttons to a bundle list
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_bundle_list_buttons($params, $smarty)
{
    if (!isset($params['profile_id'])) {
        return jrCore_smarty_missing_error('profile_id');
    }
    if (!jrCore_checktype($params['profile_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('profile_id');
    }
    $out = jrCore_item_action_buttons('bundle_list', $params, $smarty);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Add "bundle" buttons to a bundle detail template
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_bundle_detail_buttons($params, $smarty)
{
    if (!isset($params['profile_id'])) {
        return jrCore_smarty_missing_error('profile_id');
    }
    if (!jrCore_checktype($params['profile_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('profile_id');
    }
    $out = jrCore_item_action_buttons('bundle_detail', $params, $smarty);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Add module feature sections to item details
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_item_detail_features($params, $smarty)
{
    global $_conf;
    if (!isset($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (!isset($params['item'])) {
        return jrCore_smarty_missing_error('item');
    }
    if (!is_array($params['item'])) {
        return jrCore_smarty_invalid_error('item');
    }

    // See if the item is pending - we show no features for pending items
    $pfx = jrCore_db_get_prefix($params['module']);
    if (isset($params['item']["{$pfx}_pending"]) && $params['item']["{$pfx}_pending"] >= 1) {
        if (!empty($params['assign'])) {
            $smarty->assign($params['assign'], '');
        }
        return '';
    }

    // See if our order has been defined yet
    $_ord = array();
    if (isset($_conf['jrCore_detail_feature_order']) && strlen($_conf['jrCore_detail_feature_order']) > 0) {
        $_ord = array_flip(explode(',', $_conf['jrCore_detail_feature_order']));
    }

    // Prepare any excludes
    $_ex = array();
    if (isset($params['exclude']) && strlen($params['exclude']) > 0) {
        $_ex = array_flip(explode(',', $params['exclude']));
    }
    // Get all registered features
    $_tmp = jrCore_get_registered_module_features('jrCore', 'item_detail_feature');
    $text = '';
    if (is_array($_tmp)) {

        // First get things in the right order
        $_res = array();
        foreach ($_tmp as $mod => $_ft) {
            if (!jrCore_module_is_active($mod)) {
                continue;
            }
            foreach ($_ft as $nam => $_ftr) {
                $name           = "{$mod}~{$nam}";
                $_ftr['module'] = $mod;
                $_res[$name]    = $_ftr;
            }
        }

        // Order entries first
        if (count($_ord) > 0) {
            foreach ($_ord as $name => $order) {
                $_ftr = $_res[$name];
                // Make sure it is active
                if (!jrCore_module_is_active($_ftr['module'])) {
                    continue;
                }
                // Make sure it is allowed in the Quota
                if (!isset($params['item']["quota_{$_ftr['module']}_show_detail"]) || $params['item']["quota_{$_ftr['module']}_show_detail"] != 'on') {
                    continue;
                }
                // Purposefully excluded
                if (isset($_ex[$name])) {
                    continue;
                }
                if (isset($_ftr['function']) && function_exists($_ftr['function'])) {
                    $text .= $_ftr['function']($params['module'], $params['item'], $params, $smarty);
                }
                unset($_res[$name]);
            }
        }

        // Unordered (new) at bottom
        if (count($_res) > 0) {
            foreach ($_res as $name => $_ftr) {

                // Make sure it is active
                if (!jrCore_module_is_active($_ftr['module'])) {
                    continue;
                }
                // Make sure it is allowed in the Quota
                if (!isset($params['item']["quota_{$_ftr['module']}_show_detail"]) || $params['item']["quota_{$_ftr['module']}_show_detail"] != 'on') {
                    continue;
                }
                // Purposefully excluded
                if (isset($_ex[$name])) {
                    continue;
                }
                if (isset($_ftr['function']) && function_exists($_ftr['function'])) {
                    $text .= $_ftr['function']($params['module'], $params['item'], $params, $smarty);
                }
                unset($_res[$name]);
            }
        }
    }
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $text);
        return '';
    }
    return $text;
}

/**
 * Get all registered jrCore_format_string listeners
 * @return array|false
 */
function jrCore_get_format_string_listeners()
{
    if ($_tmp = jrCore_get_registered_module_features('jrCore', 'format_string')) {
        $_out = array();
        foreach ($_tmp as $_opts) {
            foreach ($_opts as $fnc => $_desc) {
                $_out[$fnc] = $_desc['label'];
            }
        }
        return $_out;
    }
    return false;
}

/**
 * Return HELP for Quota Config
 * @return string
 */
function jrCore_get_format_string_help()
{
    if ($_tmp = jrCore_get_registered_module_features('jrCore', 'format_string')) {
        $text = 'The following Text Formatters are are available:<br><br>';
        $_out = array();
        foreach ($_tmp as $_opts) {
            foreach ($_opts as $_desc) {
                $_out[] = "<b>{$_desc['label']}</b>: {$_desc['help']}<br><small>template whitelist name: {$_desc['wl']}</small>";
            }
        }
        $text .= implode('<br><br>', $_out);
    }
    else {
        $text = 'There are no formatters currently available in the system!';
    }
    return $text;
}

/**
 * Core Media Player
 * {jrCore_media_player type="jrAudio_blue_monday" module="jrAudio" field="audio_file" item_id=# autoplay=false}
 * {jrCore_media_player type="jrVideo_blue_monday" module="jrVideo" field="video_file" item_id=# autoplay=false}
 * {jrCore_media_player type="jrPlaylist_blue_monday" module="jrPlaylist" item_id=# autoplay=false}
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_media_player($params, $smarty)
{
    // Module is required
    if (!isset($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (strlen($params['module']) === 0) {
        return jrCore_smarty_invalid_error('module');
    }

    // Get registered players
    $_pl = jrCore_get_registered_media_players('all');

    // If we get a TYPE, then that is the type of player we use, otherwise
    // we default to the player registered by the active skin
    $type = false;
    $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    if (isset($params['type']) && strlen($params['type']) > 0) {
        $type = $params['type'];
    }
    else {
        // Get active skin default player...
        $_tmp = jrCore_get_registered_module_features('jrCore', 'media_player_skin');
        if (!empty($_tmp[$skin]["{$params['module']}"])) {
            $type           = $_tmp[$skin]["{$params['module']}"];
            $params['type'] = $type;
        }
        if (!$type) {
            if (!jrCore_get_flag('jrCore_media_player_no_default_error')) {
                jrCore_logger('MIN', 'core: ' . jrCore_smarty_custom_error("the {$skin} skin is missing a default player for the {$params['module']} module"));
                jrCore_set_flag('jrCore_media_player_no_default_error', 1);
            }
            // What is the default player for this skin?
            foreach ($_pl as $t => $m) {
                if ($m == $params['module']) {
                    $params['type'] = $t;
                    $type           = $t;
                    break;
                }
            }
        }
    }

    // Make sure our type is valid
    if (!$type || !isset($_pl[$type])) {
        $_tmp = jrCore_get_registered_module_features('jrCore', 'media_player_skin');
        if (!empty($_tmp[$skin]["{$params['module']}"])) {
            $type = $_tmp[$skin]["{$params['module']}"];
        }
        else {
            return jrCore_smarty_invalid_error('type');
        }
    }

    $mod = $_pl[$type];
    if (is_file(APP_DIR . "/skins/{$skin}/{$type}.tpl")) {
        $mod = false;
    }
    $curl = jrCore_get_module_url('jrCore');
    if (is_file(APP_DIR . "/skins/{$skin}/css/{$type}.css")) {
        if (jrCore_is_ajax_request()) {
            $css = APP_DIR . "/skins/{$skin}/css/{$type}.css";
        }
        else {
            $css = jrCore_get_base_url() . "/{$curl}/css/skin/{$type}.css";
        }
    }
    elseif (is_file(APP_DIR . "/modules/{$mod}/css/{$type}.css")) {
        if (jrCore_is_ajax_request()) {
            $css = APP_DIR . "/modules/{$mod}/css/{$type}.css";
        }
        else {
            $css = jrCore_get_base_url() . "/{$curl}/css/" . jrCore_get_module_url($mod) . "/{$type}.css";
        }
    }

    // Setup common parameters
    $_rep = array();

    // auto play
    $_rep['autoplay'] = 'false';
    if (isset($params['autoplay']) && ($params['autoplay'] === 1 || $params['autoplay'] === true || $params['autoplay'] == 'true' || $params['autoplay'] == 'on')) {
        $_rep['autoplay'] = 'true';
    }

    // Get our playlist info
    if (isset($params['item_id']) && jrCore_checktype($params['item_id'], 'number_nz')) {
        $_rt = array(
            jrCore_db_get_item($params['module'], $params['item_id'])
        );
        if (!isset($_rt[0]) || !is_array($_rt[0])) {
            $out = '';
            if (!empty($params['assign'])) {
                $smarty->assign($params['assign'], $out);
                return '';
            }
            return $out;
        }
    }
    // Single item as an array
    elseif (!empty($params['item']['_item_id'])) {
        $_rt = array($params['item']);
        unset($params['item']);
    }
    // Array of items
    elseif (isset($params['items']) && is_array($params['items']) && count($params['items']) > 0) {
        $_rt = $params['items'];
        unset($params['items']);
    }

    // Get items
    $_fmt = array();
    if (!isset($_rt) || !is_array($_rt)) {

        // Go get our media based on params
        $_args = array();
        foreach ($params as $k => $v) {
            // Search
            if (strpos($k, 'search') === 0) {
                if (!isset($_args['search'])) {
                    $_args['search'] = array();
                }
                $_args['search'][] = $v;
            }
            // Order by
            elseif (strpos($k, 'order_by') === 0) {
                if (!isset($_args['order_by'])) {
                    $_args['order_by'] = array();
                }
                list($fld, $dir) = explode(' ', $v);
                $fld                     = trim($fld);
                $_args['order_by'][$fld] = trim($dir);
            }
            // Group By
            elseif ($k == 'group_by') {
                $_args['group_by'] = trim($v);
            }
            // Limit
            elseif ($k == 'limit') {
                $_args['limit'] = (int) $v;
            }
        }
        if (count($_args) > 0) {
            $_args['exclude_jrProfile_quota_keys'] = true;
            $_rt                                   = jrCore_db_search_items($params['module'], $_args);
            if (isset($_rt['_items']) && is_array($_rt['_items'])) {
                $_rt = $_rt['_items'];
            }
        }

        // Make sure we got media items
        if (!isset($_rt) || !is_array($_rt)) {
            $out = '';
            if (!empty($params['assign'])) {
                $smarty->assign($params['assign'], $out);
                return '';
            }
            return $out;
        }
    }

    // Send out player playlist trigger
    $_rt = jrCore_trigger_event('jrCore', 'media_playlist', $_rt, $params);

    // Our allowed "base" formats
    // NOTE: FLV and OGG are added in automatically based on config setting
    $_fm = array(
        'mp3' => 1,
        'm4v' => 1
    );

    // Prepare our playlist setup
    $plg           = jrCore_get_active_media_system();
    $_rep['media'] = array();
    foreach ($_rt as $k => $_item) {

        // If this is a PLAYLIST player then we have $_item['module'] already
        // set - of NOT we need to setup $_item['module']
        if (empty($_item['module'])) {
            $_item['module'] = $params['module'];
        }

        // media_playlist listeners can setup their own
        // media_playlist_url and media_playlist_ext
        if (empty($_item['media_playlist_ext']) || empty($_item['media_playlist_url'])) {
            $ext = false;
            $fld = false;
            if (isset($params['field'])) {
                // We know the field, so the module is for the item
                $ext = $_item["{$params['field']}_extension"];
                $pfx = jrCore_db_get_prefix($_item['module']);
                $url = jrCore_get_module_url($_item['module']);
            }
            else {
                // We need to figure out our extension
                foreach ($_item as $ek => $v) {
                    if (strpos($ek, '_extension') && !strpos($ek, '_original') && isset($_fm[$v])) {
                        $fld = str_replace('_extension', '', $ek);
                        $ext = $v;
                        break;
                    }
                }
                if (!$ext) {
                    // unknown file type
                    continue;
                }
                // We have to figure out the module based on the item
                $pfx = jrCore_db_get_prefix($_item['module']);
                $url = jrCore_get_module_url($_item['module']);
            }
            $fld = ($fld) ?: $params['field'];
            $str = jrCore_get_base_url() . "/{$url}/stream/{$fld}/{$_item['_item_id']}/file.{$ext}";

            $fnc = "_{$plg}_media_get_image_url";
            if (function_exists($fnc)) {
                $_pm = array(
                    'module'  => $_item['module'],
                    'item_id' => $_item['_item_id'],
                    'type'    => "{$pfx}_image",
                    'size'    => 'xxxlarge'
                );
                $img = $fnc($_pm);
            }
            else {
                $ixt = (isset($_item["{$pfx}_image_extension"])) ? $_item["{$pfx}_image_extension"] : 'png';
                $img = jrCore_get_base_url() . "/{$url}/image/{$pfx}_image/{$_item['_item_id']}/xxxlarge/image.{$ixt}";
            }
            $_item["{$pfx}_artist"] = $_item['profile_name'];
        }
        else {
            $url = jrCore_get_module_url($_item['module']);
            $pfx = jrCore_db_get_prefix($_item['module']);
            $ext = $_item['media_playlist_ext'];
            $str = $_item['media_playlist_url'];
            $img = (isset($_item['media_playlist_img'])) ? $_item['media_playlist_img'] : '';
        }
        if (isset($_fm[$ext])) {
            $_rep['media'][$k] = array(
                'title'      => jrCore_entity_string(jrCore_get_item_title($_item['module'], $_item)),
                'artist'     => jrCore_entity_string($_item['profile_name']),
                'poster'     => $img,
                'module'     => $_item['module'],
                'module_url' => $url,
                'prefix'     => $pfx,
                'item_id'    => $_item['_item_id'],
                '_item'      => $_item,
                'formats'    => array(
                    $ext => $str
                )
            );
            if ($ext == 'm4v') {

                // This is a VIDEO - Add in Legacy FLV if enabled
                if (jrCore_get_config_value('jrVideo', 'enable_flash', 'off') == 'on') {
                    if (!isset($fld) || strlen($fld) === 0) {
                        // See if we can figure it out...
                        $pfx = jrCore_db_get_prefix($_item['module']);
                        $fld = "{$pfx}_file";
                    }
                    $_rep['media'][$k]['formats']['flv'] = jrCore_get_base_url() . "/{$url}/stream/{$fld}/{$_item['_item_id']}/file.flv";
                    $_fmt['flv']                         = 'flv';
                }

            }
            elseif ($ext == 'mp3') {

                // This is an AUDIO file - are we including OGG as well?
                $mdir = jrCore_get_media_directory($_item['_profile_id']);
                if (is_file("{$mdir}/jrAudio_{$_item['_item_id']}_audio_file.ogg")) {
                    if (!isset($fld) || strlen($fld) === 0) {
                        // See if we can figure it out...
                        $pfx = jrCore_db_get_prefix($_item['module']);
                        $fld = "{$pfx}_file";
                    }
                    $_rep['media'][$k]['formats']['oga'] = jrCore_get_base_url() . "/{$url}/stream/{$fld}/{$_item['_item_id']}/file.ogg";
                    $_fmt['oga']                         = 'oga';
                }
            }
            $_fmt[$ext] = $ext;
        }
    }

    // Additional items
    $_rep['uniqid']   = 'm' . jrCore_create_unique_string(6);
    $_rep['formats']  = implode(',', $_fmt);
    $_rep['params']   = $params;
    $_rep['solution'] = 'html,flash';

    // Let other modules manipulate our final params if needed 
    $_rep = jrCore_trigger_event('jrCore', 'media_player_params', $_rep, $params);

    // Parse and return
    $out = '';
    if (isset($css) && strlen($css) > 0) {
        if (jrCore_is_ajax_request()) {
            // If this is an AJAX request, we must INLINE the CSS or it won't work properly (even using jQuery $.get())
            $crl = jrCore_get_module_url('jrImage');
            $mrl = jrCore_get_module_url($mod);
            $_rp = array(
                '{$' . $mod . '_img_url}' => jrCore_get_base_url() . "/{$crl}/img/{$mrl}"
            );
            $out = '<style>' . str_replace(array_keys($_rp), $_rp, jrCore_file_get_contents($css)) . '</style>';
        }
        else {
            $out .= '<link rel="stylesheet" property="stylesheet" href="' . $css . '?v=1" media="screen">' . "\n";
        }
    }
    jrCore_set_media_location_cookie();
    $out .= jrCore_parse_template("{$type}.tpl", $_rep, $mod);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Display the USER MENU for a skin
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_skin_menu($params, $smarty)
{
    global $_conf, $_user;

    // If this user is NOT logged in we can cache for
    // our regular cache length - for logged in users
    // we are going to cache for 30 seconds
    $key = "core_skin_menu_0";
    $sec = 0;
    if (jrUser_is_logged_in()) {
        $key = "core_skin_menu_" . intval($_user['_user_id']);
        $sec = 30;
    }
    if ($tmp = jrCore_is_cached('jrCore', $key)) {
        if ($tmp == 'no_menu_items') {
            $tmp = '';
        }
        if (!empty($params['assign'])) {
            $smarty->assign($params['assign'], $tmp);
        }
        return $tmp;
    }

    // Get core menu items
    // @note: This can always be cached for the default cache_seconds
    if (!$_rt = jrCore_is_cached('jrCore', 'skin_menu_items')) {
        $tbl = jrCore_db_table_name('jrCore', 'menu');
        $req = "SELECT *, CONCAT_WS('/',menu_module,menu_unique) AS menu_key FROM {$tbl} ORDER BY menu_order ASC";
        $_rt = jrCore_db_query($req, 'menu_key', false, null, false);
        if (!$_rt || !is_array($_rt)) {
            $_rt = 'no_items';
        }
        jrCore_add_to_cache('jrCore', 'skin_menu_items', $_rt);
    }
    if ($_rt == 'no_items') {
        $_rt = array();
    }

    // See if we have anything that has registered
    if ($_tmp = jrCore_get_registered_module_features('jrCore', 'skin_menu_item')) {

        // We have registered skin menu entries.  We need to go through each
        // one and make sure it is setup in the menu table
        foreach ($_tmp as $module => $_options) {

            if (!jrCore_module_is_active($module)) {
                // This module is not active
                continue;
            }

            // Modules can register different functionality with the core that
            // controls the type of information show in the skin menu item:
            // "group"       - (required) user_group to show to
            // "label"       - (required) Text to show for menu item - if NUMERIC and the number corresponds to
            //                 an entry in the registering modules lang file, the lang file entry will be used
            // "url"         - (optional) The URL to load when clicked.  URLs that do not begin with http will be
            //                 relative to the MODULE URL - i.e. site.com/MODULE_URL/[entry]
            // "onclick"     - (optional) onclick JS function to run when entry is clicked
            // "function"    - (optional) PHP function to run - can return a NUMBER, HTML or an IMAGE
            // "order"       - (optional) ordering index for default placement in skin menu
            // "menu_action" - (optional) PHP Function to override any customizations in skin menu editor

            foreach ($_options as $unq => $_inf) {

                // We are not setup ...
                if (!isset($_rt["{$module}/{$unq}"])) {
                    // We always place new entries at the bottom - find out our lowest number in
                    // our configured skin menu entries
                    $ord = 100;
                    foreach ($_tmp as $_op) {
                        foreach ($_op as $_o) {
                            if (isset($_o['order']) && $_o['order'] >= $ord) {
                                $ord = ($_o['order'] + 1);
                            }
                        }
                    }
                    $mod = jrCore_db_escape($module);
                    $lbl = jrCore_db_escape($_inf['label']);
                    $grp = jrCore_db_escape($_inf['group']);
                    $act = (isset($_inf['menu_action'])) ? jrCore_db_escape($_inf['menu_action']) : jrCore_db_escape($_inf['url']);
                    $fnc = (!is_null($_inf['function']) && strlen($_inf['function']) > 0) ? jrCore_db_escape($_inf['function']) : '';
                    $onc = (!is_null($_inf['onclick']) && strlen($_inf['onclick']) > 0) ? jrCore_db_escape($_inf['onclick']) : '';

                    // Check for field
                    $fad = '';
                    $fin = '';
                    $fup = '';
                    if (isset($_inf['field']) && strlen($_inf['field']) > 0) {
                        if (!isset($_fd)) {
                            $_fd = jrCore_db_table_columns('jrCore', 'menu');
                        }
                        if (isset($_fd['menu_field'])) {
                            $fld = jrCore_db_escape($_inf['field']);
                            $fad = ", menu_field";
                            $fin = ",'{$fld}'";
                            $fup = ", menu_field = '{$fld}'";
                        }
                    }
                    $tbl = jrCore_db_table_name('jrCore', 'menu');
                    $req = "INSERT INTO {$tbl} (menu_module,menu_unique,menu_active,menu_label,menu_action,menu_groups,menu_order,menu_function,menu_onclick{$fad})
                            VALUES ('{$mod}','{$unq}','on','{$lbl}','{$act}','{$grp}','{$ord}','{$fnc}','{$onc}'{$fin})
                            ON DUPLICATE KEY UPDATE menu_action = '{$act}', menu_function = '{$fnc}', menu_onclick = '{$onc}'{$fup}";
                    $cnt = jrCore_db_query($req, 'COUNT', false, null, false);
                    if ($cnt === 0) {
                        jrCore_logger('CRI', "core: unable to create new menu entry for {$module}/{$act}", $_inf);
                    }
                    else {
                        $_rt["{$module}/{$unq}"] = array(
                            'menu_module'   => $module,
                            'menu_unique'   => $unq,
                            'menu_active'   => 'on',
                            'menu_label'    => $_inf['label'],
                            'menu_action'   => (!empty($_inf['menu_action'])) ? jrCore_db_escape($_inf['menu_action']) : jrCore_db_escape($_inf['url']),
                            'menu_groups'   => $_inf['group'],
                            'menu_order'    => (isset($_inf['order']) && jrCore_checktype($_inf['order'], 'number_nz')) ? (int) $_inf['order'] : 90,
                            'menu_function' => $fnc,
                            'menu_onclick'  => $onc
                        );
                        if (isset($_inf['field']) && strlen($_inf['field']) > 0) {
                            $_rt["{$module}/{$unq}"]['menu_field'] = jrCore_db_escape($_inf['field']);
                        }
                    }
                }

                // Check if a DB update is needed for this
                else {

                    $update = false;
                    if ($_rt["{$module}/{$unq}"]['menu_action'] != $_inf['url'] && (!isset($_inf['menu_action']) || $_rt["{$module}/{$unq}"]['menu_action'] != $_inf['menu_action'])) {
                        $update = true;
                    }
                    elseif (!empty($_inf['group']) && $_rt["{$module}/{$unq}"]['menu_groups'] != $_inf['group']) {
                        $update = true;
                    }
                    elseif (!isset($_inf['function']) && !empty($_rt["{$module}/{$unq}"]['menu_function'])) {
                        $update = true;
                    }
                    elseif (!empty($_inf['field']) && $_inf['field'] != $_rt["{$module}/{$unq}"]['menu_field']) {
                        $update = true;
                    }
                    elseif (!empty($_inf['onclick']) && $_inf['onclick'] != $_rt["{$module}/{$unq}"]['menu_onclick']) {
                        $update = true;
                    }
                    elseif (is_numeric($_inf['label']) && isset($_rt["{$module}/{$unq}"]['menu_label']) && $_rt["{$module}/{$unq}"]['menu_label'] != $_inf['label']) {
                        $update = true;
                    }
                    if ($update) {

                        $act = (!empty($_inf['menu_action'])) ? jrCore_db_escape($_inf['menu_action']) : jrCore_db_escape($_inf['url']);
                        $fnc = (!empty($_inf['function'])) ? jrCore_db_escape($_inf['function']) : '';
                        $onc = (!empty($_inf['onclick'])) ? jrCore_db_escape($_inf['onclick']) : '';
                        // Check for field
                        $fup = '';
                        if (isset($_inf['field']) && strlen($_inf['field']) > 0) {
                            if (!isset($_fd)) {
                                $_fd = jrCore_db_table_columns('jrCore', 'menu');
                            }
                            if (isset($_fd['menu_field'])) {
                                $fld = jrCore_db_escape($_inf['field']);
                                $fup = ", menu_field = '{$fld}'";
                            }
                        }

                        $tbl = jrCore_db_table_name('jrCore', 'menu');
                        $req = "UPDATE {$tbl} SET
                                  menu_label = " . intval($_inf['label']) . ",
                                  menu_action = '{$act}',
                                  menu_function = '{$fnc}',
                                  menu_onclick = '{$onc}'{$fup}
                                 WHERE menu_id = '" . intval($_rt["{$module}/{$unq}"]['menu_id']) . "'";
                        jrCore_db_query($req);

                        $_rt["{$module}/{$unq}"]['menu_label']    = $_inf['label'];
                        $_rt["{$module}/{$unq}"]['menu_action']   = $_inf['url'];
                        $_rt["{$module}/{$unq}"]['menu_groups']   = trim($_inf['group']);
                        $_rt["{$module}/{$unq}"]['menu_field']    = (!empty($_inf['field'])) ? $_inf['field'] : '';
                        $_rt["{$module}/{$unq}"]['menu_function'] = (!empty($_inf['function'])) ? $_inf['function'] : '';
                        $_rt["{$module}/{$unq}"]['menu_onclick']  = (!empty($_inf['onclick'])) ? $_inf['onclick'] : '';
                    }
                }
            }
        }
    }

    // No menu entries
    if (!$_rt || !is_array($_rt) || count($_rt) === 0) {
        jrCore_add_to_cache('jrCore', $key, 'no_menu_items');
        if (!empty($params['assign'])) {
            $smarty->assign($params['assign'], '');
        }
        return '';
    }

    // Let other modules add/remove on event
    $_rt = jrCore_trigger_event('jrCore', 'skin_menu_items', $_rt);

    // Bring in strings
    $_lang = jrUser_load_lang_strings();

    // Go through each and process via template
    $alert = 0;
    $_ct   = array();
    $_ci   = array();
    $tpl   = 'skin_menu.tpl';
    $dir   = 'jrCore';
    $skin  = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    if (isset($params['template']) && is_file(APP_DIR . "/skins/{$skin}/{$params['template']}")) {
        $tpl = $params['template'];
        $dir = null;
    }
    foreach ($_rt as $k => $_opt) {

        // Active Check
        if ($_opt['menu_module'] != 'CustomEntry' && !jrCore_module_is_active($_opt['menu_module'])) {
            unset($_rt[$k]);
            continue;
        }

        // Category Check
        if (isset($params['category']) && strlen($params['category']) > 0 && $_opt['menu_category'] != $params['category']) {
            unset($_rt[$k]);
            continue;
        }

        // Turned off in ACP
        if (!isset($_opt['menu_active']) || $_opt['menu_active'] != 'on') {
            unset($_rt[$k]);
            continue;
        }

        // User Group Check
        if (isset($_opt['menu_groups']) && strpos($_opt['menu_groups'], ',')) {
            $_grp = explode(',', $_opt['menu_groups']);
            if (is_array($_grp)) {
                foreach ($_grp as $gk => $group) {
                    $_grp[$gk] = trim($group);
                }
            }
        }
        else {
            $_grp = array(trim($_opt['menu_groups']));
        }
        if (!is_array($_grp)) {
            unset($_rt[$k]);
            continue;
        }

        // Quota Check
        $show      = false;
        $grp_check = true;
        $fld       = "quota_{$_opt['menu_module']}_allowed";
        $fvl       = jrUser_get_profile_home_key($fld);
        if (isset($_opt['menu_field']) && strlen($_opt['menu_field']) > 0) {
            if (strpos($_opt['menu_field'], ':')) {
                list($home, $fld) = explode(':', $_opt['menu_field']);
                if ($home == 'home_profile') {
                    $fld = trim($fld);
                    $fvl = jrUser_get_profile_home_key($fld);
                }
            }
            else {
                $fld = trim($_opt['menu_field']);
                $fvl = (isset($_user[$fld])) ? $_user[$fld] : 'off';
            }
        }
        if (isset($fvl) && $fvl == 'off') {
            // User is not allowed based on quota - check if this menu item is setup for admin/master
            // access - if it is, and the user is a master or admin, allow it
            if ((in_array('master', $_grp) && jrUser_is_master()) || (in_array('admin', $_grp) && jrUser_is_admin())) {
                // admin user - we're good
                $grp_check = false;
                $show      = true;
            }
            else {
                unset($_rt[$k]);
                continue;
            }
        }

        if ($grp_check) {
            foreach ($_grp as $group) {
                if (jrCore_user_is_part_of_group($group)) {
                    $show = true;
                    break;
                }
            }
        }
        if (!$show) {
            unset($_rt[$k]);
            continue;
        }

        // Do not allow the DB to override what has been setup in the module
        // in regards to skin menu FUNCTIONS
        if (isset($_tmp["{$_opt['menu_module']}"]["{$_opt['menu_unique']}"]['menu_action'])) {
            $_opt['menu_action'] = $_tmp["{$_opt['menu_module']}"]["{$_opt['menu_unique']}"]['menu_action'];
        }
        if (isset($_tmp["{$_opt['menu_module']}"]["{$_opt['menu_unique']}"]['menu_function'])) {
            $_opt['menu_function'] = $_tmp["{$_opt['menu_module']}"]["{$_opt['menu_unique']}"]['menu_action'];
        }

        // Build our categories...
        $cat = 'default';
        if (isset($_opt['menu_category']) && strlen($_opt['menu_category']) > 0) {
            if (is_numeric($_opt['menu_category']) && isset($_lang[$skin]["{$_opt['menu_category']}"])) {
                $_opt['menu_category'] = $_lang[$skin]["{$_opt['menu_category']}"];
            }
            $cat = $_opt['menu_category'];
        }
        if (!isset($_ct[$cat])) {
            $_ct[$cat] = 0;
            $_ci[$cat] = array();
        }
        $_ct[$cat]++;

        $lbl                   = $_opt['menu_label'];
        $_rt[$k]['menu_label'] = (isset($lbl) && isset($_lang["{$_opt['menu_module']}"][$lbl])) ? $_lang["{$_opt['menu_module']}"][$lbl] : $lbl;
        if (strpos($_opt['menu_action'], 'http') === 0) {
            // We have been given a FULL URL - use it
            $_rt[$k]['menu_url'] = $_opt['menu_action'];
        }
        elseif (function_exists($_opt['menu_action'])) {
            // We get our URL from a function
            $_rt[$k]['menu_url'] = $_opt['menu_action']();
        }
        else {
            if ($_opt['menu_module'] != 'CustomEntry' && !strpos($_opt['menu_action'], '/')) {
                $murl                = jrCore_get_module_url($_opt['menu_module']);
                $_rt[$k]['menu_url'] = jrCore_get_base_url() . "/{$murl}/{$_opt['menu_action']}";
            }
            else {
                if (strpos(trim(trim($_opt['menu_action']), '/'), '/') || $_opt['menu_module'] == 'CustomEntry') {
                    $_rt[$k]['menu_url'] = jrCore_get_base_url() . "/{$_opt['menu_action']}";
                }
                else {
                    $murl                = jrCore_get_module_url($_opt['menu_module']);
                    $_rt[$k]['menu_url'] = jrCore_get_base_url() . "/{$murl}/{$_opt['menu_action']}";
                }
            }
        }

        // See if this menu item has a FUNCTION that needs to be run
        if (isset($_opt['menu_function']) && function_exists($_opt['menu_function'])) {

            // Our menu function can return a NUMBER, an IMAGE or bool TRUE/FALSE
            $res = $_opt['menu_function']($_conf, $_user);
            if (!$res) {
                // Function returned FALSE - don't show menu item
                unset($_rt[$k]);
                continue;
            }
            elseif (is_numeric($res)) {
                // Number - show next to title - i.e. this is a "notification"
                $_rt[$k]['menu_function_result'] = $res;
                $alert                           += $res;
            }
            elseif (strlen($res) > 0 && is_file(APP_DIR . "/modules/{$_opt['menu_module']}/img/{$res}")) {
                // Image
                switch (jrCore_file_extension($res)) {
                    case 'gif':
                    case 'png':
                    case 'jpg':
                    case 'jpeg':
                        $_rt[$k]['menu_function_result'] = $res;
                        break;
                }
            }
            elseif (!is_bool($res) && strlen($res) > 1) {
                // Allow HTML
                $_rt[$k]['menu_function_result'] = $res;
            }
        }
        // By category too
        $_ci[$cat][$k] = $_rt[$k];
    }
    $params['menu_id'] = (isset($params['menu_id'])) ? $params['menu_id'] : 'skin_menu';
    if (isset($params['label'])) {
        if (isset($_lang[$skin]["{$params['label']}"])) {
            $params['label'] = $_lang[$skin]["{$params['label']}"];
        }
    }
    else {
        $params['label'] = '**no label**';
    }
    $_rp = array(
        '_items'             => $_rt,
        '_categories'        => $_ct,
        '_items_by_category' => $_ci,
        'params'             => $params,
        'alert'              => $alert
    );
    unset($_rt);
    $out = jrCore_parse_template($tpl, $_rp, $dir);
    jrCore_add_to_cache('jrCore', $key, $out, $sec);

    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Count a hit for a module item
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_counter($params, $smarty)
{
    if (empty($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (!jrCore_module_is_active($params['module'])) {
        // Not installed or wrong...
        return '';
    }
    if (empty($params['name'])) {
        return jrCore_smarty_missing_error('name');
    }
    if (!isset($params['item_id'])) {
        return jrCore_smarty_missing_error('_item_id');
    }
    if (!jrCore_checktype($params['item_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('_item_id');
    }
    $inc = 1;
    if (isset($params['increment']) && jrCore_checktype($params['increment'], 'number_nz')) {
        $inc = intval($params['increment']);
    }
    // Count it
    jrCore_counter($params['module'], $params['item_id'], $params['name'], $inc);
    return '';
}

/**
 * Get Count for a module item
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_get_count($params, $smarty)
{
    if (empty($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (!jrCore_module_is_active($params['module']) || !jrCore_db_get_prefix($params['module'])) {
        if (!empty($params['assign'])) {
            $smarty->assign($params['assign'], 0);
            return '';
        }
        return 0;
    }
    if (empty($params['name'])) {

        // Profile id?
        if (!empty($params['profile_id'])) {
            $cnt = 0;
            if ($_sc = jrCore_db_get_all_values_for_key_by_profile_id($params['profile_id'], $params['module'], '_created')) {
                $cnt = count($_sc);
            }
        }
        else {
            // No specific field - get counts for entire module
            $cnt = jrCore_db_get_datastore_item_count($params['module']);
        }
    }
    else {
        // If we get a profile_id it must be valid
        if (isset($params['profile_id']) && !jrCore_checktype($params['profile_id'], 'number_nz')) {
            return jrCore_smarty_invalid_error('profile_id');
        }
        // Counts for a specific counter field
        if (!isset($params['item_id'])) {
            // We're doing ALL counts for a specific type
            $cnt = jrCore_get_count($params['module'], $params['name'], null, $params['profile_id']);
        }
        else {
            if (!jrCore_checktype($params['item_id'], 'number_nz')) {
                return jrCore_smarty_missing_error('item_id');
            }
            $cnt = jrCore_get_count($params['module'], $params['name'], $params['item_id'], $params['profile_id']);
        }
    }
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $cnt);
        return '';
    }
    return $cnt;
}

/**
 * Show a file type image
 * @param $params array
 * @param $smarty object
 * @return string
 */
function smarty_function_jrCore_file_type_image($params, $smarty)
{
    if (!isset($params['extension'])) {
        return jrCore_smarty_missing_error('extension');
    }
    $params['module'] = 'jrCore';
    if (is_file(APP_DIR . "/modules/jrCore/img/file_type_{$params['extension']}.png")) {
        $params['image'] = "file_type_{$params['extension']}.png";
    }
    elseif (isset($params['default_image'])) {
        $params['module'] = (isset($params['default_module'])) ? $params['default_module'] : 'jrCore';
        $params['image']  = $params['default_image'];
    }
    else {
        $params['image'] = 'file_type_unknown.png';
    }
    $params = jrCore_trigger_event('jrCore', 'file_type_image', $params);
    return smarty_function_jrCore_image($params, $smarty);
}

/**
 * Embed an image in a template
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_image($params, $smarty)
{
    global $_conf;
    if (!isset($params['image'])) {
        return jrCore_smarty_missing_error('image');
    }
    // See if we have a custom file for this image
    $skn = $_conf['jrCore_active_skin'];
    $tag = '';
    $typ = 'skin';
    if (isset($params['module'])) {
        $skn = $params['module'];
        $tag = 'mod_';
        $typ = 'module';
    }
    $_im = array();
    if (!empty($_conf["jrCore_{$skn}_custom_images"])) {
        $_im = json_decode($_conf["jrCore_{$skn}_custom_images"], true);
    }
    if (!empty($_im["{$params['image']}"][1]) && $_im["{$params['image']}"][1] == 'on') {
        $plg = jrCore_get_active_media_system();
        $fnc = "_{$plg}_media_get_custom_image_url";
        if (function_exists($fnc)) {
            $src = $fnc("{$tag}{$skn}_{$params['image']}");
        }
        else {
            $src = jrCore_get_base_url() . "/data/media/0/0/{$tag}{$skn}_{$params['image']}?r=" . $_im["{$params['image']}"][0]; // GOOD
        }
    }
    else {
        $url = jrCore_get_module_url('jrImage');
        if (isset($params['module'])) {
            // Do we have a skin override on this image?
            if (file_exists(APP_DIR . "/skins/{$_conf['jrCore_active_skin']}/img/{$params['module']}_{$params['image']}")) {
                $img = APP_DIR . "/skins/{$_conf['jrCore_active_skin']}/img/{$params['module']}_{$params['image']}";
                $src = jrCore_get_base_url() . "/{$url}/img/skin/{$_conf['jrCore_active_skin']}/{$params['module']}_{$params['image']}";
            }
            else {
                $img = APP_DIR . "/modules/{$skn}/img/{$params['image']}";
                $src = jrCore_get_base_url() . "/{$url}/img/module/{$skn}/{$params['image']}";
            }
        }
        else {
            $img = APP_DIR . "/skins/{$skn}/img/{$params['image']}";
            $src = jrCore_get_base_url() . "/{$url}/img/skin/{$skn}/{$params['image']}";
        }
        if (!isset($params['v'])) {
            $params['v'] = @filemtime($img);
        }
    }
    // Send out image trigger for source
    $_tm = array(
        'function'           => 'jrCore_image',
        'params'             => $params,
        'smarty'             => $smarty,
        'skin_custom_images' => $_im
    );
    $src = jrCore_trigger_event('jrImage', "{$typ}_image", $src, $_tm);
    $src .= "?skin={$_conf['jrCore_active_skin']}";
    if (!empty($params['v'])) {
        $src .= "&v={$params['v']}";
    }

    if (isset($params['src_only']) && $params['src_only'] === true) {
        if (!empty($params['assign'])) {
            $smarty->assign($params['assign'], $src);
            return '';
        }
        return $src;
    }
    if (!isset($params['title']) && isset($params['alt'])) {
        $params['title'] = $params['alt'];
    }
    $out = "<img src=\"{$src}\" ";
    // Our other params are optional
    foreach ($params as $k => $v) {
        $k = jrCore_str_to_lower($k);
        switch ($k) {
            case 'width':
            case 'height':
            case 'id':
            case 'class':
            case 'style':
            case 'onclick':
            case 'onmouseover':
            case 'onmouseout':
                $out .= "{$k}=\"{$v}\" ";
                break;
            case 'alt':
            case 'title':
                $out .= "{$k}=\"" . jrCore_entity_string($v) . "\" ";
                break;
        }
    }
    $out = rtrim($out, ' ') . '>';
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Embed a Power List into a template
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return array|string
 */
function smarty_function_jrCore_list($params, $smarty)
{
    global $_conf, $_post;
    if (empty($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (!jrCore_module_is_active($params['module'])) {
        // Not installed or wrong...
        return '';
    }
    // Must be a DS module...
    if (!jrCore_is_datastore_module($params['module'])) {
        return '';
    }

    $pid = 0;
    if (isset($params['profile_id']) && jrCore_checktype($params['profile_id'], 'number_nz')) {
        $pid = (int) $params['profile_id'];
    }
    elseif (isset($_post['_profile_id']) && jrCore_checktype($_post['_profile_id'], 'number_nz')) {
        $pid = (int) $_post['_profile_id'];
    }

    // Trigger - listeners can provide list_content!
    $params = jrCore_trigger_event('jrCore', 'template_list', $params, array('module' => $params['module'], 'profile_id' => $pid));
    $key    = md5(serialize($_post) . serialize($params));
    if (isset($params['list_content']) && strlen($params['list_content']) > 0) {
        jrCore_add_to_cache($params['module'], $key, $params['list_content'], false, $pid);
        if (!empty($params['assign'])) {
            $smarty->assign($params['assign'], $params['list_content']);
            return '';
        }
        return $params['list_content'];
    }

    // Check for cache
    if ((!isset($params['no_cache']) || $params['no_cache'] === false) && $tmp = jrCore_is_cached($params['module'], $key)) {
        if (!empty($params['assign'])) {
            $smarty->assign($params['assign'], $tmp);
            return '';
        }
        return $tmp;
    }

    $tpl_dir = null;
    if (!isset($params['template']) || strlen($params['template']) === 0) {
        // Check for Skin override
        if (is_file(APP_DIR . "/skins/{$_conf['jrCore_active_skin']}/{$params['module']}_item_list.tpl")) {
            $params['template'] = "{$params['module']}_item_list.tpl";
        }
        // See if this module provides one
        elseif (is_file(APP_DIR . "/modules/{$params['module']}/templates/item_list.tpl")) {
            $tpl_dir            = $params['module'];
            $params['template'] = 'item_list.tpl';
        }
        else {
            return jrCore_smarty_custom_error("{$params['module']}/templates/item_list.tpl not found");
        }
    }
    else {
        if (isset($params['tpl_dir']) && is_file(APP_DIR . "/modules/{$params['tpl_dir']}/templates/{$params['template']}")) {
            $tpl_dir = $params['tpl_dir'];
        }
        // Check for template
        if (!is_file(APP_DIR . "/skins/{$_conf['jrCore_active_skin']}/{$params['template']}") && is_file(APP_DIR . "/modules/{$params['module']}/templates/{$params['template']}")) {
            $tpl_dir = $params['module'];
        }
    }
    $module = $params['module'];
    unset($params['module']);

    // $params = array(
    //     'search' => array(
    //         'user_name = brian',
    //         'user_weight > 100'
    //     ),
    //     'order_by' => array(
    //         'user_name' => 'asc',
    //         'user_weight' => 'desc'
    //     ),
    //     'limit' => 50
    // );
    // {jrCore_list module="jrProfile" search1="profile_name = brian" search2="profile_name != test" order_by="created desc" template="list_profile_row.tpl" limit=5}
    // Set params for our function
    $_args = array(
        'jrcore_list_function_call_is_active' => 1
    );
    foreach ($params as $k => $v) {
        // Search
        if (strpos($k, 'search') === 0 && strlen($v) > 0) {
            if (!isset($_args['search'])) {
                $_args['search'] = array();
            }
            $_args['search'][] = $v;
        }
        // Order by
        elseif (strpos($k, 'order_by') === 0) {
            if (!isset($_args['order_by'])) {
                $_args['order_by'] = array();
            }
            list($fld, $dir) = explode(' ', $v);
            $fld                     = trim($fld);
            $_args['order_by'][$fld] = trim($dir);
        }
        // Group By
        elseif ($k == 'group_by') {
            $_args['group_by'] = trim($v);
        }
        // Limit
        elseif ($k == 'limit') {
            $_args['limit'] = (int) $v;
        }
        // Page break
        elseif ($k == 'pagebreak') {
            $_args['pagebreak'] = (int) $v;
        }
        // Page
        elseif ($k == 'page') {
            $_args['page'] = (int) $v;
        }
        elseif ($k == 'return_keys') {
            $_args['return_keys'] = explode(',', $v);
        }
        elseif ($k == 'fdebug' && $v == 1) {
            $_args['fdebug'] = true;
        }
        else {
            // Everything else
            $_args[$k] = $v;
        }
    }

    // Prep our data for display
    $_rs = jrCore_db_search_items($module, $_args);
    if (isset($params['return_count'])) {
        $_rs = (int) $_rs;
        if (!empty($params['assign'])) {
            $smarty->assign($params['assign'], $_rs);
            return '';
        }
        return $_rs;
    }
    if ($_rs && !is_array($_rs) && strpos($_rs, 'error') === 0) {
        if (!empty($params['assign'])) {
            $smarty->assign($params['assign'], $_rs);
            return '';
        }
        return $_rs;
    }
    if (isset($_rs['_items']) && is_array($_rs['_items']) && count($_rs['_items']) > 0) {
        foreach ($_rs['_items'] as $k => $_v) {
            $_rs['_items'][$k]['list_rank'] = (isset($_rs['info']['page']) && $_rs['info']['page'] > 1) ? intval($k + ((($_rs['info']['page'] - 1) * $_rs['info']['pagebreak']) + 1)) : intval($k + 1);
        }
    }

    // If we have been given NO template, just assign vars and return
    if ($params['template'] == 'null' && !empty($params['assign'])) {
        if (isset($_rs['_items'])) {
            $tmp = $_rs['_items'];
        }
        else {
            $tmp = $_rs;
        }
    }
    else {
        // Parse our template and return results
        $tmp = jrCore_parse_template($params['template'], $_rs, $tpl_dir);

        // See if we are including the default pager
        if (isset($params['pager']) && $params['pager'] !== "false") {
            $tpl = 'list_pager.tpl';
            $dir = 'jrCore';
            if (isset($params['pager_template'])) {
                $tpl = $params['pager_template'];
                $dir = $_conf['jrCore_active_skin'];
                if (!is_file(APP_DIR . "/skins/{$_conf['jrCore_active_skin']}/{$params['pager_template']}") && is_file(APP_DIR . "/modules/{$module}/templates/{$params['pager_template']}")) {
                    $dir = $module;
                }
            }
            // Check for AJAX loader
            if (isset($params['pager_load_id'])) {
                if (isset($params['pager_load_template'])) {
                    $_rs['pager_load_template'] = $params['pager_load_template'];
                }
                else {
                    $_rs['pager_load_template'] = $params['template'];
                }
                $_rs['pager_load_template'] = str_replace('.tpl', '', $_rs['pager_load_template']);

                if (isset($params['pager_load_url'])) {
                    $_rs['pager_load_url'] = $params['pager_load_url'];
                }
                else {
                    $_rs['pager_load_url'] = "{$_conf['jrCore_base_url']}/{$_rs['pager_load_template']}";
                }
                $_rs['pager_load_id'] = $params['pager_load_id'];
            }
            if (!is_array($_rs)) {
                $_rs = array();
            }
            $_rs['pager_show_jumper'] = (isset($_rs['info']['simplepagebreak'])) ? '0' : '1';
            $tmp                      .= jrCore_parse_template($tpl, $_rs, $dir);
        }
    }
    jrCore_add_to_cache($module, $key, $tmp, false, $pid);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $tmp);
        return '';
    }
    return $tmp;
}

/**
 * Run a Smarty template function for a module
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_module_function($params, $smarty)
{
    if (empty($params['function'])) {
        return jrCore_smarty_missing_error('function');
    }
    $mod = substr($params['function'], 0, strpos($params['function'], '_'));
    if (!jrCore_module_is_active($mod)) {
        return '';
    }
    $func = "smarty_function_{$params['function']}";
    if (!function_exists($func)) {
        // Not installed or wrong...
        return '';
    }
    unset($params['function']);
    return $func($params, $smarty);
}

/**
 * Jamroom CSS SRC URL generator
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_css_src($params, $smarty)
{
    if (empty($params['skin'])) {
        $params['skin'] = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    }
    $skn = $params['skin'];
    $sec = jrCore_get_config_value('jrCore', 'default_cache_seconds', 3600);
    $sum = jrCore_get_config_value('jrCore', "{$skn}_css_version", false);
    $prt = jrCore_get_server_protocol();
    $css = ($prt === 'https') ? "S{$sum}" : $sum;
    $src = jrCore_get_base_url() . "/data/cache/{$skn}/{$css}.css";
    $cdr = jrCore_get_module_cache_dir($skn);
    if (!$sum || !is_file("{$cdr}/{$css}.css") || $sec == 0 || jrCore_is_developer_mode()) {
        $sum = jrCore_create_master_css($skn);
        $css = ($prt === 'https') ? "S{$sum}" : $sum;
        $src = jrCore_get_base_url() . "/data/cache/{$skn}/{$css}.css";
        $_tm = jrCore_trigger_event('jrCore', 'master_css_src', array('src' => $src, 'sum' => $sum), $params);
        if (!empty($_tm['src']) && $_tm['src'] != $src) {
            // We were changed by a listener
            $src = $_tm['src'];
        }
    }
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $src);
        return '';
    }
    return $src;
}

/**
 * Jamroom Javascript SRC URL generator
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_javascript_src($params, $smarty)
{
    if (empty($params['skin'])) {
        $params['skin'] = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    }
    $skn = $params['skin'];
    $sec = jrCore_get_config_value('jrCore', 'default_cache_seconds', 3600);
    $sum = jrCore_get_config_value('jrCore', "{$skn}_javascript_version", false);
    $prt = jrCore_get_server_protocol();
    $jsv = ($prt === 'https') ? "S{$sum}" : $sum;
    $src = jrCore_get_base_url() . "/data/cache/{$skn}/{$jsv}.js";
    $cdr = jrCore_get_module_cache_dir($skn);
    if (!$sum || !is_file("{$cdr}/{$jsv}.js") || $sec == 0 || jrCore_is_developer_mode()) {
        $sum = jrCore_create_master_javascript($skn);
        $jsv = ($prt === 'https') ? "S{$sum}" : $sum;
        $src = jrCore_get_base_url() . "/data/cache/{$skn}/{$jsv}.js";
        $_tm = jrCore_trigger_event('jrCore', 'master_javascript_src', array('src' => $src, 'sum' => $sum), $params);
        if (!empty($_tm['src']) && $_tm['src'] != $src) {
            // We were changed by a listener
            $src = $_tm['src'];
        }
    }
    if (jrUser_is_admin() && is_file("{$cdr}/{$jsv}-admin.js")) {
        jrCore_set_flag('admin_javascript_src', jrCore_get_base_url() . "/data/cache/{$skn}/{$jsv}-admin.js");
    }
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $src);
        return '';
    }
    return $src;
}

/**
 * Generate a unique Form Session for an embedded template form
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_form_create_session($params, $smarty)
{
    global $_conf;
    if (!isset($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (!jrCore_module_is_active($params['module'])) {
        return '';
    }
    if (!isset($params['option']) || strlen($params['option']) === 0) {
        return jrCore_smarty_invalid_error('option');
    }
    $url = jrCore_get_module_url($params['module']);
    $_fm = array(
        'token'  => jrCore_form_token_create("{$params['module']}/{$params['option']}"),
        'module' => $params['module'],
        'option' => $params['option'],
        'action' => "{$_conf['jrCore_base_url']}/{$url}/{$params['option']}_save"
    );
    if (!empty($params['action'])) {
        $_fm['action'] = $params['action'];
    }
    jrCore_form_create_session($_fm['token'], $_fm);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $_fm['token']);
        return '';
    }
    return $_fm['token'];
}

/**
 * Generate a unique Form Token for use in a form
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_form_token($params, $smarty)
{
    global $_post;
    $uri = (isset($_post['_uri'])) ? $_post['_uri'] : '/';
    $out = jrCore_form_token_create($uri);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Display a language string by language ID
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_lang($params, $smarty)
{
    global $_conf;
    if (empty($params['module'])) {
        if (empty($params['skin'])) {
            $out = 'NO_LANG_MODULE_OR_SKIN';
            if (!empty($params['assign'])) {
                $smarty->assign($params['assign'], $out);
                return '';
            }
            return $out;
        }
        $params['module'] = $params['skin'];
    }
    if (!isset($params['id'])) {
        $out = 'INVALID_LANG_ID';
        if (!empty($params['assign'])) {
            $smarty->assign($params['assign'], $out);
            return '';
        }
        return $out;
    }

    // Bring in strings
    $_lang = jrUser_load_lang_strings();

    $out = 'NO_LANG_FOR_ID';
    if (isset($_lang["{$params['module']}"]["{$params['id']}"])) {
        $out = $_lang["{$params['module']}"]["{$params['id']}"];
        if (jrCore_module_is_active('jrDeveloper') && isset($_conf['jrDeveloper_language_verify']) && $_conf['jrDeveloper_language_verify'] == "on") {
            $tpl = $smarty->tpl_vars['jr_template_directory']->value . ' / ' . $smarty->tpl_vars['jr_template']->value;
            $uri = $smarty->tpl_vars['_post']->value['_uri'];
            jrDeveloper_verify_lang($out, $params['default'], $params['module'], $params['id'], $_lang['_settings'], $tpl, $uri);
        }
    }
    elseif (!empty($params['default'])) {
        if (jrUser_is_master()) {
            $out = '*' . $params['default'] . '*';
        }
        else {
            $out = $params['default'];
        }
    }
    if (strpos(' ' . $out, '%')) {
        $_rp = array();
        foreach ($params as $k => $v) {
            if (is_numeric($k)) {
                $_rp["%{$k}"] = $v;
            }
            $out = str_replace(array_keys($_rp), $_rp, $out);
        }
    }
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Get current server protocol request came in on
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_server_protocol($params, $smarty)
{
    $prt = jrCore_get_server_protocol();
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $prt);
        return '';
    }
    return $prt;
}

/**
 * Get the configured URL for a specific module
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_module_url($params, $smarty)
{
    global $_urls;
    if (empty($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    $url = array_search($params['module'], $_urls);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $url);
        return '';
    }
    return $url;
}

/**
 * Format a date_birthday field to the system configured time format
 * @param string $date in YYYYMMDD format
 * @param string $format date format
 * @return string
 */
function smarty_modifier_jrCore_date_birthday_format($date, $format = null)
{
    global $_conf;
    if (strlen($date) != 8 || !jrCore_checktype($date, 'number_nn')) {
        return $date;
    }
    if (is_null($format) || strlen($format) === 0) {
        $format = (isset($_conf['jrBirthday_display_format']) && strlen($_conf['jrBirthday_display_format']) > 0) ? $_conf['jrBirthday_display_format'] : $_conf['jrCore_date_format'];
    }
    if ($format == '%D') {
        $format = '%m/%d/%y';
    }
    $out     = $format;
    $tformat = ' ' . $format;
    if (strpos($tformat, '%d')) {
        $out = str_replace('%d', substr($date, 6), $out);
    }
    if (strpos($tformat, '%e')) {
        $x   = (int) substr($date, 6);
        $out = str_replace('%e', $x, $out);
    }
    if (strpos($tformat, '%s')) {
        switch (substr($date, 6)) {
            case '1':
            case '21':
            case '31':
                $x = 'st';
                break;
            case '2':
            case '22':
                $x = 'nd';
                break;
            case '3':
            case '23':
                $x = 'rd';
                break;
            default:
                $x = 'th';
                break;
        }
        $out = str_replace('%s', $x, $out);
    }
    if (strpos($tformat, '%m')) {
        $out = str_replace('%m', substr($date, 4, 2), $out);
    }
    if (strpos($tformat, '%n')) {
        $x   = (int) substr($date, 4, 2);
        $out = str_replace('%n', $x, $out);
    }
    if (strpos($tformat, '%b')) {
        $x   = (int) substr($date, 4, 2);
        $x   = date('M', strtotime("2000-{$x}-01"));
        $out = str_replace('%b', $x, $out);
    }
    if (strpos($tformat, '%B')) {
        $x   = (int) substr($date, 4, 2);
        $x   = date('F', strtotime("2000-{$x}-01"));
        $out = str_replace('%B', $x, $out);
    }
    if (strpos($tformat, '%Y')) {
        $out = str_replace('%Y', substr($date, 0, 4), $out);
    }
    if (strpos($tformat, '%y')) {
        $out = str_replace('%y', substr($date, 2, 2), $out);
    }
    return $out;
}

/**
 * Format an epoch time stamp to the system configured time format
 * @param int $timestamp Epoch Time Stamp to convert
 * @param string $format Format for output
 * @param bool $adjust set to FALSE to prevent adjusting time
 * @return string
 */
function smarty_modifier_jrCore_date_format($timestamp, $format = null, $adjust = true)
{
    return jrCore_format_time($timestamp, false, $format, $adjust);
}

/**
 * Convert @ tags into links to profiles
 * @param string $text String to convert at tags in
 * @return string
 */
function smarty_modifier_jrCore_convert_at_tags($text)
{
    global $_conf;
    if (strpos(' ' . $text, '@')) {

        preg_match_all('/([^A-Za-z\d.])(@([_a-z\d\-]+))/i', "\n{$text}\n", $_tmp);
        if ($_tmp && is_array($_tmp) && isset($_tmp[3])) {
            $_tm = jrCore_get_flag('jrcore_convert_at_tags_urls');
            if (!$_tm) {
                $_tm = array();
            }
            $_pu = array();
            foreach ($_tmp[3] as $url) {
                if (!isset($_tm[$url])) {
                    $_pu[$url] = $url;
                }
            }
            if (count($_pu) > 0) {
                $_sc = array(
                    'search'         => array(
                        'profile_url in ' . implode(',', $_pu)
                    ),
                    'return_keys'    => array('_profile_id', 'profile_url'),
                    'skip_triggers'  => true,
                    'ignore_pending' => true,
                    'privacy_check'  => false,
                    'order_by'       => false,
                    'limit'          => count($_pu)
                );
                $_sc = jrCore_db_search_items('jrProfile', $_sc);
                if ($_sc && is_array($_sc) && isset($_sc['_items'])) {
                    foreach ($_sc['_items'] as $_p) {
                        $_tm["{$_p['profile_url']}"] = $_p['profile_url'];
                    }
                }
            }
            if (count($_tm) > 0) {
                foreach ($_tmp[3] as $url) {
                    // If the URL is NOT in $_tm then it's not a valid profile_url
                    if (isset($_tm[strtolower($url)])) {
                        $text = trim(preg_replace('/([^A-Za-z\d.])@(' . $url . ')/i', '$1<a href="' . $_conf['jrCore_base_url'] . '/$2"><span class="at_link">@$2</span></a>', "\n{$text}\n"));
                    }
                }
                jrCore_set_flag('jrcore_convert_at_tags_urls', $_tm);
            }
        }
        return trim($text);
    }
    return $text;
}

/**
 * Run internal text format functions on a string
 * @param string $string String to format
 * @param int $quota_id Quota ID for Profile ID. Or 'allow_all_formatters' for everything.
 * @param string $whitelist Only allow defined string format listeners to run
 * @param string $blacklist Do not allow modifiers in blacklist to run
 * @return string
 */
function smarty_modifier_jrCore_format_string($string, $quota_id = 0, $whitelist = null, $blacklist = null)
{
    global $_mods;
    jrCore_set_flag('jrcore_format_string_active', 1);

    // Check for white list
    $_wl = false;
    if (!empty($whitelist)) {
        $whitelist = explode(',', $whitelist);
        if (is_array($whitelist)) {
            $_wl = array();
            foreach ($whitelist as $v) {
                $v       = trim($v);
                $_wl[$v] = $v;
            }
        }
        unset($whitelist);
    }

    // Check for black list
    $_bl = false;
    if (!empty($blacklist)) {
        $blacklist = explode(',', $blacklist);
        if (is_array($blacklist)) {
            $_bl = array();
            foreach ($blacklist as $v) {
                $v       = trim($v);
                $_bl[$v] = $v;
            }
        }
        unset($blacklist);
    }

    // Get Quota info
    $_qt = array();
    $_qf = array();
    if ($quota_id != 'allow_all_formatters' && $quota_id > 0) {
        $_qt = jrProfile_get_quota($quota_id);
        if (!empty($_qt['quota_jrCore_active_formatters'])) {
            $_qf = explode(',', $_qt['quota_jrCore_active_formatters']);
        }
    }

    $clean = true;
    $_tmp  = jrCore_get_registered_module_features('jrCore', 'format_string');
    if ($_tmp && is_array($_tmp)) {

        // Have we reordered yet?
        if (!$_ord = jrCore_get_flag('jrcore_format_string_ordered')) {
            $_ord = array();
            foreach ($_mods as $m => $i) {
                if (isset($_tmp[$m])) {
                    $_ord[$m] = $_tmp[$m];
                }
            }
            jrCore_set_flag('jrcore_format_string_ordered', $_ord);
        }
        unset($_tmp);

        // If we have a white list defined, we do it in the defined order
        if (is_array($_wl) && count($_wl) > 0) {

            // NOTE: If we are allowed BBCode, then we have to run the CODE
            // formatter first if we have any code blocks - otherwise string
            // formatters can mess with the contents of the code block
            if (stripos(' ' . $string, '[code]') && isset($_wl['jrCore_format_string_bbcode'])) {
                // We have a code block
                require_once APP_DIR . '/modules/jrCore/bbcode/code.php';
                $string = jrCore_bbcode_code($string);
            }

            foreach ($_wl as $name) {
                foreach ($_ord as $_opts) {
                    foreach ($_opts as $fnc => $_desc) {
                        if ($_desc['wl'] == $name) {
                            if ($fnc == 'jrCore_format_string_allowed_html') {
                                if ($_qt['quota_jrCore_allow_all_html'] != 'on') {
                                    $string = $fnc($string, $quota_id);
                                    $clean  = false;
                                }
                            }
                            else {
                                $string = $fnc($string, $quota_id);
                            }
                        }
                    }
                }
            }
        }
        else {

            // NOTE: If we are allowed BBCode, then we have to run the CODE
            // formatter first if we have any code blocks - otherwise string
            // formatters can mess with the contents of the code block
            if (stripos(' ' . $string, '[code]')) {
                // We have a code block
                // Is it blacklisted on this call?
                if (!$_bl || !isset($_bl['jrCore_format_string_bbcode'])) {
                    // We're not purposefully blacklisted - check quota for bbcode
                    if ($quota_id == 'allow_all_formatters' || (in_array('jrCore_format_string_bbcode', $_qf) && (!$_wl || isset($_wl['jrCore_format_string_bbcode'])))) {
                        require_once APP_DIR . '/modules/jrCore/bbcode/code.php';
                        $string = jrCore_bbcode_code($string);
                    }
                }
            }

            // Process
            foreach ($_ord as $_opts) {
                foreach ($_opts as $fnc => $_desc) {
                    if (function_exists($fnc)) {
                        // See if this formatter is active in the Quota
                        $tn = (isset($_desc['wl'])) ? $_desc['wl'] : '';
                        if ($quota_id == 'allow_all_formatters') {
                            if ($_bl && isset($_bl[$tn])) {
                                // Blacklisted
                                continue;
                            }
                            $string = $fnc($string, $quota_id);
                            if ($fnc == 'jrCore_format_string_allowed_html') {
                                $clean = false;
                            }
                        }
                        else {
                            if ((in_array($fnc, $_qf) && (!$_wl || isset($_wl[$tn]))) && (!$_bl || !isset($_bl[$tn]))) {
                                if ($fnc == 'jrCore_format_string_allowed_html') {
                                    if ($_qt['quota_jrCore_allow_all_html'] != 'on') {
                                        $string = $fnc($string, $quota_id);
                                        $clean  = false;
                                    }
                                }
                                else {
                                    $string = $fnc($string, $quota_id);
                                }
                            }
                        }
                    }
                }
            }
        }
    }

    if ((!$_wl || isset($_wl['nl2br'])) && (!$_bl || !isset($_bl['nl2br']))) {
        $string = str_replace("\r\n", "\n", $string);
        $string = nl2br($string);
    }

    // Handle BBCode [code] blocks
    if ($_ctemp = jrCore_get_flag('jrcore_bbcode_replace_blocks')) {
        $_ctemp = array_reverse($_ctemp);
        // NOTE: We process this TWICE here on purpose - the second time
        // through catches quotes, URLs and code blocks INSIDE a quote
        $string = str_replace(array_keys($_ctemp), $_ctemp, $string);
        $string = str_replace(array_keys($_ctemp), $_ctemp, $string);
        jrCore_delete_flag('jrcore_bbcode_replace_blocks');
    }

    // Trigger
    $_data = array('string' => $string);
    $_data = jrCore_trigger_event('jrCore', 'format_string_display', $_data);

    $_vp             = array(
        ' />'   => '>',
        '/>'    => '>',
        ' <br>' => '<br>'
    );
    $_data['string'] = str_replace(array_keys($_vp), $_vp, $_data['string']);

    // Cleanup output tags
    $_vp             = array(
        '<p><br>'       => '<p>',
        '</p><br>'      => '</p>',
        '<ul><br>'      => '<ul>',
        '</ul><br>'     => '</ul>',
        "<br>\n<ul>"    => "\n<ul>",
        '<ol><br>'      => '<ol>',
        "<br>\n<ol>"    => "\n<ol>",
        '</ol><br>'     => '</ol>',
        '</li><br>'     => '</li>',
        '<li><br>'      => '<li>',
        '<br>\n<li>'    => '<li>',
        '<tr><br>'      => '<tr>',
        '</tr><br>'     => '</tr>',
        '<td><br>'      => '<td>',
        '</td><br>'     => '</td>',
        "<br>\n<tbody>" => "\n<tbody>",
        '<tbody><br>'   => '<tbody>',
        '</tbody><br>'  => '</tbody>',
        "<br>\n<table>" => "\n<table>"
    );
    $_data['string'] = str_replace(array_keys($_vp), $_vp, $_data['string']);
    $_data['string'] = jrCore_remove_unwanted_breaks($_data['string']);

    if ($clean) {
        // Cleanup output HTML
        $_data['string'] = jrCore_clean_html($_data['string']);
    }
    jrCore_delete_flag('jrcore_format_string_active');
    return $_data['string'];
}

/**
 * Strip all HTML from a string
 * @param string $string
 * @return string
 */
function smarty_modifier_jrCore_strip_html($string)
{
    return jrCore_strip_html($string, null, true);
}

/**
 * We don't want breaks in some places
 * @param string $string string to check for unwanted breaks
 * @return string
 */
function jrCore_remove_unwanted_breaks($string)
{
    if (strpos(' ' . $string, '<br>')) {

        // Fix up tables
        if (strpos(' ' . $string, '<table')) {
            $_tmp = explode("\n", $string);
            if ($_tmp && is_array($_tmp)) {
                $in_table = false;
                foreach ($_tmp as $k => $line) {
                    if (strpos(' ' . $line, '<table')) {
                        $in_table = true;
                        $_tmp[$k] = str_replace('><br>', '>', $line);
                    }
                    elseif ($in_table) {
                        $_tmp[$k] = str_replace('><br>', '>', $line);
                    }
                }
                $string = implode("\n", $_tmp);
            }
        }
    }
    return $string;
}

/**
 * Registered Core string format - Allowed HTML
 * @param string $string String to format
 * @param int $quota_id Quota ID for Profile ID
 * @return string
 */
function jrCore_format_string_allowed_html($string, $quota_id = 0)
{
    if ($quota_id == 'allow_all_formatters') {
        return $string;
    }
    // Allowed HTML
    if (jrCore_checktype($quota_id, 'number_nz') && strpos(' ' . $string, '<')) {
        // If we have an active Quota ID we need to properly strip tags
        $_qt = jrProfile_get_quota($quota_id);
        if (!empty($_qt['quota_jrCore_allowed_tags']) > 0) {
            return jrCore_strip_html($string, $_qt['quota_jrCore_allowed_tags']);
        }
    }
    // not allowed
    return jrCore_strip_html($string);
}

/**
 * Registered core string formatter - Convert @ tags
 * @param string $string String to format
 * @param int $quota_id Quota ID for Profile ID
 * @return string
 */
function jrCore_format_string_convert_at_tags($string, $quota_id = 0)
{
    return smarty_modifier_jrCore_convert_at_tags($string);
}

/**
 * Registered core string formatter - Clickable URLs
 * @param string $string String to format
 * @param int $quota_id Quota ID for Profile ID
 * @return string
 */
function jrCore_format_string_clickable_urls($string, $quota_id = 0)
{
    // Convert URL strings
    if (!stripos(' ' . $string, 'http')) {
        return $string;
    }
    return jrCore_string_to_url($string);
}

/**
 * Registered core string formatter - BBCode
 * @param string $string String to format
 * @param int $quota_id Quota ID for Profile ID
 * @return string
 */
function jrCore_format_string_bbcode($string, $quota_id = 0)
{
    if (strpos(' ' . $string, '[') && strpos(' ' . $string, ']')) {
        $_tags = jrCore_get_flag('jrcore_bbcode_tags');
        if (!$_tags) {
            $_tags = array();
            foreach (glob(APP_DIR . '/modules/jrCore/bbcode/*.php') as $file) {
                require_once $file; // Get function loaded
                $file = basename($file);
                $name = "jrCore_bbcode_" . str_replace('.php', '', $file);
                if (function_exists($name)) {
                    $_tags[$name] = $file;
                }
            }
            jrCore_set_flag('jrcore_bbcode_tags', $_tags);
        }
        // We always run CODE first...
        if (isset($_tags['jrCore_bbcode_code'])) {
            $string = jrCore_bbcode_code($string);
            unset($_tags['jrCore_bbcode_code']);
        }
        foreach ($_tags as $name => $file) {
            $string = $name($string);
        }
    }
    return $string;
}

/**
 * Return portion of string up to first <!-- pagebreak -->
 * @param $text string String to return substring of
 * @return string
 * @deprecated - this functionality has moved to jrBlog
 */
function smarty_modifier_jrCore_readmore($text)
{
    if (function_exists('smarty_modifier_jrBlog_readmore')) {
        return smarty_modifier_jrBlog_readmore($text);
    }
    return $text;
}

/**
 * Return "clean" HTML
 * @param $html string HTML string
 * @return string
 */
function smarty_modifier_jrCore_clean_html($html)
{
    return jrCore_clean_html($html);
}

/**
 * Clean HTML so it is properly formatted
 * @param $html string HTML
 * @return string
 */
function jrCore_clean_html($html)
{
    if (class_exists('DOMDocument')) {
        libxml_use_internal_errors(true);
        $doc                     = new DOMDocument();
        $doc->substituteEntities = false;
        if (!$enc = @mb_detect_encoding($html, 'auto')) {
            $enc = 'UTF-8';
        }
        $temp = @mb_convert_encoding($html, 'HTML-ENTITIES', $enc);
        if (!$temp) {
            $temp = $html;
        }
        $html = jrCore_balance_html_tags($temp);
        if (!strpos(' ' . $html, '<html')) {
            $html = '<html lang="en"><body>' . $html . '</body></html>';
        }
        $doc->loadHTML($html);
        return substr($doc->saveHTML($doc->getElementsByTagName('body')->item(0)), 6, -7);
    }
    return jrCore_closetags($html);
}

/**
 * Balance HTML tags
 * @param $html
 * @return mixed
 */
function jrCore_balance_html_tags($html)
{
    if (substr_count($html, '<') !== substr_count($html, '>')) {
        $_tm = explode('<', $html);
        $_nw = array();
        $rep = false;
        if ($_tm && is_array($_tm)) {
            if (strpos($html, '<') !== 0) {
                $_nw[] = $_tm[0];
                unset($_tm[0]);
            }
            if (count($_tm) > 0) {
                // We know the rest STARTED with a <
                foreach ($_tm as $chunk) {
                    if (strlen($chunk) > 0) {
                        if (!strpos($chunk, '>')) {
                            $_nw[] = '&lt;' . $chunk;
                            $rep   = true;
                        }
                        else {
                            $_nw[] = '<' . $chunk;
                        }
                    }
                }
            }
            if ($rep) {
                $html = implode('', $_nw);
            }
        }
    }
    return $html;
}

/**
 * Close open HTML tags in a string that are left unclosed
 * http://stackoverflow.com/questions/3810230/php-how-to-close-open-html-tag-in-a-string
 * @param $html string HTML to close tags for
 * @return string
 */
function jrCore_closetags($html)
{
    preg_match_all('#<(?!meta|img|br|hr|input\b)\b([a-z]+)(?: .*)?(?<![/| ])>#iU', $html, $result);
    $openedtags = $result[1];
    preg_match_all('#</([a-z]+)>#iU', $html, $result);
    $closedtags = $result[1];
    $len_opened = count($openedtags);
    if (count($closedtags) == $len_opened) {
        return $html;
    }
    $openedtags = array_reverse($openedtags);
    for ($i = 0; $i < $len_opened; $i++) {
        if (!in_array($openedtags[$i], $closedtags)) {
            $html .= '</' . $openedtags[$i] . '>';
        }
        else {
            unset($closedtags[array_search($openedtags[$i], $closedtags)]);
        }
    }
    return $html;
}

/**
 * Get Admin Menu index page for a module
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_get_module_index($params, $smarty)
{
    if (empty($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    $out = jrCore_get_module_index($params['module']);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Create a page button using a Material Design icon
 * @link https://material.io/resources/icons
 * @params string $id DOM ID for button
 * @params string $icon Material Design Icon
 * @params string $onclick JS Onclick event function
 * @params array $_att Additional HTML attributes
 * @return string Returns HTML of button code
 */
function smarty_function_jrCore_icon_button($params, $smarty)
{
    if (empty($params['icon'])) {
        return jrCore_smarty_missing_error('icon');
    }
    $cls = 'form_icon_button';
    if (!empty($params['class'])) {
        $cls = $params['class'];
    }

    $lnk = '';
    $key = "material_design_css_included_{$cls}";
    if (!$class = jrCore_get_flag($key)) {
        // What style are we doing? "outlined" is default
        $class = 'material-icons-outlined';
        $style = '+Outlined';
        if (!empty($params['style'])) {
            $style = '+' . $params['style'];
            $class = 'material-icons-' . strtolower($params['style']);
        }
        if ($_tmp = jrCore_get_registered_module_features('jrCore', 'icon_style')) {
            $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
            if (isset($_tmp[$skin])) {
                $value = array_keys($_tmp[$skin]);
                $value = reset($value);
                switch ($value) {
                    case 'two-tone':
                        $class = 'material-icons-two-tone';
                        $style = '+Two+Tone';
                        break;
                    case 'round':
                        $class = 'material-icons-round';
                        $style = '+Round';
                        break;
                    case 'sharp':
                        $class = 'material-icons-sharp';
                        $style = '+Sharp';
                        break;
                    case 'filled':
                        $class = 'material-icons';
                        $style = '';
                        break;
                }
            }
        }
        if (!isset($params['inline_css'])) {
            $_css = array('source' => "https://fonts.googleapis.com/icon?family=Material+Icons{$style}");
            jrCore_create_page_element('css_href', $_css);
        }
        else {
            $lnk = '<link rel="stylesheet" property="stylesheet" href="https://fonts.googleapis.com/icon?family=Material+Icons' . $style . '">';
        }
        jrCore_set_flag($key, $class);
    }
    $icon = trim($params['icon']);
    $icon = "<i class=\"{$class} {$cls}-icon\">{$icon}</i>";

    $add = '';
    if (!empty($params['onclick'])) {
        if ($params['onclick'] == 'disabled') {
            $cls .= " form_button_disabled";
            $add = ' disabled="disabled"';
        }
    }
    $id = '';
    if (!empty($params['id'])) {
        $id = ' id="' . trim($params['id']) . '"';
    }
    $html = '<div' . $id . ' class="' . $cls . '"' . $add;
    if ($params['onclick'] != 'disabled') {
        $html .= ' onclick="' . $params['onclick'] . '"';
    }
    $html .= '>' . $icon . '</div>';
    return $lnk . $html;
}

/**
 * Create a Create button for a new DataStore item
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_item_create_button($params, $smarty)
{
    global $_conf, $_user;
    if (!jrUser_is_logged_in() || (!jrProfile_is_profile_view() && (!isset($params['profile_check']) || $params['profile_check'] === true))) {
        return '';
    }

    if (empty($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (!jrCore_module_is_active($params['module'])) {
        return '';
    }
    // Check for "icon" param
    if (!empty($params['create_icon'])) {
        $params['icon'] = $params['create_icon'];
    }
    if (empty($params['icon'])) {
        $params['icon'] = 'plus';
    }
    if (!isset($params['profile_id'])) {
        return jrCore_smarty_missing_error('profile_id');
    }
    if (!jrCore_checktype($params['profile_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('profile_id');
    }

    // Check for group requirement
    if (isset($params['group'])) {
        switch ($params['group']) {
            case 'master':
                if (!jrUser_is_master()) {
                    return '';
                }
                break;
            case 'admin':
                if (!jrUser_is_admin()) {
                    return '';
                }
                break;
        }
    }

    $out = '';

    // See if this user has access to perform this action on this profile
    if (jrProfile_is_profile_owner($params['profile_id'])) {

        // Bring in language strings
        $_lang = jrUser_load_lang_strings();

        // See if we are using the default view
        $def = 'create';
        if (!empty($params['view'])) {
            $def = trim($params['view']);
        }

        // button ID
        $bid = "{$params['module']}_create";
        if (isset($params['id'])) {
            $bid = $params['id'];
        }

        $url = jrCore_get_module_url($params['module']);
        if (isset($params['create_action'])) {
            $onc = "jrCore_window_location('{$_conf['jrCore_base_url']}/{$params['create_action']}');return false";
            $url = "{$_conf['jrCore_base_url']}/{$params['create_action']}";
        }
        elseif (isset($params['action'])) {
            $onc = "jrCore_window_location('{$_conf['jrCore_base_url']}/{$params['action']}');return false";
            $url = "{$_conf['jrCore_base_url']}/{$params['action']}";
        }
        else {
            $onc = "jrCore_window_location('{$_conf['jrCore_base_url']}/{$url}/{$def}');return false";
            $url = "{$_conf['jrCore_base_url']}/{$url}/{$def}";
        }
        // See if we are being given the onclick
        if (!empty($params['onclick'])) {
            $onc = $params['onclick'];
            $url = '';
        }

        // See if we need to show the user they have reached their limit
        if (!jrUser_is_admin()) {
            $q_max = isset($_user["quota_{$params['module']}_max_items"]) ? intval($_user["quota_{$params['module']}_max_items"]) : 0;
            if ($q_max > 0) {
                // See if we are limiting the number of items that can be created by a profile in this quota
                $p_cnt = jrCore_get_flag("profile_{$params['module']}_item_count");
                if (!$p_cnt) {
                    $p_cnt = jrCore_db_get_item_key('jrProfile', $_user['user_active_profile_id'], "profile_{$params['module']}_item_count");
                }
                if ($p_cnt >= $q_max) {
                    $onc = "jrCore_alert('" . addslashes($_lang['jrCore'][70]) . "');return false;";
                    $url = '';
                }
            }
        }

        if (isset($params['create_title'])) {
            $params['title'] = $params['create_title'];
        }
        elseif (isset($params['create_alt'])) {
            $params['title'] = $params['create_alt'];
        }
        elseif (isset($params['alt']) && jrCore_checktype($params['alt'], 'number_nz') && isset($_lang["{$params['module']}"]["{$params['alt']}"])) {
            $params['title'] = $_lang["{$params['module']}"]["{$params['alt']}"];
        }
        if (empty($params['alt'])) {
            $params['title'] = $_lang['jrCore'][36];
        }

        $siz = null;
        if (isset($params['size']) && jrCore_checktype($params['size'], 'number_nz')) {
            $siz = (int) $params['size'];
        }
        $cls = null;
        if (isset($params['class']) && strlen($params['class']) > 0) {
            $cls = $params['class'];
        }
        $clr = null;
        if (isset($params['color']) && strlen($params['color']) === 6 && jrCore_checktype($params['color'], 'hex')) {
            $clr = $params['color'];
        }

        $_tmp = array(
            'id'      => $bid,
            'url'     => $url,
            'onclick' => $onc,
            'icon'    => jrCore_get_icon_html($params['icon'], $siz, $cls, $clr),
            'title'   => jrCore_entity_string($params['title']),
            'params'  => $params
        );
        $out  = jrCore_parse_template('item_action_button.tpl', $_tmp, 'jrCore');

    }
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Create an Update button for a DataStore item
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_item_update_button($params, $smarty)
{
    global $_conf;
    if (!jrUser_is_logged_in() || (!jrProfile_is_profile_view() && (!isset($params['profile_check']) || $params['profile_check'] === true))) {
        return '';
    }
    if (empty($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (!jrCore_module_is_active($params['module'])) {
        return '';
    }
    // Check for "icon" param
    if (!empty($params['update_icon'])) {
        $params['icon'] = $params['update_icon'];
    }
    if (empty($params['icon'])) {
        $params['icon'] = 'gear';
    }
    if (!isset($params['action']) && !isset($params['update_action'])) {
        if (!isset($params['item_id'])) {
            return jrCore_smarty_missing_error('item_id');
        }
        if (!jrCore_checktype($params['item_id'], 'number_nz')) {
            return jrCore_smarty_invalid_error('item_id');
        }
    }
    if (!isset($params['profile_id'])) {
        return jrCore_smarty_missing_error('profile_id');
    }
    if (!jrCore_checktype($params['profile_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('profile_id');
    }

    // Check for group requirement
    if (!isset($params['group'])) {
        $params['group'] = 'owner';
    }
    switch ($params['group']) {
        case 'master':
            if (!jrUser_is_master()) {
                return '';
            }
            break;
        case 'admin':
            if (!jrUser_is_admin()) {
                return '';
            }
            break;
        case 'owner':
            if (!jrProfile_is_profile_owner($params['profile_id'])) {
                return '';
            }
            break;
    }

    // Bring in language strings
    $_lang = jrUser_load_lang_strings();

    // See if we are using the default view
    $def = 'update';
    if (!empty($params['view'])) {
        $def = trim($params['view']);
    }
    // Figure button ID
    $bid = "{$params['module']}_update_{$params['item_id']}";
    if (isset($params['id'])) {
        $bid = $params['id'];
    }

    $url = jrCore_get_module_url($params['module']);
    if (isset($params['update_action'])) {
        $onc = "jrCore_window_location('{$_conf['jrCore_base_url']}/{$params['update_action']}')";
        $url = "{$_conf['jrCore_base_url']}/{$params['update_action']}";
    }
    elseif (isset($params['action'])) {
        $onc = "jrCore_window_location('{$_conf['jrCore_base_url']}/{$params['action']}')";
        $url = "{$_conf['jrCore_base_url']}/{$params['action']}";
    }
    else {
        $onc = "jrCore_window_location('{$_conf['jrCore_base_url']}/{$url}/{$def}/id={$params['item_id']}')";
        $url = "{$_conf['jrCore_base_url']}/{$url}/{$def}/id={$params['item_id']}";
    }
    // See if we are being given the onclick
    if (!empty($params['onclick'])) {
        $onc = $params['onclick'];
        $url = '';
    }

    if (isset($params['update_title'])) {
        $params['title'] = $params['update_title'];
    }
    elseif (isset($params['update_alt'])) {
        $params['title'] = $params['update_alt'];
    }
    elseif (isset($params['alt']) && jrCore_checktype($params['alt'], 'number_nz') && isset($_lang["{$params['module']}"]["{$params['alt']}"])) {
        $params['title'] = $_lang["{$params['module']}"]["{$params['alt']}"];
    }
    if (empty($params['alt'])) {
        $params['title'] = $_lang['jrCore'][37];
    }

    $siz = null;
    if (isset($params['size']) && jrCore_checktype($params['size'], 'number_nz')) {
        $siz = (int) $params['size'];
    }
    $cls = null;
    if (isset($params['class']) && strlen($params['class']) > 0) {
        $cls = $params['class'];
    }
    $clr = null;
    if (isset($params['color']) && strlen($params['color']) === 6 && jrCore_checktype($params['color'], 'hex')) {
        $clr = $params['color'];
    }

    $_tmp = array(
        'id'      => $bid,
        'url'     => $url,
        'onclick' => $onc,
        'icon'    => jrCore_get_icon_html($params['icon'], $siz, $cls, $clr),
        'title'   => jrCore_entity_string($params['title']),
        'params'  => $params
    );
    $out  = jrCore_parse_template('item_action_button.tpl', $_tmp, 'jrCore');

    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Create a delete button for a DataStore item
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_item_delete_button($params, $smarty)
{
    global $_conf;
    if (!jrUser_is_logged_in() || (!jrProfile_is_profile_view() && (!isset($params['profile_check']) || $params['profile_check'] === true))) {
        return '';
    }
    if (empty($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (!jrCore_module_is_active($params['module'])) {
        return '';
    }
    // Check for "icon" param
    if (!empty($params['delete_icon'])) {
        $params['icon'] = $params['delete_icon'];
    }
    if (empty($params['icon'])) {
        $params['icon'] = 'trash';
    }
    if (!isset($params['action']) && !isset($params['delete_action'])) {
        if (!isset($params['item_id'])) {
            return jrCore_smarty_missing_error('item_id');
        }
        if (!jrCore_checktype($params['item_id'], 'number_nz')) {
            return jrCore_smarty_invalid_error('item_id');
        }
    }
    if (!isset($params['profile_id'])) {
        return jrCore_smarty_missing_error('profile_id');
    }
    if (!jrCore_checktype($params['profile_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('profile_id');
    }

    // Check for group requirement
    if (!isset($params['group'])) {
        $params['group'] = 'owner';
    }
    switch ($params['group']) {
        case 'master':
            if (!jrUser_is_master()) {
                return '';
            }
            break;
        case 'admin':
            if (!jrUser_is_admin()) {
                return '';
            }
            break;
        case 'owner':
            if (!jrProfile_is_profile_owner($params['profile_id'])) {
                return '';
            }
            break;
    }

    // Bring in language strings
    $_lang = jrUser_load_lang_strings();

    // See if we are using the default view
    $def = 'delete';
    if (!empty($params['view'])) {
        $def = trim($params['view']);
    }

    // Button ID
    $bid = "{$params['module']}_delete_{$params['item_id']}";
    if (isset($params['id'])) {
        $bid = $params['id'];
    }

    // Action URL
    $url = jrCore_get_module_url($params['module']);
    if (isset($params['delete_action'])) {
        $url = "{$_conf['jrCore_base_url']}/{$params['delete_action']}";
    }
    elseif (isset($params['action'])) {
        $url = "{$_conf['jrCore_base_url']}/{$params['action']}";
    }
    else {
        $url = "{$_conf['jrCore_base_url']}/{$url}/{$def}/id={$params['item_id']}";
    }

    // Check for onclick prompt
    $ptx = $_lang['jrCore'][40];
    if (isset($params['prompt'])) {
        if ($params['prompt'] === false) {
            // do not show a prompt
            $ptx = false;
        }
        elseif (is_numeric($params['prompt']) && isset($_lang["{$params['module']}"]["{$params['prompt']}"])) {
            $ptx = $_lang["{$params['module']}"]["{$params['prompt']}"];
        }
        else {
            $ptx = $params['prompt'];
        }
    }

    $onc = false;
    if (!empty($params['onclick'])) {
        $onc = $params['onclick'];
    }

    // If we have a prompt, check for custom onclick
    if ($ptx) {
        if ($onc) {
            // We have a custom onclick - wrap it
            $onc = "jrCore_confirm('', '" . addslashes($ptx) . "', function() { {$onc} })";
            $url = false;
        }
        else {
            $onc = "jrCore_confirm('', '" . addslashes($ptx) . "', function() { jrCore_window_location('{$url}') })";
            $url = false;
        }
    }
    else {
        if (!$onc) {
            $onc = "jrCore_window_location('{$url}')";
            $url = false;
        }
    }

    if (isset($params['delete_title'])) {
        $params['title'] = $params['delete_title'];
    }
    elseif (isset($params['delete_alt'])) {
        $params['title'] = $params['delete_alt'];
    }
    elseif (isset($params['alt']) && jrCore_checktype($params['alt'], 'number_nz') && isset($_lang["{$params['module']}"]["{$params['alt']}"])) {
        $params['title'] = $_lang["{$params['module']}"]["{$params['alt']}"];
    }
    if (empty($params['alt'])) {
        $params['title'] = $_lang['jrCore'][38];
    }

    $siz = null;
    if (isset($params['size']) && jrCore_checktype($params['size'], 'number_nz')) {
        $siz = (int) $params['size'];
    }
    $cls = null;
    if (isset($params['class']) && strlen($params['class']) > 0) {
        $cls = $params['class'];
    }
    $clr = null;
    if (isset($params['color']) && strlen($params['color']) === 6 && jrCore_checktype($params['color'], 'hex')) {
        $clr = $params['color'];
    }

    $_tmp = array(
        'id'      => $bid,
        'url'     => $url,
        'onclick' => $onc,
        'icon'    => jrCore_get_icon_html($params['icon'], $siz, $cls, $clr),
        'title'   => jrCore_entity_string($params['title']),
        'params'  => $params
    );
    $out  = jrCore_parse_template('item_action_button.tpl', $_tmp, 'jrCore');

    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Create an array within a template
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_item_order_button($params, $smarty)
{
    global $_conf;
    if (!jrUser_is_logged_in()) {
        return '';
    }
    if (empty($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (!jrCore_module_is_active($params['module'])) {
        return '';
    }
    if (!isset($params['profile_id'])) {
        return jrCore_smarty_missing_error('profile_id');
    }
    if (!jrCore_checktype($params['profile_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('profile_id');
    }
    if ($_pr = jrCore_get_flag('jrprofile_active_profile_data')) {
        if (!isset($_pr["profile_{$params['module']}_item_count"]) || $_pr["profile_{$params['module']}_item_count"] === 0) {
            // No items = no ordering
            return '';
        }
    }

    // Check for group requirement
    if (!isset($params['group'])) {
        $params['group'] = 'owner';
    }
    switch ($params['group']) {
        case 'master':
            if (!jrUser_is_master()) {
                return '';
            }
            break;
        case 'admin':
            if (!jrUser_is_admin()) {
                return '';
            }
            break;
        case 'owner':
            if (!jrProfile_is_profile_owner($params['profile_id'])) {
                return '';
            }
            break;
    }

    // See if this user has access to perform this action on this profile
    if (!isset($params['icon'])) {
        $params['icon'] = 'refresh';
    }
    $_ln = jrUser_load_lang_strings();
    $siz = null;
    if (isset($params['size']) && jrCore_checktype($params['size'], 'number_nz')) {
        $siz = (int) $params['size'];
    }
    $cls = null;
    if (isset($params['class']) && strlen($params['class']) > 0) {
        $cls = $params['class'];
    }
    $clr = null;
    if (isset($params['color']) && strlen($params['color']) === 6 && jrCore_checktype($params['color'], 'hex')) {
        $clr = $params['color'];
    }
    $out = '<a href="' . $_conf['jrCore_base_url'] . '/' . jrCore_get_module_url($params['module']) . '/item_display_order" title="' . $_ln['jrCore'][83] . '">' . jrCore_get_icon_html($params['icon'], $siz, $cls, $clr) . '</a>';

    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Create an array within a template
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_array($params, $smarty)
{
    if (!isset($params['name']) || strlen($params['name']) === 0) {
        return jrCore_smarty_missing_error('name');
    }
    // check for our separator
    if (isset($params['explode']) && (!isset($params['separator']) || strlen($params['separator']) === 0)) {
        $params['separator'] = ',';
    }
    // See if we have a comma and our explode value
    $_tmp = array();
    if (isset($params['explode'])) {
        if (isset($params['key']) && strlen($params['key']) > 0) {
            $_tmp["{$params['key']}"] = explode($params['separator'], $params['value']);
        }
        else {
            $_tmp = explode($params['separator'], $params['value']);
        }
    }
    else {
        if (isset($params['key']) && strlen($params['key']) > 0) {
            $_tmp["{$params['key']}"] = $params['value'];
        }
        else {
            $_tmp = $params['value'];
        }
    }
    // see if we already exists - if so we need to append
    if (is_array($smarty->getTemplateVars($params['name']))) {
        $smarty->append($params['name'], $_tmp, true);
    }
    else {
        $smarty->assign($params['name'], $_tmp);
    }
    return '';
}

/**
 * Load a remote URL into a template variable
 * @param array $params parameters for function
 * @param object $smarty Smarty object
 * @return string
 */
function smarty_function_jrCore_load_url($params, $smarty)
{
    $out = '';
    if (jrCore_checktype($params['url'], 'url')) {
        $out = jrCore_load_url($params['url']);
    }
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Set a page title from a Template
 * @param array $params function params
 * @param object $smarty Smarty Object
 * @return string
 */
function smarty_function_jrCore_page_title($params, $smarty)
{
    if (!empty($params['title'])) {
        $overwrite = (isset($params['overwrite'])) ? $params['overwrite'] : true;
        jrCore_page_title($params['title'], $overwrite);
    }
    // return "<!-- :__page_title:{$params['title']}: -->";
    return '';
}

/**
 * Include a Jamroom Template within another template
 * @param array $params function params
 * @param object $smarty Smarty Object
 * @return string
 */
function smarty_function_jrCore_include($params, $smarty)
{
    $dir = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    if (isset($params['module'])) {
        $dir = $params['module'];
    }
    elseif (isset($params['skin'])) {
        $dir = $params['skin'];
    }
    if (!isset($params['template'])) {
        return jrCore_smarty_missing_error('template');
    }
    if (strpos(' ' . $params['template'], '/')) {
        $params['template'] = basename($params['template']);
    }
    $_rp = $smarty->getTemplateVars();
    foreach ($params as $k => $v) {
        $_rp[$k] = $v;
    }
    $chk = false;
    if (isset($params['disable_override'])) {
        $chk = $params['disable_override'];
    }
    $out = jrCore_parse_template($params['template'], $_rp, $dir, $chk);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Get DataStore prefix for a given module and save to template variable
 * @param array $params function params
 * @param object $smarty Smarty Object
 * @return string
 */
function smarty_function_jrCore_get_datastore_prefix($params, $smarty)
{
    if (empty($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    $out = jrCore_db_get_prefix($params['module']);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Random number generator
 * @param array $params function params
 * @param object $smarty Smarty Object
 * @return string
 */
function smarty_function_jrCore_random_number($params, $smarty)
{
    if (!isset($params['assign'])) {
        return jrCore_smarty_missing_error('assign');
    }
    if (!isset($params['min'])) {
        $params['min'] = 0;
    }
    if (!isset($params['max'])) {
        $params['max'] = 10;
    }
    $tmp = mt_rand($params['min'], $params['max']);
    $smarty->assign($params['assign'], $tmp);
    return '';
}

/**
 * Get a jrCore SVG Icon
 * @param array $params function params
 * @param object $smarty Smarty Object
 * @return string
 */
function smarty_function_jrCore_svg_icon($params, $smarty)
{
    if (!empty($params['skin'])) {
        $params['module'] = $params['skin'];
    }
    if (empty($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (empty($params['icon'])) {
        return jrCore_smarty_missing_error('icon');
    }
    if (isset($params['size']) && jrCore_checktype($params['size'], 'number_nz')) {
        $size = (int) $params['size'];
    }
    else {
        $size = jrCore_get_skin_icon_size();
    }
    if (!strpos($params['icon'], '.svg')) {
        $params['icon'] = "{$params['icon']}.svg";
    }
    if (!isset($params['class'])) {
        $params['class'] = 'svg_icon';
    }
    if (!isset($params['version'])) {
        $params['version'] = 0;
    }
    $icon = jrCore_get_cached_svg($params['module'], $params['icon'], $size, $params['class'], $params['version']);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $icon);
        return '';
    }
    return $icon;
}

/**
 * Create an Icon in a template
 * @param array $params function params
 * @param object $smarty Smarty Object
 * @return string
 */
function smarty_function_jrCore_icon($params, $smarty)
{
    global $_conf;
    if (empty($params['icon'])) {
        return jrCore_smarty_missing_error('icon');
    }
    if (isset($params['size']) && jrCore_checktype($params['size'], 'number_nz') && $params['size'] < 65) {
        $size = (int) $params['size'];
    }
    else {
        $size = jrCore_get_skin_icon_size();
    }
    // Get the color our skin is requesting
    $color = 'white';
    if (isset($params['color']) && strlen($params['color']) === 6 && jrCore_checktype($params['color'], 'hex')) {
        $color = $params['color'];
    }
    else {
        $_tmp = jrCore_get_registered_module_features('jrCore', 'icon_color');
        if (isset($_tmp["{$_conf['jrCore_active_skin']}"])) {
            $color = array_keys($_tmp["{$_conf['jrCore_active_skin']}"]);
            $color = reset($color);
        }
    }
    $id = null;
    if (isset($params['id'])) {
        $id = $params['id'];
    }
    $cls = null;
    if (isset($params['class'])) {
        $cls = $params['class'];
    }
    $out = jrCore_get_icon_html($params['icon'], $size, $cls, $color, $id);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Get HTML for a Sprite Icon
 * @param $icon string Icon
 * @param $size int Icon size in pixels (square)
 * @param $class string additional icon class
 * @param $color string Icon color
 * @param $id string Icon unique ID
 * @return string
 */
function jrCore_get_icon_html($icon, $size = null, $class = null, $color = null, $id = null)
{
    // See if our skin has registered an icon size
    if (is_null($size)) {
        $size = jrCore_get_active_skin_icon_size();
    }
    if (is_null($color)) {
        $color = jrCore_get_active_skin_icon_color();
    }
    $_data = array(
        'icon'  => $icon,
        'size'  => $size,
        'class' => $class,
        'color' => $color,
        'id'    => $id
    );
    $_data = jrCore_trigger_event('jrCore', 'get_icon_html', $_data);
    if (!empty($_data['html'])) {
        return $_data['html'];
    }
    return jrCore_get_sprite_html($icon, $size, $class, $color, $id);
}

/**
 * Get the active skin icon size
 * @return int
 */
function jrCore_get_active_skin_icon_size()
{
    $size = 64;
    $skin = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    $_tmp = jrCore_get_registered_module_features('jrCore', 'icon_size');
    if (isset($_tmp[$skin])) {
        $size = array_keys($_tmp[$skin]);
        $size = (int) reset($size);
        if ($size > 64) {
            $size = 64;
        }
    }
    return $size;
}

/**
 * Get the active skin icon size
 * @return int
 */
function jrCore_get_active_skin_icon_color()
{
    $color = 'white';
    $skin  = jrCore_get_config_value('jrCore', 'active_skin', 'jrElastic2');
    $_tmp  = jrCore_get_registered_module_features('jrCore', 'icon_color');
    if (isset($_tmp[$skin])) {
        $color = array_keys($_tmp[$skin]);
        $color = reset($color);
    }
    if (!$color) {
        $color = 'white';
    }
    return $color;
}

/**
 * Site Statistics for modules that have registered
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrCore_stats($params, $smarty)
{
    if (empty($params['template'])) {
        return jrCore_smarty_missing_error('template');
    }

    // Get total DS row counts
    $_rt = jrCore_db_get_all_ds_row_counts();

    $_st = array();
    if ($_rt && is_array($_rt)) {

        $_ln = jrUser_load_lang_strings();
        $_st = array(
            '_stats' => array(
                $_ln['jrProfile'][26] => array(
                    'count'  => $_rt['jrProfile'],
                    'module' => 'jrProfile'
                ),
                $_ln['jrUser'][128]   => array(
                    'count'  => $_rt['jrUser'],
                    'module' => 'jrUser'
                ),
            )
        );

        if ($_tmp = jrCore_get_registered_module_features('jrProfile', 'profile_stats')) {
            foreach ($_tmp as $mod => $_stats) {
                foreach ($_stats as $key => $title) {
                    if (is_numeric($title) && isset($_ln[$mod][$title])) {
                        $title = $_ln[$mod][$title];
                    }
                    // See if we have been given a function
                    $count = false;
                    if (function_exists($key)) {
                        $count = $key();
                    }
                    if (isset($_rt[$mod])) {
                        $count = $_rt[$mod];
                    }
                    if ($count) {
                        $_st['_stats'][$title] = array(
                            'count'  => $count,
                            'module' => $mod
                        );
                    }
                }
            }
        }
    }

    $out = '';
    if (isset($_st['_stats']) && is_array($_st['_stats'])) {
        $out = jrCore_parse_template($params['template'], $_st, 'jrProfile');
    }
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Get DataStore item into an array
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrCore_db_get_item($params, $smarty)
{
    if (!isset($params['module']) || !jrCore_module_is_active($params['module'])) {
        return '';
    }
    if (!isset($params['item_id'])) {
        return jrCore_smarty_missing_error('item_id');
    }
    if (!jrCore_checktype($params['item_id'], 'number_nz')) {
        return jrCore_smarty_invalid_error('item_id');
    }
    if (!isset($params['assign'])) {
        return jrCore_smarty_missing_error('assign');
    }
    $st = false;
    if (isset($params['skip_triggers']) && jrCore_checktype($params['skip_triggers'], 'is_true')) {
        $st = true;
    }
    $smarty->assign($params['assign'], jrCore_db_get_item($params['module'], $params['item_id'], $st));
    return '';
}

/**
 * Get the MEDIA URL to a profile
 * @param $params array function parameters
 * @param $smarty object Smarty object
 * @return bool|string
 */
function smarty_function_jrCore_get_media_url($params, $smarty)
{
    if (!isset($params['profile_id'])) {
        return jrCore_smarty_missing_error('profile_id');
    }
    if (!jrCore_checktype($params['profile_id'], 'number_nn')) {
        return jrCore_smarty_invalid_error('profile_id');
    }
    $out = jrCore_get_media_url($params['profile_id']);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Show a pending item notice in a template
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrCore_pending_notice($params, $smarty)
{
    if (!isset($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (!jrCore_module_is_active($params['module'])) {
        return '';
    }
    if (!isset($params['item'])) {
        return jrCore_smarty_missing_error('item');
    }
    if (!is_array($params['item'])) {
        return jrCore_smarty_invalid_error('item');
    }
    $out = jrCore_show_pending_notice($params['module'], $params['item'], true);
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Show an Editor Form Field in a template
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrCore_editor_field($params, $smarty)
{
    global $_mods, $_user;
    $out = '';
    $tmp = jrCore_get_flag('jrcore_editor_js_included');
    if (!$tmp) {
        $url = jrCore_get_module_url('jrCore');
        $out .= '<script type="text/javascript" src="' . jrCore_get_base_url() . "/{$url}/js/{$url}/contrib/tinymce/tinymce.min.js?v=" . $_mods['jrCore']['module_updated'] . "\"></script>\n";
        jrCore_set_flag('jrcore_editor_js_included', 1);
    }
    // Initialize fields
    $_rp          = array(
        'field_name'     => $params['name'],
        'form_editor_id' => 'e' . $params['name'],
        'theme'          => 'silver'
    );
    $allowed_tags = explode(',', $_user['quota_jrCore_allowed_tags']);
    foreach ($allowed_tags as $tag) {
        $tag       = trim($tag);
        $_rp[$tag] = true;
    }

    // See what modules are providing
    if (!$tmp) {
        $_tm = jrCore_get_registered_module_features('jrCore', 'editor_button');
        if ($_tm && is_array($_tm)) {
            $crl = jrCore_get_module_url('jrCore');
            foreach ($_tm as $mod => $_items) {
                // Make sure the user is allowed Quota access
                if (jrCore_module_is_active($mod) && isset($_user["quota_{$mod}_allowed"]) && $_user["quota_{$mod}_allowed"] == 'on') {
                    // Does this HTML element have a plugin?
                    if (is_file(APP_DIR . "/modules/{$mod}/tinymce/plugin.min.js")) {
                        $url = jrCore_get_module_url($mod);
                        $out .= '<script type="text/javascript" src="' . jrCore_get_base_url() . "/{$crl}/js/{$url}/tinymce/plugin.min.js?v=" . $_mods[$mod]['module_updated'] . "\"></script>\n";
                    }
                    $tag       = trim(strtolower($mod));
                    $_rp[$tag] = true;
                }
            }
        }
    }

    $ini = @jrCore_parse_template('form_editor.tpl', $_rp, 'jrCore');
    $out .= '<script type="text/javascript">$(document).ready(function(){' . $ini . '})</script>';
    $out .= '<div class="form_editor_holder"><textarea cols="6" rows="40" id="e' . $params['name'] . '" class="form_textarea form_editor" name="' . $params['name'] . '" tabindex="1">';

    $val = '';
    if (isset($params['value']) && strlen($params['value']) > 0) {
        $val = $params['value'];
    }
    elseif (isset($params['default']) && strlen($params['default']) > 0) {
        $val = $params['default'];
    }
    $out .= $val . '</textarea><input type="hidden" id="' . $params['name'] . '_editor_contents" name="' . $params['name'] . '_editor_contents" value=""></div>';
    if (!empty($params['assign'])) {
        $smarty->assign($params['assign'], $out);
        return '';
    }
    return $out;
}

/**
 * Add 'Powered by Jamroom' link
 * @param $params array parameters for function
 * @param $smarty object Smarty object
 * @return string
 */
function smarty_function_jrCore_powered_by($params, $smarty)
{
    $_op = array(
        '11/jamroom-social-media-cms'               => 'social media CMS',
        '12/jamroom-social-media-script'            => 'social media script',
        '13/jamroom-the-social-network-cms'         => 'social network CMS',
        '14/jamroom-the-social-networking-cms'      => 'social networking CMS',
        '15/powerful-social-network-software'       => 'social network software',
        '16/powerful-social-networking-software'    => 'social networking software',
        '17/the-powerful-social-network-script'     => 'social network script',
        '18/the-powerful-social-networking-script'  => 'social networking script',
        '19/jamroom-social-media-software'          => 'social media software',
        '20/jamroom-powerful-social-media-platform' => 'social media platform'
    );
    $url = array_rand($_op);
    return '<span style="font-size:9px;"><a href="https://www.jamroom.net/the-jamroom-network/about/' . $url . '">' . $_op[$url] . '</a> | Powered by <a href="https://www.jamroom.net">Jamroom</a></span>';
}

/**
 * Adds a button to click to attach a file to a forum post (or other)
 * @param $params array Smarty function params
 * @param $smarty object Smarty Object
 * @return string
 */
function smarty_function_jrCore_upload_button($params, $smarty)
{
    if (!jrUser_is_logged_in()) {
        return false;
    }

    if (!isset($params['field'])) {
        return jrCore_smarty_missing_error('field');
    }

    if (!isset($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }

    if (!isset($params['allowed'])) {
        return jrCore_smarty_missing_error('allowed');
    }

    $pfx = jrCore_db_get_prefix($params['module']);
    if (strpos($params['field'], $pfx) !== 0) {
        return jrCore_smarty_invalid_error('field'); // the field must have the modules prefix.
    }

    $_ln = jrUser_load_lang_strings();

    // For multiple uploads we can have:
    // false - multiple uploads NOT allowed (single)
    // true - unlimited uploads
    // (int) - number of allowed uploads
    $maxup    = 1;
    $multiple = 'false';
    if (!empty($params['multiple'])) {
        if (jrCore_checktype($params['multiple'], 'number_nz')) {
            $maxup    = (int) $params['multiple'];
            $multiple = 'true';
        }
        elseif (jrCore_checktype($params['multiple'], 'is_true')) {
            $maxup    = 0;
            $multiple = 'true';
        }
    }

    // Max allowed upload size
    if (isset($params['maxsize']) && jrCore_checktype($params['maxsize'], 'number_nz')) {
        $size = (int) $params['maxsize'];
    }
    else {
        $size = jrUser_get_profile_home_key('quota_jrCore_max_upload_size');
    }

    // Allowed Extensions
    $ext = array();
    foreach (explode(',', $params['allowed']) as $e) {
        if (strlen($e) > 0) {
            $e       = trim($e);
            $ext[$e] = ".{$e}";
        }
    }
    $ext = implode(',', $ext);
    if (strpos(' ' . $ext, 'jp')) {
        if (jrCore_is_heic_supported()) {
            $ext .= ",.heic,.heif";
        }
    }

    // Since our extensions and size limit are plain to see in the source, we need
    // to make sure they don't get tampered with when coming from the client
    $md5 = jrCore_get_config_value('jrCore', 'unique_string', '');
    $md5 = md5($md5 . $ext . $size);

    if (!$_rep = jrCore_get_flag('jrcore_page_elements')) {
        $_rep = array();
    }
    $_rep['upload_token'] = jrCore_form_token_create($params['module']);

    if (isset($params['token']) && jrCore_checktype($params['token'], 'md5')) {
        $_rep['upload_token'] = $params['token'];
    }
    $_rep['module']   = $params['module'];
    $_rep['field']    = $params['field'];
    $_rep['multiple'] = $multiple;
    $_rep['maxup']    = $maxup;
    $_rep['allowed']  = $ext;
    $_rep['size']     = $size;
    $_rep['md5']      = $md5;
    $_rep['resize']   = 'false';
    if (!empty($params['resize']) && strpos($params['resize'], 'x')) {
        list($w, $h) = explode('x', $params['resize']);
        $w              = (int) $w;
        $h              = (int) $h;
        $_rep['resize'] = "{ width: {$w}, height: {$h} }";
    }
    $_rep['upload_text'] = (isset($params['upload_text'])) ? $params['upload_text'] : $_ln['jrCore'][89];
    $_rep['cancel_text'] = (isset($params['cancel_text'])) ? $params['cancel_text'] : $_ln['jrCore'][2];
    $_rep['oncomplete']  = (isset($params['oncomplete'])) ? $params['oncomplete'] : '';
    $_rep['icon']        = (isset($params['icon'])) ? $params['icon'] : false;

    return jrCore_parse_template('upload_button.tpl', $_rep, 'jrCore');
}

/**
 * Display any attached files for an item
 * @param $params array Smarty function params
 * @param $smarty object Smarty Object
 * @return string
 */
function smarty_function_jrCore_get_uploaded_files($params, $smarty)
{
    if (!isset($params['field'])) {
        return jrCore_smarty_missing_error('field');
    }
    if (!isset($params['item'])) {
        return jrCore_smarty_missing_error('item');
    }
    if (!is_array($params['item'])) {
        return jrCore_smarty_invalid_error('item');
    }
    if (!isset($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    $params['upload_module']       = $params['module'];
    $params['item']['attachments'] = array();

    foreach ($params['item'] as $k => $v) {
        if (strpos($k, "{$params['field']}_") === 0 && strpos($k, '_extension') && strlen($v) > 0) {

            // tracker_file_7_extension
            $field = str_replace('_extension', '', $k);
            // tracker_file_7
            $index                                 = (int) substr($field, strlen("{$params['field']}_"));
            $params['item']['attachments'][$index] = array(
                'idx'       => $index,
                'field'     => $field,
                'name'      => $params['item']["{$field}_name"],
                'size'      => $params['item']["{$field}_size"],
                'time'      => $params['item']["{$field}_time"],
                'extension' => $params['item']["{$field}_extension"]
            );
            if (jrImage_is_image_file($params['item']["{$field}_name"])) {
                $params['item']['attachments'][$index]['type'] = 'image';
            }
            else {
                $params['item']['attachments'][$index]['type'] = 'file';
            }
        }
    }
    if (count($params['item']['attachments']) > 0) {
        ksort($params['item']['attachments'], SORT_NUMERIC);
        return jrCore_parse_template('upload_attachments.tpl', $params, 'jrCore');
    }
    return '';
}

/**
 * Get the icon HTML for a module
 * @param $params array
 * @param $smarty object
 * @return string
 */
function smarty_function_jrCore_get_module_icon_html($params, $smarty)
{
    if (!isset($params['module'])) {
        return jrCore_smarty_missing_error('module');
    }
    if (!isset($params['size'])) {
        return jrCore_smarty_missing_error('size');
    }
    $class = null;
    if (isset($params['class'])) {
        $class = $params['class'];
    }
    return jrCore_get_module_icon_html($params['module'], $params['size'], $class);
}

/**
 * Get the icon HTML for a skin
 * @param $params array
 * @param $smarty object
 * @return string
 */
function smarty_function_jrCore_get_skin_icon_html($params, $smarty)
{
    if (!isset($params['skin'])) {
        return jrCore_smarty_missing_error('skin');
    }
    if (!isset($params['size'])) {
        return jrCore_smarty_missing_error('size');
    }
    $class = null;
    if (isset($params['class'])) {
        $class = $params['class'];
    }
    return jrCore_get_skin_icon_html($params['skin'], $params['size'], $class);
}

/**
 * template fdebug function
 * @param array $params
 * @param object $smarty
 * @return string
 */
function smarty_function_fdebug($params, $smarty)
{
    fdebug($params); // OK
    return '';
}

/**
 * returns the next tab index for a page
 * @param array $params
 * @param object $smarty
 * @return int|mixed
 */
function smarty_function_jrCore_next_tabindex($params, $smarty)
{
    $index = jrCore_get_flag('jr_form_tab_index');
    if (!$index) {
        $index = 0;
    }
    $index++;
    jrCore_set_flag('jr_form_tab_index', $index);
    return $index;
}
