<?php
 /**
 * Jamroom Favicon Creator module
 *
 * copyright 2024 The Jamroom Network
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0.  Please see the included "license.html" file.
 *
 * This module may include works that are not developed by
 * The Jamroom Network
 * and are used under license - any licenses are included and
 * can be found in the "contrib" directory within this module.
 *
 * Jamroom may use modules and skins that are licensed by third party
 * developers, and licensed under a different license  - please
 * reference the individual module or skin license that is included
 * with your installation.
 *
 * This software is provided "as is" and any express or implied
 * warranties, including, but not limited to, the implied warranties
 * of merchantability and fitness for a particular purpose are
 * disclaimed.  In no event shall the Jamroom Network be liable for
 * any direct, indirect, incidental, special, exemplary or
 * consequential damages (including but not limited to, procurement
 * of substitute goods or services; loss of use, data or profits;
 * or business interruption) however caused and on any theory of
 * liability, whether in contract, strict liability, or tort
 * (including negligence or otherwise) arising from the use of this
 * software, even if advised of the possibility of such damage.
 * Some jurisdictions may not allow disclaimers of implied warranties
 * and certain statements in the above disclaimer may not apply to
 * you as regards implied warranties; the other terms and conditions
 * remain enforceable notwithstanding. In some jurisdictions it is
 * not permitted to limit liability and therefore such limitations
 * may not apply to you.
 *
 * @copyright 2012 Talldude Networks, LLC.
 */

// make sure we are not being called directly
defined('APP_DIR') or exit();

/**
 * meta
 */
function jrFavicon_meta()
{
    return array(
        'name'        => 'Favicon Creator',
        'url'         => 'favicon',
        'version'     => '1.2.0',
        'developer'   => 'The Jamroom Network, &copy;' . date('Y'),
        'description' => 'Create a site Favicon that supports multiple resolutions including High DPI displays',
        'doc_url'     => 'https://www.jamroom.net/the-jamroom-network/documentation/modules/2954/favicon-creator',
        'requires'    => 'jrCore:6.5.0b6',
        'license'     => 'mpl',
        'category'    => 'site'
    );
}

/**
 * init
 */
function jrFavicon_init()
{
    jrCore_register_module_feature('jrCore', 'tool_view', 'jrFavicon', 'browse', array('favicon browser', 'Browse, Create and Upload a new Favicon for your site'));
    jrCore_register_module_feature('jrCore', 'admin_tab', 'jrFavicon', 'browse', 'Favicon Browser');

    jrCore_register_event_listener('jrCore', 'view_results', 'jrFavicon_view_results_listener');
    return true;
}

//---------------------
// EVENT LISTENERS
//---------------------

/**
 * Add favicon HTML to pages
 * @param string $_data HTML
 * @param array $_user current user info
 * @param array $_conf Global config
 * @param array $_args additional info about the module
 * @param string $event Event Trigger name
 * @return string
 */
function jrFavicon_view_results_listener($_data, $_user, $_conf, $_args, $event)
{
    if (jrCore_get_config_value('jrFavicon', 'enabled', 'on') == 'on') {
        if ($aid = jrCore_get_config_value('jrFavicon', 'active', false)) {
            // Only add to HTML if we have a HEAD section on the page
            if (strpos($_data, '</head>')) {
                $key = "favicon_head_html_{$aid}";
                if (!$html = jrCore_is_cached('jrFavicon', $key, false, false)) {
                    $_temp = array(
                        'item_id'   => $aid,
                        '_icons'    => array(),
                        'media_url' => jrCore_get_media_url(0)
                    );
                    $dir   = jrCore_get_media_directory(0, FORCE_LOCAL);
                    foreach (jrFavicon_get_available_icons() as $vendor => $_sizes) {
                        foreach ($_sizes as $size) {
                            // Use NEW URL format
                            $url = "/favicon_{$size}.png?v={$aid}";
                            if (is_file("{$dir}/favicon_{$aid}_{$size}.png")) {
                                // This site has NOT updated yet - use existing active one
                                $url = "/favicon_{$aid}_{$size}.png";
                            }
                            $_temp['_icons'][] = array(
                                'url'    => jrCore_get_media_url(0) . $url,
                                'vendor' => $vendor,
                                'size'   => $size
                            );
                        }
                    }
                    $html = jrCore_parse_template('favicon.tpl', $_temp, 'jrFavicon');
                    jrCore_add_to_cache('jrFavicon', $key, trim($html), 0, 0, false, false);
                }
                if (!empty($html)) {
                    $_data = str_replace('</title>', "</title>\n" . $html, $_data);
                }
            }
        }
    }
    return $_data;
}

//---------------------
// FUNCTIONS
//---------------------

/**
 * Get required icons and sizes
 * @return array
 */
function jrFavicon_get_available_icons()
{
    return array(
        'apple-touch-icon'             => array(57, 72, 114, 120, 144, 152),
        'apple-touch-icon-precomposed' => array(57, 72, 114, 120, 144, 152),
        'icon'                         => array(96, 195, 228),
        'msapplication-TileImage'      => array(144)
    );
}

/**
 * Get sizes of images we are going to create favicons of
 * @see https://github.com/audreyr/favicon-cheat-sheet
 * @return array
 */
function jrFavicon_get_image_sizes()
{
    return array(57, 72, 96, 114, 120, 144, 152, 195, 228);
}

/**
 * Create the Favicon images from a given Favicon DS item
 * @param array $_item DS Item
 * @return bool
 */
function jrFavicon_create_favicon_images($_item)
{
    $fid = (int) $_item['_item_id'];
    $pid = (int) $_item['_profile_id'];
    $cdr = jrCore_get_module_cache_dir('jrFavicon');
    $nam = "jrFavicon_{$fid}_icon_image.png";
    if (jrCore_confirm_media_file_is_local($pid, $nam, "{$cdr}/{$nam}", true)) {
        // Create Favicon with 4 sizes
        require_once APP_DIR . '/modules/jrFavicon/contrib/php-ico/class-php-ico.php';
        $out = "{$cdr}/favicon_{$fid}.ico";
        $ico = new PHP_ICO("{$cdr}/{$nam}", array(array(16, 16), array(24, 24), array(32, 32), array(48, 48), array(64, 64)));
        $ico->save_ico($out);
        if (is_file($out)) {
            // Create main ICO file
            jrCore_write_media_file(0, 'favicon.ico', $out, 'public-read');
        }
        foreach (jrFavicon_get_image_sizes() as $size) {
            // Specific device images
            jrImage_resize_image("{$cdr}/{$nam}", "{$cdr}/favicon_{$fid}_{$size}.png", $size);
            if (is_file("{$cdr}/favicon_{$fid}_{$size}.png")) {
                jrCore_write_media_file(0, "favicon_{$size}.png", "{$cdr}/favicon_{$fid}_{$size}.png", 'public-read');
                unlink("{$cdr}/favicon_{$fid}_{$size}.png");
            }
        }

        // Symlink on all front ends if we are on a cluster
        if (jrCore_module_is_active('jrCloudClient')) {
            $_data = array('time' => time());
            jrCloudClient_run_server_role_function('app', 'jrFavicon_create_favicon_symlinks', $_data);
        }
        return true;
    }
    return false;
}

/**
 * Create root directory symlinks for common favicon images
 * @return bool
 */
function jrFavicon_create_favicon_symlinks()
{
    if (is_writable(APP_DIR)) {
        // Symlink some images into the web root
        $_sy = array(
            'favicon.ico'                              => 'favicon.ico',
            'apple-touch-icon.png'                     => 'favicon_120.png',
            'apple-touch-icon-120x120.png'             => 'favicon_120.png',
            'apple-touch-icon-precomposed.png'         => 'favicon_120.png',
            'apple-touch-icon-120x120-precomposed.png' => 'favicon_120.png'
        );
        $old = getcwd();
        chdir(APP_DIR);
        foreach ($_sy as $name => $icon) {
            // @see https://en.wikipedia.org/wiki/Favicon
            if (!is_file(APP_DIR . "/data/media/0/0/{$icon}")) {
                continue;
            }
            if (is_file(APP_DIR . "/{$name}") || is_link(APP_DIR . "/{$name}")) {
                @unlink(APP_DIR . "/{$name}");
            }
            symlink("data/media/0/0/{$icon}", $name);
        }
        chdir($old);
        return true;
    }
    return false;
}
