package overload;

our $VERSION = '1.28';

%ops = (
    with_assign         => "+ - * / % ** << >> x .",
    assign              => "+= -= *= /= %= **= <<= >>= x= .=",
    num_comparison      => "< <= >  >= == !=",
    '3way_comparison'   => "<=> cmp",
    str_comparison      => "lt le gt ge eq ne",
    binary              => '& &= | |= ^ ^= &. &.= |. |.= ^. ^.=',
    unary               => "neg ! ~ ~.",
    mutators            => '++ --',
    func                => "atan2 cos sin exp abs log sqrt int",
    conversion          => 'bool "" 0+ qr',
    iterators           => '<>',
    filetest            => "-X",
    dereferencing       => '${} @{} %{} &{} *{}',
    matching            => '~~',
    special             => 'nomethod fallback =',
);

my %ops_seen;
@ops_seen{ map split(/ /), values %ops } = ();

sub nil {}

sub OVERLOAD {
  $package = shift;
  my %arg = @_;
  my $sub;
  *{$package . "::(("} = \&nil; # Make it findable via fetchmethod.
  for (keys %arg) {
    if ($_ eq 'fallback') {
      for my $sym (*{$package . "::()"}) {
	*$sym = \&nil; # Make it findable via fetchmethod.
	$$sym = $arg{$_};
      }
    } else {
      warnings::warnif("overload arg '$_' is invalid")
        unless exists $ops_seen{$_};
      $sub = $arg{$_};
      if (not ref $sub) {
	$ {$package . "::(" . $_} = $sub;
	$sub = \&nil;
      }
      #print STDERR "Setting '$ {'package'}::\cO$_' to \\&'$sub'.\n";
      *{$package . "::(" . $_} = \&{ $sub };
    }
  }
}

sub import {
  $package = (caller())[0];
  # *{$package . "::OVERLOAD"} = \&OVERLOAD;
  shift;
  $package->overload::OVERLOAD(@_);
}

sub unimport {
  $package = (caller())[0];
  shift;
  *{$package . "::(("} = \&nil;
  for (@_) {
      warnings::warnif("overload arg '$_' is invalid")
        unless exists $ops_seen{$_};
      delete $ {$package . "::"}{$_ eq 'fallback' ? '()' : "(" .$_};
  }
}

sub Overloaded {
  my $package = shift;
  $package = ref $package if ref $package;
  mycan ($package, '()') || mycan ($package, '((');
}

sub ov_method {
  my $globref = shift;
  return undef unless $globref;
  my $sub = \&{*$globref};
  no overloading;
  return $sub if $sub != \&nil;
  return shift->can($ {*$globref});
}

sub OverloadedStringify {
  my $package = shift;
  $package = ref $package if ref $package;
  #$package->can('(""')
  ov_method mycan($package, '(""'), $package
    or ov_method mycan($package, '(0+'), $package
    or ov_method mycan($package, '(bool'), $package
    or ov_method mycan($package, '(nomethod'), $package;
}

sub Method {
  my $package = shift;
  if(ref $package) {
    local $@;
    local $!;
    require Scalar::Util;
    $package = Scalar::Util::blessed($package);
    return undef if !defined $package;
  }
  #my $meth = $package->can('(' . shift);
  ov_method mycan($package, '(' . shift), $package;
  #return $meth if $meth ne \&nil;
  #return $ {*{$meth}};
}

sub AddrRef {
  no overloading;
  "$_[0]";
}

*StrVal = *AddrRef;

sub mycan {				# Real can would leave stubs.
  my ($package, $meth) = @_;

  local $@;
  local $!;
  require mro;

  my $mro = mro::get_linear_isa($package);
  foreach my $p (@$mro) {
    my $fqmeth = $p . q{::} . $meth;
    return \*{$fqmeth} if defined &{$fqmeth};
  }

  return undef;
}

%constants = (
	      'integer'	  =>  0x1000, # HINT_NEW_INTEGER
	      'float'	  =>  0x2000, # HINT_NEW_FLOAT
	      'binary'	  =>  0x4000, # HINT_NEW_BINARY
	      'q'	  =>  0x8000, # HINT_NEW_STRING
	      'qr'	  => 0x10000, # HINT_NEW_RE
	     );

use warnings::register;
sub constant {
  # Arguments: what, sub
  while (@_) {
    if (@_ == 1) {
        warnings::warnif ("Odd number of arguments for overload::constant");
        last;
    }
    elsif (!exists $constants {$_ [0]}) {
        warnings::warnif ("'$_[0]' is not an overloadable type");
    }
    elsif (!ref $_ [1] || "$_[1]" !~ /(^|=)CODE\(0x[0-9a-f]+\)$/) {
        # Can't use C<ref $_[1] eq "CODE"> above as code references can be
        # blessed, and C<ref> would return the package the ref is blessed into.
        if (warnings::enabled) {
            $_ [1] = "undef" unless defined $_ [1];
            warnings::warn ("'$_[1]' is not a code reference");
        }
    }
    else {
        $^H{$_[0]} = $_[1];
        $^H |= $constants{$_[0]};
    }
    shift, shift;
  }
}

sub remove_constant {
  # Arguments: what, sub
  while (@_) {
    delete $^H{$_[0]};
    $^H &= ~ $constants{$_[0]};
    shift, shift;
  }
}

1;

__END__

=head1 NAME

overload - Package for overloading Perl operations

=head1 SYNOPSIS

    package SomeThing;

    use overload
	'+' => \&myadd,
	'-' => \&mysub;
	# etc
    ...

    package main;
    $a = SomeThing->new( 57 );
    $b = 5 + $a;
    ...
    if (overload::Overloaded $b) {...}
    ...
    $strval = overload::StrVal $b;

=head1 DESCRIPTION

This pragma allows overloading of Perl's operators for a class.
To overload built-in functions, see L<perlsub/Overriding Built-in Functions> instead.

=head2 Fundamentals

=head3 Declaration

Arguments of the C<use overload> directive are (key, value) pairs.
For the full set of legal keys, see L<Overloadable Operations> below.

Operator implementations (the values) can be subroutines,
references to subroutines, or anonymous subroutines
- in other words, anything legal inside a C<&{ ... }> call.
Values specified as strings are interpreted as method names.
Thus

    package Number;
    use overload
        "-" => "minus",
        "*=" => \&muas,
        '""' => sub { ...; };

declares that subtraction is to be implemented by method C<minus()>
in the class C<Number> (or one of its base classes),
and that the function C<Number::muas()> is to be used for the
assignment form of multiplication, C<*=>.
It also defines an anonymous subroutine to implement stringification:
this is called whenever an object blessed into the package C<Number>
is used in a string context (this subroutine might, for example,
return the number as a Roman numeral).

=head3 Calling Conventions and Magic Autogeneration

The following sample implementation of C<minus()> (which assumes
that C<Number> objects are simply blessed references to scalars)
illustrates the calling conventions:

    package Number;
    sub minus {
        my ($self, $other, $swap) = @_;
        my $result = $$self - $other;         # *
        $result = -$result if $swap;
        ref $result ? $result : bless \$result;
    }
    # * may recurse once - see table below

Three arguments are passed to all subroutines specified in the
C<use overload> directive (with exceptions - see below, particularly
L</nomethod>).

The first of these is the operand providing the overloaded
operator implementation -
in this case, the object whose C<minus()> method is being called.

The second argument is the other operand, or C<undef> in the
case of a unary operator.

The third argument is set to TRUE if (and only if) the two
operands have been swapped.  Perl may do this to ensure that the
first argument (C<$self>) is an object implementing the overloaded
operation, in line with general object calling conventions.
For example, if C<$x> and C<$y> are C<Number>s:

    operation   |   generates a call to
    ============|======================
    $x - $y     |   minus($x, $y, '')
    $x - 7      |   minus($x, 7, '')
    7 - $x      |   minus($x, 7, 1)

Perl may also use C<minus()> to implement other operators which
have not been specified in the C<use overload> directive,
according to the rules for L<Magic Autogeneration> described later.
For example, the C<use overload> above declared no subroutine
for any of the operators C<-->, C<neg> (the overload key for
unary minus), or C<-=>.  Thus

    operation   |   generates a call to
    ============|======================
    -$x         |   minus($x, 0, 1)
    $x--        |   minus($x, 1, undef)
    $x -= 3     |   minus($x, 3, undef)

Note the C<undef>s:
where autogeneration results in the method for a standard
operator which does not change either of its operands, such
as C<->, being used to implement an operator which changes
the operand ("mutators": here, C<--> and C<-=>),
Perl passes undef as the third argument.
This still evaluates as FALSE, consistent with the fact that
the operands have not been swapped, but gives the subroutine
a chance to alter its behaviour in these cases.

In all the above examples, C<minus()> is required
only to return the result of the subtraction:
Perl takes care of the assignment to $x.
In fact, such methods should I<not> modify their operands,
even if C<undef> is passed as the third argument
(see L<Overloadable Operations>).

The same is not true of implementations of C<++> and C<-->:
these are expected to modify their operand.
An appropriate implementation of C<--> might look like

    use overload '--' => "decr",
        # ...
    sub decr { --${$_[0]}; }

If the experimental "bitwise" feature is enabled (see L<feature>), a fifth
TRUE argument is passed to subroutines handling C<&>, C<|>, C<^> and C<~>.
This indicates that the caller is expecting numeric behaviour.  The fourth
argument will be C<undef>, as that position (C<$_[3]>) is reserved for use
by L</nomethod>.

=head3 Mathemagic, Mutators, and Copy Constructors

The term 'mathemagic' describes the overloaded implementation
of mathematical operators.
Mathemagical operations raise an issue.
Consider the code:

    $a = $b;
    --$a;

If C<$a> and C<$b> are scalars then after these statements

    $a == $b - 1

An object, however, is a reference to blessed data, so if
C<$a> and C<$b> are objects then the assignment C<$a = $b>
copies only the reference, leaving C<$a> and C<$b> referring
to the same object data.
One might therefore expect the operation C<--$a> to decrement
C<$b> as well as C<$a>.
However, this would not be consistent with how we expect the
mathematical operators to work.

Perl resolves this dilemma by transparently calling a copy
constructor before calling a method defined to implement
a mutator (C<-->, C<+=>, and so on.).
In the above example, when Perl reaches the decrement
statement, it makes a copy of the object data in C<$a> and
assigns to C<$a> a reference to the copied data.
Only then does it call C<decr()>, which alters the copied
data, leaving C<$b> unchanged.
Thus the object metaphor is preserved as far as possible,
while mathemagical operations still work according to the
arithmetic metaphor.

Note: the preceding paragraph describes what happens when
Perl autogenerates the copy constructor for an object based
on a scalar.
For other cases, see L<Copy Constructor>.

=head2 Overloadable Operations

The complete list of keys that can be specified in the C<use overload>
directive are given, separated by spaces, in the values of the
hash C<%overload::ops>:

 with_assign	  => '+ - * / % ** << >> x .',
 assign		  => '+= -= *= /= %= **= <<= >>= x= .=',
 num_comparison	  => '< <= > >= == !=',
 '3way_comparison'=> '<=> cmp',
 str_comparison	  => 'lt le gt ge eq ne',
 binary		  => '& &= | |= ^ ^= &. &.= |. |.= ^. ^.=',
 unary		  => 'neg ! ~ ~.',
 mutators	  => '++ --',
 func		  => 'atan2 cos sin exp abs log sqrt int',
 conversion	  => 'bool "" 0+ qr',
 iterators	  => '<>',
 filetest         => '-X',
 dereferencing	  => '${} @{} %{} &{} *{}',
 matching	  => '~~',
 special	  => 'nomethod fallback ='

Most of the overloadable operators map one-to-one to these keys.
Exceptions, including additional overloadable operations not
apparent from this hash, are included in the notes which follow.
This list is subject to growth over time.

A warning is issued if an attempt is made to register an operator not found
above.

=over 5

=item * C<not>

The operator C<not> is not a valid key for C<use overload>.
However, if the operator C<!> is overloaded then the same
implementation will be used for C<not>
(since the two operators differ only in precedence).

=item * C<neg>

The key C<neg> is used for unary minus to disambiguate it from
binary C<->.

=item * C<++>, C<-->

Assuming they are to behave analogously to Perl's C<++> and C<-->,
overloaded implementations of these operators are required to
mutate their operands.

No distinction is made between prefix and postfix forms of the
increment and decrement operators: these differ only in the
point at which Perl calls the associated subroutine when
evaluating an expression.

=item * I<Assignments>

    +=  -=  *=  /=  %=  **=  <<=  >>=  x=  .=
    &=  |=  ^=  &.=  |.=  ^.=

Simple assignment is not overloadable (the C<'='> key is used
for the L<Copy Constructor>).
Perl does have a way to make assignments to an object do whatever
you want, but this involves using tie(), not overload -
see L<perlfunc/tie> and the L</COOKBOOK> examples below.

The subroutine for the assignment variant of an operator is
required only to return the result of the operation.
It is permitted to change the value of its operand
(this is safe because Perl calls the copy constructor first),
but this is optional since Perl assigns the returned value to
the left-hand operand anyway.

An object that overloads an assignment operator does so only in
respect of assignments to that object.
In other words, Perl never calls the corresponding methods with
the third argument (the "swap" argument) set to TRUE.
For example, the operation

    $a *= $b

cannot lead to C<$b>'s implementation of C<*=> being called,
even if C<$a> is a scalar.
(It can, however, generate a call to C<$b>'s method for C<*>).

=item * I<Non-mutators with a mutator variant>

     +  -  *  /  %  **  <<  >>  x  .
     &  |  ^  &.  |.  ^.

As described L<above|"Calling Conventions and Magic Autogeneration">,
Perl may call methods for operators like C<+> and C<&> in the course
of implementing missing operations like C<++>, C<+=>, and C<&=>.
While these methods may detect this usage by testing the definedness
of the third argument, they should in all cases avoid changing their
operands.
This is because Perl does not call the copy constructor before
invoking these methods.

=item * C<int>

Traditionally, the Perl function C<int> rounds to 0
(see L<perlfunc/int>), and so for floating-point-like types one
should follow the same semantic.

=item * I<String, numeric, boolean, and regexp conversions>

    ""  0+  bool

These conversions are invoked according to context as necessary.
For example, the subroutine for C<'""'> (stringify) may be used
where the overloaded object is passed as an argument to C<print>,
and that for C<'bool'> where it is tested in the condition of a flow
control statement (like C<while>) or the ternary C<?:> operation.

Of course, in contexts like, for example, C<$obj + 1>, Perl will
invoke C<$obj>'s implementation of C<+> rather than (in this
example) converting C<$obj> to a number using the numify method
C<'0+'> (an exception to this is when no method has been provided
for C<'+'> and L</fallback> is set to TRUE).

The subroutines for C<'""'>, C<'0+'>, and C<'bool'> can return
any arbitrary Perl value.
If the corresponding operation for this value is overloaded too,
the operation will be called again with this value.

As a special case if the overload returns the object itself then it will
be used directly.  An overloaded conversion returning the object is
probably a bug, because you're likely to get something that looks like
C<YourPackage=HASH(0x8172b34)>.

    qr

The subroutine for C<'qr'> is used wherever the object is
interpolated into or used as a regexp, including when it
appears on the RHS of a C<=~> or C<!~> operator.

C<qr> must return a compiled regexp, or a ref to a compiled regexp
(such as C<qr//> returns), and any further overloading on the return
value will be ignored.

=item * I<Iteration>

If C<E<lt>E<gt>> is overloaded then the same implementation is used
for both the I<read-filehandle> syntax C<E<lt>$varE<gt>> and
I<globbing> syntax C<E<lt>${var}E<gt>>.

=item * I<File tests>

The key C<'-X'> is used to specify a subroutine to handle all the
filetest operators (C<-f>, C<-x>, and so on: see L<perlfunc/-X> for
the full list);
it is not possible to overload any filetest operator individually.
To distinguish them, the letter following the '-' is passed as the
second argument (that is, in the slot that for binary operators
is used to pass the second operand).

Calling an overloaded filetest operator does not affect the stat value
associated with the special filehandle C<_>.  It still refers to the
result of the last C<stat>, C<lstat> or unoverloaded filetest.

This overload was introduced in Perl 5.12.

=item * I<Matching>

The key C<"~~"> allows you to override the smart matching logic used by
the C<~~> operator and the switch construct (C<given>/C<when>).  See
L<perlsyn/Switch Statements> and L<feature>.

Unusually, the overloaded implementation of the smart match operator
does not get full control of the smart match behaviour.
In particular, in the following code:

    package Foo;
    use overload '~~' => 'match';

    my $obj =  Foo->new();
    $obj ~~ [ 1,2,3 ];

the smart match does I<not> invoke the method call like this:

    $obj->match([1,2,3],0);

rather, the smart match distributive rule takes precedence, so $obj is
smart matched against each array element in turn until a match is found,
so you may see between one and three of these calls instead:

    $obj->match(1,0);
    $obj->match(2,0);
    $obj->match(3,0);

Consult the match table in  L<perlop/"Smartmatch Operator"> for
details of when overloading is invoked.

=item * I<Dereferencing>

    ${}  @{}  %{}  &{}  *{}

If these operators are not explicitly overloaded then they
work in the normal way, yielding the underlying scalar,
array, or whatever stores the object data (or the appropriate
error message if the dereference operator doesn't match it).
Defining a catch-all C<'nomethod'> (see L<below|/nomethod>)
makes no difference to this as the catch-all function will
not be called to implement a missing dereference operator.

If a dereference operator is overloaded then it must return a
I<reference> of the appropriate type (for example, the
subroutine for key C<'${}'> should return a reference to a
scalar, not a scalar), or another object which overloads the
operator: that is, the subroutine only determines what is
dereferenced and the actual dereferencing is left to Perl.
As a special case, if the subroutine returns the object itself
then it will not be called again - avoiding infinite recursion.

=item * I<Special>

    nomethod  fallback  =

See L<Special Keys for C<use overload>>.

=back

=head2 Magic Autogeneration

If a method for an operation is not found then Perl tries to
autogenerate a substitute implementation from the operations
that have been defined.

Note: the behaviour described in this section can be disabled
by setting C<fallback> to FALSE (see L</fallback>).

In the following tables, numbers indicate priority.
For example, the table below states that,
if no implementation for C<'!'> has been defined then Perl will
implement it using C<'bool'> (that is, by inverting the value
returned by the method for C<'bool'>);
if boolean conversion is also unimplemented then Perl will
use C<'0+'> or, failing that, C<'""'>.

    operator | can be autogenerated from
             |
             | 0+   ""   bool   .   x
    =========|==========================
       0+    |       1     2
       ""    |  1          2
       bool  |  1    2
       int   |  1    2     3
       !     |  2    3     1
       qr    |  2    1     3
       .     |  2    1     3
       x     |  2    1     3
       .=    |  3    2     4    1
       x=    |  3    2     4        1
       <>    |  2    1     3
       -X    |  2    1     3

Note: The iterator (C<'E<lt>E<gt>'>) and file test (C<'-X'>)
operators work as normal: if the operand is not a blessed glob or
IO reference then it is converted to a string (using the method
for C<'""'>, C<'0+'>, or C<'bool'>) to be interpreted as a glob
or filename.

    operator | can be autogenerated from
             |
             |  <   <=>   neg   -=    -
    =========|==========================
       neg   |                        1
       -=    |                        1
       --    |                   1    2
       abs   | a1    a2    b1        b2    [*]
       <     |        1
       <=    |        1
       >     |        1
       >=    |        1
       ==    |        1
       !=    |        1

    * one from [a1, a2] and one from [b1, b2]

Just as numeric comparisons can be autogenerated from the method
for C<< '<=>' >>, string comparisons can be autogenerated from
that for C<'cmp'>:

     operators          |  can be autogenerated from
    ====================|===========================
     lt gt le ge eq ne  |  cmp

Similarly, autogeneration for keys C<'+='> and C<'++'> is analogous
to C<'-='> and C<'--'> above:

    operator | can be autogenerated from
             |
             |  +=    +
    =========|==========================
        +=   |        1
        ++   |   1    2

And other assignment variations are analogous to
C<'+='> and C<'-='> (and similar to C<'.='> and C<'x='> above):

              operator ||  *= /= %= **= <<= >>= &= ^= |= &.= ^.= |.=
    -------------------||-------------------------------------------
    autogenerated from ||  *  /  %  **  <<  >>  &  ^  |  &.  ^.  |.

Note also that the copy constructor (key C<'='>) may be
autogenerated, but only for objects based on scalars.
See L<Copy Constructor>.

=head3 Minimal Set of Overloaded Operations

Since some operations can be automatically generated from others, there is
a minimal set of operations that need to be overloaded in order to have
the complete set of overloaded operations at one's disposal.
Of course, the autogenerated operations may not do exactly what the user
expects.  The minimal set is:

    + - * / % ** << >> x
    <=> cmp
    & | ^ ~ &. |. ^. ~.
    atan2 cos sin exp log sqrt int
    "" 0+ bool
    ~~

Of the conversions, only one of string, boolean or numeric is
needed because each can be generated from either of the other two.

=head2 Special Keys for C<use overload>

=head3 C<nomethod>

The C<'nomethod'> key is used to specify a catch-all function to
be called for any operator that is not individually overloaded.
The specified function will be passed four parameters.
The first three arguments coincide with those that would have been
passed to the corresponding method if it had been defined.
The fourth argument is the C<use overload> key for that missing
method.  If the experimental "bitwise" feature is enabled (see L<feature>),
a fifth TRUE argument is passed to subroutines handling C<&>, C<|>, C<^> and C<~> to indicate that the caller is expecting numeric behaviour.

For example, if C<$a> is an object blessed into a package declaring

    use overload 'nomethod' => 'catch_all', # ...

then the operation

    3 + $a

could (unless a method is specifically declared for the key
C<'+'>) result in a call

    catch_all($a, 3, 1, '+')

See L<How Perl Chooses an Operator Implementation>.

=head3 C<fallback>

The value assigned to the key C<'fallback'> tells Perl how hard
it should try to find an alternative way to implement a missing
operator.

=over

=item * defined, but FALSE

    use overload "fallback" => 0, # ... ;

This disables L<Magic Autogeneration>.

=item * C<undef>

In the default case where no value is explicitly assigned to
C<fallback>, magic autogeneration is enabled.

=item * TRUE

The same as for C<undef>, but if a missing operator cannot be
autogenerated then, instead of issuing an error message, Perl
is allowed to revert to what it would have done for that
operator if there had been no C<use overload> directive.

Note: in most cases, particularly the L<Copy Constructor>,
this is unlikely to be appropriate behaviour.

=back

See L<How Perl Chooses an Operator Implementation>.

=head3 Copy Constructor

As mentioned L<above|"Mathemagic, Mutators, and Copy Constructors">,
this operation is called when a mutator is applied to a reference
that shares its object with some other reference.
For example, if C<$b> is mathemagical, and C<'++'> is overlo                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                                  