<?php

declare(strict_types = 1);

namespace Redaxscript;

use function array_key_exists;
use function base64_encode;
use function chunk_split;
use function current;
use function function_exists;
use function implode;
use function is_file;
use function key;
use function mail;
use function trim;
use function uniqid;

/**
 * parent class to send an mail
 *
 * @since 2.0.0
 *
 * @package Redaxscript
 * @category Mailer
 * @author Henry Ruhs
 */

class Mailer
{
	/**
	 * array of the recipient
	 *
	 * @var string[]
	 */

	protected array $_toArray = [];

	/**
	 * array of the sender
	 *
	 * @var string[]
	 */

	protected array $_fromArray = [];

	/**
	 * subject of the email
	 */

	protected string $_subject;

	/**
	 * body of the email
	 *
	 * @var string[]
	 */

	protected array $_bodyArray = [];

	/**
	 * array of the attachments
	 *
	 * @var string[]
	 */

	protected array $_attachmentArray = [];

	/**
	 * built recipient contents
	 */

	protected string $_fromString;

	/**
	 * built subject contents
	 */

	protected string $_subjectString;

	/**
	 * built body contents
	 */

	protected string $_bodyString;

	/**
	 * built header contents
	 */

	protected string $_headerString;

	/**
	 * init the class
	 *
	 * @since 5.0.0
	 *
	 * @param string[] $toArray array of the recipient
	 * @param string[] $fromArray array of the sender
	 * @param ?string $subject subject of the email
	 * @param ?string[] $bodyArray array of the body
	 * @param ?string[] $attachmentArray array of the attachments
	 */

	public function init(array $toArray = [], array $fromArray = [], ?string $subject = null, ?array $bodyArray = [], ?array $attachmentArray = []) : void
	{
		$this->_toArray = $toArray;
		$this->_fromArray = $fromArray;
		$this->_subject = $subject;
		$this->_bodyArray = $bodyArray;
		$this->_attachmentArray = $attachmentArray;

		/* create as needed */

		$this->_createFromString();
		$this->_createSubjectString();
		$this->_createBodyString();
		$this->_createHeaderString();
	}

	/**
	 * create the recipient contents
	 *
	 * @since 2.0.0
	 */

	protected function _createFromString() : void
	{
		/* create from string */

		$from = current($this->_fromArray);
		$fromName = key($this->_fromArray);

		/* from name fallback */

		if (!$fromName)
		{
			$fromName = $from;
		}
		$this->_fromString = $fromName . ' <' . $from . '>';
	}

	/**
	 * create the subject contents
	 *
	 * @since 2.0.0
	 */

	protected function _createSubjectString() : void
	{
		$settingModel = new Model\Setting();

		/* collect subject string */

		$subject = $settingModel->get('subject');

		/* extend subject string */

		if ($subject)
		{
			$this->_subjectString = $subject;
			if ($this->_subject)
			{
				$this->_subjectString .= $settingModel->get('divider');
			}
		}
		$this->_subjectString .= $this->_subject;
	}

	/**
	 * create the body contents
	 *
	 * @since 5.0.0
	 */

	protected function _createBodyString() : void
	{
		$this->_bodyString = array_key_exists(0, $this->_bodyArray) ? implode(PHP_EOL, $this->_bodyArray) : '';
	}

	/**
	 * create the header contents
	 *
	 * @since 2.0.0
	 */

	protected function _createHeaderString() : void
	{
		$settingModel = new Model\Setting();

		/* collect header string */

		$this->_headerString = 'MIME-Version: 1.0' . PHP_EOL;

		/* handle attachment */

		if ($this->_attachmentArray)
		{
			foreach ($this->_attachmentArray as $attachment)
			{
				if (is_file($attachment))
				{
					$content = trim(chunk_split(base64_encode($attachment)));
					$boundary = uniqid();
					$this->_headerString .= 'Content-Type: multipart/mixed; boundary="' . $boundary . '"' . PHP_EOL;
					$this->_headerString .= '--' . $boundary . PHP_EOL;

					/* handle body string */

					if ($this->_bodyString)
					{
						$this->_headerString .= 'Content-Type: text/html; charset=' . $settingModel->get('charset') . PHP_EOL;
						$this->_headerString .= 'Content-Transfer-Encoding: 8bit' . PHP_EOL;
						$this->_headerString .= $this->_bodyString . PHP_EOL;
						$this->_headerString .= '--' . $boundary . PHP_EOL;

						/* reset body string */

						$this->_bodyString = '';
					}
					$this->_headerString .= 'Content-Type: application/octet-stream; name="' . $attachment . '"' . PHP_EOL;
					$this->_headerString .= 'Content-Transfer-Encoding: base64' . PHP_EOL;
					$this->_headerString .= 'Content-Disposition: attachment; filename="' . $attachment . '"' . PHP_EOL;
					$this->_headerString .= $content . PHP_EOL;
					$this->_headerString .= '--' . $boundary . '--';
				}
			}
		}
		else
		{
			$this->_headerString .= 'Content-Type: text/html; charset=' . $settingModel->get('charset') . PHP_EOL;
		}

		/* collect header string */

		$this->_headerString .= 'From: ' . $this->_fromString . PHP_EOL;
		$this->_headerString .= 'Reply-To: ' . $this->_fromString . PHP_EOL;
		$this->_headerString .= 'Return-Path: ' . $this->_fromString . PHP_EOL;
	}

	/**
	 * send the email
	 *
	 * @since 2.6.2
	 */

	public function send() : bool
	{
		foreach ($this->_toArray as $to)
		{
			if (!function_exists('mail') || !mail($to, $this->_subjectString, $this->_bodyString, $this->_headerString))
			{
				return false;
			}
		}
		return true;
	}
}
