<?php
namespace App;
/**
 * Core: Redirection - Main Class
 * 
 * Core - Redirection
 * 
 * @copyright 2019 SCHLIX Web Inc
 *
 * @license GPLv3
 *
 * @package core
 * @version 1.0
 * @author  Roy <roy.hadrianoro@schlix.com>
 * @link    http://www.schlix.com
 */
class Core_Redirection extends \SCHLIX\cmsApplication_List {

    /**
     * Constructor
     */
    public function __construct() {
        parent::__construct("URL Redirection", 'gk_redirection_items');
        /* You can modify this  */
        $this->has_versioning = true; // set to false if you don't need versioning capability if this app wants versioning enabled
        $this->disable_frontend_runtime = true; //  set this to true if this is a backend only app
        
    }

    /**
     * View Main Page
     * @param int $pg
     * @return boolean
     */
    //_______________________________________________________________________________________________________________//
    public function viewMainPage($pg = 1) {
        return false;
    }

    /**
     * Validates save item. If there's an error, it will return an array
     * with one or more error string, otherwise it will return a boolean true
     * @global \App\Users $CurrentUser
     * @param array $datavalues
     * @return bool|array String array
     */
    public function getValidationErrorListBeforeSaveItem($datavalues)
    {
        $parent_error_list = parent::getValidationErrorListBeforeSaveItem($datavalues);
        $error_list = array();
        if (!in_array($datavalues['type'], ['301', '302']))
            $error_list[] = ___('Invalid redirect type.');
        
        $datavalues['path'] = trim ($datavalues['path']);
        if (empty($datavalues['path']) || $datavalues['path'] == '/')
            $error_list[] = ___('Old URL cannot be empty');
        else 
        {
            $psrc = mb_pathinfo(SCHLIX_SITE_HTTPBASE.'/'.$datavalues['path']);
            $valid_path = !empty($psrc ['dirname']) || !empty($psrc['filename']);
            if (!$valid_path)
            {
                $error_list[] = ___('Invalid source URL format');
            }
        }
        $rt = $datavalues['redirect_to'];
        $valid_rt_external_url = filter_var($rt, FILTER_VALIDATE_URL);
        $pi = mb_pathinfo($rt);
        # allow empty rt for redirection to home
        $valid_rt_path = empty($rt) || !empty($pi['dirname']) || !empty($pi['filename']);
        if (!$valid_rt_external_url && !$valid_rt_path)
            $error_list[] = ___('Target URL redirection URL format is invalid, it is neither a URL or a path');
        if (empty($error_list))
        {
            $existing_item  = $this->getItemByExactPath($datavalues['path']);
            if ($existing_item['id'] > 0 && ($existing_item['id'] != $datavalues['id']))
                $error_list[] = ___('There is already an existing redirection rule for this URL');
        }
        
        return array_merge($parent_error_list, $error_list);
    }

    /**
     * Before save item
     * @param array $datavalues
     * @return array
     */
    public function modifyDataValuesBeforeSaveItem($datavalues) {
        $datavalues = parent::modifyDataValuesBeforeSaveItem($datavalues);
        
        $datavalues['path'] = $this->removePrecedingSlash(trim($datavalues['path']));
        $datavalues['redirect_to'] = $this->removePrecedingSlash(trim($datavalues['redirect_to']));
        
        return $datavalues;
    }

    /**
     * Remove preceding slash from string
     * @param string $path
     * @return string
     */
    public function removePrecedingSlash( $path) {
        return preg_replace('/\A\//', '', $path);
    }
    
    /**
     * Return request URL, optionally without URL path.
     * @param bool $without_path
     * @return string
     */
    private function getRequestURL($without_path = false) {
        $url_proto = 'http' . (empty($_SERVER['HTTPS']) ? '' : 's') . '://';
        $url = $url_proto . $_SERVER['SERVER_NAME'];
        if (!$without_path) $url .= $_SERVER['REQUEST_URI'];
        return  $url;
    }
    
    /**
     * find Redirection item from URL string
     * @param string $url
     * @param string $fields
     * @return array
     */
    public function getItemFromURL($url,  $fields = '*') {
        if (!preg_match('/\Ahttps?:/', $url)) 
            $url = 'http://' . $url; // doesn't matter whether http / https used
        $url_parts = parse_url($url);
        if ($url_parts) {
            $path = sanitize_string($this->removePrecedingSlash($url_parts['path']));
            $result = $this->getAllItems($fields, "{$path} LIKE REPLACE(REPLACE(path, '?', '_'), '*', '%') AND status = 1", 0, 1);
            return ___c($result) > 0 ? $result[0] : null;
        } else {
            return null;
        }
    }
    
    /**
     * Similar to getItemFromURL but searching from array of rules instead of DB table.
     * Useful when array of rules stored in cache.
     * @param string $url
     * @param array $rules
     * @return varied
     */
    public function getArrayItemFromURL($url, $rules) {
        if (!preg_match('/\Ahttps?:/', $url)) 
            $url = 'http://' . $url; // doesn't matter whether http / https used
        $url_parts = parse_url($url);
        if ($url_parts) {
            $path = $this->removePrecedingSlash($url_parts['path']);
            foreach ($rules as $rule) {
                $regex = preg_quote($rule['path'], '/');
                $regex = str_replace(['\?', '\*'], ['.', '.*'], $regex);
                if ($rule['status'] == 1 && preg_match('/\A'.$regex.'\z/', $path))
                    return $rule;
            }
            return NULL;
        } else {
            return false;
        }
    }

    /**
     * Get exact path
     * @global \SCHLIX\cmsDatabase $SystemDB
     * @param string $path
     * @return array
     */
    public function getItemByExactPath($path)
    {
        global $SystemDB;

        $path = $this->removePrecedingSlash(trim($path));
        $p = $SystemDB->getQueryResultSingleRow("SELECT * FROM {$this->table_items} WHERE `path` = :path", ['path' => $path]);
        return $p;
    }
    
    /**
     * Add hits count for specified rule.
     * @global type $SystemDB
     * @param array $rule
     */
    public function addHits($rule) {
        global $SystemDB;
        $SystemDB->query("UPDATE gk_redirection_items SET hits = hits + 1 WHERE id = :id", ['id' => $rule['id']]);
    }

    /**
     * Return redirect URL from Redirect Item.
     * @param array $rule
     * @return string
     */
    private function getRedirectURL($rule) {
        // TODO: support wildcard replacement like $1, $2, etc.
        if (preg_match('/\Ahttps?:/', $rule['redirect_to']))
            return $rule['redirect_to'];
        else
            return '/'.$this->removePrecedingSlash($rule['redirect_to']);
    }
    
    /**
     * Trigger redirect for current request when there's matching rule.
     */
    public function doRedirect() {
        $url = $this->getRequestURL();
        $rule = $this->getItemFromURL($url, 'id, type, path, redirect_to');
        if ($rule) {
            $this->addHits($rule);
            $url = $this->getRedirectURL($rule);
            ob_end_clean();
            ob_start();
            header("Location: {$url}", true, $rule['type']);
            exit();
        }
    }

    /**
     * Run Command
     * @param array $command
     * @return boolean
     */
    public function Run($command) {
        return false;
    }




}
