<?php
/**
 * Kkart Admin Usage Tracking Opt In Note Provider.
 *
 * Adds a Usage Tracking Opt In extension note.
 */

namespace Automattic\Kkart\Admin\Notes;

defined( 'ABSPATH' ) || exit;

/**
 * Tracking_Opt_In
 */
class TrackingOptIn {
	/**
	 * Note traits.
	 */
	use NoteTraits;

	/**
	 * Name of the note for use in the database.
	 */
	const NOTE_NAME = 'kkart-admin-usage-tracking-opt-in';

	/**
	 * Attach hooks.
	 */
	public function __construct() {
		add_action( 'kkart_note_action_tracking-opt-in', array( $this, 'opt_in_to_tracking' ) );
	}

	/**
	 * Get the note.
	 *
	 * @return Note|null
	 */
	public static function get_note() {
		// Only show this note to stores that are opted out.
		if ( 'yes' === get_option( 'kkart_allow_tracking', 'no' ) ) {
			return;
		}

		// We want to show the note after one week.
		if ( ! self::kkart_admin_active_for( WEEK_IN_SECONDS ) ) {
			return;
		}

		/* translators: 1: open link to Kkart.com settings, 2: open link to Kkart.com tracking documentation, 3: close link tag. */
		$content_format = __(
			'Gathering usage data allows us to improve Kkart. Your store will be considered as we evaluate new features, judge the quality of an update, or determine if an improvement makes sense. You can always visit the %1$sSettings%3$s and choose to stop sharing data. %2$sRead more%3$s about what data we collect.',
			'kkart'
		);

		$note_content = sprintf(
			$content_format,
			'<a href="' . esc_url( admin_url( 'admin.php?page=kkart-settings&tab=advanced&section=kkart_com' ) ) . '" target="_blank">',
			'<a href="https://kkart.com/usage-tracking" target="_blank">',
			'</a>'
		);

		$note = new Note();
		$note->set_title( __( 'Help Kkart improve with usage tracking', 'kkart' ) );
		$note->set_content( $note_content );
		$note->set_content_data( (object) array() );
		$note->set_type( Note::E_KKART_ADMIN_NOTE_INFORMATIONAL );
		$note->set_name( self::NOTE_NAME );
		$note->set_source( 'kkart-admin' );
		$note->add_action( 'tracking-opt-in', __( 'Activate usage tracking', 'kkart' ), false, Note::E_KKART_ADMIN_NOTE_ACTIONED, true );
		return $note;
	}

	/**
	 * Opt in to usage tracking when note is actioned.
	 *
	 * @param Note $note Note being acted upon.
	 */
	public function opt_in_to_tracking( $note ) {
		if ( self::NOTE_NAME === $note->get_name() ) {
			// Opt in to tracking and schedule the first data update.
			// Same mechanism as in KKART_Admin_Setup_Wizard::kkart_setup_store_setup_save().
			update_option( 'kkart_allow_tracking', 'yes' );
			wp_schedule_single_event( time() + 10, 'kkart_tracker_send_event', array( true ) );
		}
	}
}
