/**
 * External dependencies
 */
import { __, sprintf, _n } from '@wordpress/i18n';
import { Fragment, useState } from '@wordpress/element';
import { InspectorControls, BlockControls } from '@wordpress/block-editor';
import {
	Placeholder,
	Disabled,
	PanelBody,
	ToggleControl,
	Button,
	Toolbar,
	withSpokenMessages,
} from '@wordpress/components';
import { Icon, server, external } from '@kkart/icons';
import { SearchListControl } from '@kkart/components';
import { mapValues, toArray, sortBy, find } from 'lodash';
import { ATTRIBUTES } from '@kkart/block-settings';
import { getAdminLink } from '@kkart/settings';
import HeadingToolbar from '@kkart/editor-components/heading-toolbar';
import BlockTitle from '@kkart/editor-components/block-title';
import ToggleButtonControl from '@kkart/editor-components/toggle-button-control';

/**
 * Internal dependencies
 */
import Block from './block.js';
import './editor.scss';

const Edit = ( { attributes, setAttributes, debouncedSpeak } ) => {
	const {
		attributeId,
		className,
		displayStyle,
		heading,
		headingLevel,
		isPreview,
		queryType,
		showCounts,
		showFilterButton,
	} = attributes;

	const [ isEditing, setIsEditing ] = useState(
		! attributeId && ! isPreview
	);

	const getBlockControls = () => {
		return (
			<BlockControls>
				<Toolbar
					controls={ [
						{
							icon: 'edit',
							title: __( 'Edit', 'kkart' ),
							onClick: () => setIsEditing( ! isEditing ),
							isActive: isEditing,
						},
					] }
				/>
			</BlockControls>
		);
	};

	const getInspectorControls = () => {
		return (
			<InspectorControls key="inspector">
				<PanelBody
					title={ __( 'Content', 'kkart' ) }
				>
					<ToggleControl
						label={ __(
							'Product count',
							'kkart'
						) }
						help={
							showCounts
								? __(
										'Product count is visible.',
										'kkart'
								  )
								: __(
										'Product count is hidden.',
										'kkart'
								  )
						}
						checked={ showCounts }
						onChange={ () =>
							setAttributes( {
								showCounts: ! showCounts,
							} )
						}
					/>
					<p>
						{ __(
							'Heading Level',
							'kkart'
						) }
					</p>
					<HeadingToolbar
						isCollapsed={ false }
						minLevel={ 2 }
						maxLevel={ 7 }
						selectedLevel={ headingLevel }
						onChange={ ( newLevel ) =>
							setAttributes( { headingLevel: newLevel } )
						}
					/>
				</PanelBody>
				<PanelBody
					title={ __(
						'Block Settings',
						'kkart'
					) }
				>
					<ToggleButtonControl
						label={ __(
							'Query Type',
							'kkart'
						) }
						help={
							queryType === 'and'
								? __(
										'Products that have all of the selected attributes will be shown.',
										'kkart'
								  )
								: __(
										'Products that have any of the selected attributes will be shown.',
										'kkart'
								  )
						}
						value={ queryType }
						options={ [
							{
								label: __(
									'And',
									'kkart'
								),
								value: 'and',
							},
							{
								label: __(
									'Or',
									'kkart'
								),
								value: 'or',
							},
						] }
						onChange={ ( value ) =>
							setAttributes( {
								queryType: value,
							} )
						}
					/>
					<ToggleButtonControl
						label={ __(
							'Display Style',
							'kkart'
						) }
						value={ displayStyle }
						options={ [
							{
								label: __(
									'List',
									'kkart'
								),
								value: 'list',
							},
							{
								label: __(
									'Dropdown',
									'kkart'
								),
								value: 'dropdown',
							},
						] }
						onChange={ ( value ) =>
							setAttributes( {
								displayStyle: value,
							} )
						}
					/>
					<ToggleControl
						label={ __(
							'Filter button',
							'kkart'
						) }
						help={
							showFilterButton
								? __(
										'Products will only update when the button is pressed.',
										'kkart'
								  )
								: __(
										'Products will update as options are selected.',
										'kkart'
								  )
						}
						checked={ showFilterButton }
						onChange={ ( value ) =>
							setAttributes( {
								showFilterButton: value,
							} )
						}
					/>
				</PanelBody>
				<PanelBody
					title={ __(
						'Filter Products by Attribute',
						'kkart'
					) }
					initialOpen={ false }
				>
					{ renderAttributeControl() }
				</PanelBody>
			</InspectorControls>
		);
	};

	const noAttributesPlaceholder = () => (
		<Placeholder
			className="kkart-block-attribute-filter"
			icon={ <Icon srcElement={ server } /> }
			label={ __(
				'Filter Products by Attribute',
				'kkart'
			) }
			instructions={ __(
				'Display a list of filters based on a chosen attribute.',
				'kkart'
			) }
		>
			<p>
				{ __(
					"Attributes are needed for filtering your products. You haven't created any attributes yet.",
					'kkart'
				) }
			</p>
			<Button
				className="kkart-block-attribute-filter__add-attribute-button"
				isDefault
				isLarge
				href={ getAdminLink(
					'edit.php?post_type=product&page=product_attributes'
				) }
			>
				{ __( 'Add new attribute', 'kkart' ) +
					' ' }
				<Icon srcElement={ external } />
			</Button>
			<Button
				className="kkart-block-attribute-filter__read_more_button"
				isTertiary
				href="https://docs.kkart.com/document/managing-product-taxonomies/"
			>
				{ __( 'Learn more', 'kkart' ) }
			</Button>
		</Placeholder>
	);

	const onDone = () => {
		setIsEditing( false );
		debouncedSpeak(
			__(
				'Showing Filter Products by Attribute block preview.',
				'kkart'
			)
		);
	};

	const onChange = ( selected ) => {
		if ( ! selected || ! selected.length ) {
			return;
		}

		const selectedId = selected[ 0 ].id;
		const productAttribute = find( ATTRIBUTES, [
			'attribute_id',
			selectedId.toString(),
		] );

		if ( ! productAttribute || attributeId === selectedId ) {
			return;
		}

		const attributeName = productAttribute.attribute_label;

		setAttributes( {
			attributeId: selectedId,
			heading: sprintf(
				// Translators: %s attribute name.
				__( 'Filter by %s', 'kkart' ),
				attributeName
			),
		} );
	};

	const renderAttributeControl = () => {
		const messages = {
			clear: __(
				'Clear selected attribute',
				'kkart'
			),
			list: __( 'Product Attributes', 'kkart' ),
			noItems: __(
				"Your store doesn't have any product attributes.",
				'kkart'
			),
			search: __(
				'Search for a product attribute:',
				'kkart'
			),
			selected: ( n ) =>
				sprintf(
					// Translators: %d is the number of attributes selected.
					_n(
						'%d attribute selected',
						'%d attributes selected',
						n,
						'kkart'
					),
					n
				),
			updated: __(
				'Product attribute search results updated.',
				'kkart'
			),
		};

		const list = sortBy(
			toArray(
				mapValues( ATTRIBUTES, ( item ) => {
					return {
						id: parseInt( item.attribute_id, 10 ),
						name: item.attribute_label,
					};
				} )
			),
			'name'
		);

		return (
			<SearchListControl
				className="kkart-product-attributes"
				list={ list }
				selected={ list.filter( ( { id } ) => id === attributeId ) }
				onChange={ onChange }
				messages={ messages }
				isSingle
			/>
		);
	};

	const renderEditMode = () => {
		return (
			<Placeholder
				className="kkart-block-attribute-filter"
				icon={ <Icon srcElement={ server } /> }
				label={ __(
					'Filter Products by Attribute',
					'kkart'
				) }
				instructions={ __(
					'Display a list of filters based on a chosen attribute.',
					'kkart'
				) }
			>
				<div className="kkart-block-attribute-filter__selection">
					{ renderAttributeControl() }
					<Button isPrimary onClick={ onDone }>
						{ __( 'Done', 'kkart' ) }
					</Button>
				</div>
			</Placeholder>
		);
	};

	return Object.keys( ATTRIBUTES ).length === 0 ? (
		noAttributesPlaceholder()
	) : (
		<Fragment>
			{ getBlockControls() }
			{ getInspectorControls() }
			{ isEditing ? (
				renderEditMode()
			) : (
				<div className={ className }>
					<BlockTitle
						headingLevel={ headingLevel }
						heading={ heading }
						onChange={ ( value ) =>
							setAttributes( { heading: value } )
						}
					/>
					<Disabled>
						<Block attributes={ attributes } isEditor />
					</Disabled>
				</div>
			) }
		</Fragment>
	);
};

export default withSpokenMessages( Edit );
