// @ts-nocheck
/**
 * External dependencies
 */
import PropTypes from 'prop-types';
import { __ } from '@wordpress/i18n';
import {
	SubtotalsItem,
	TotalsFeesItem,
	TotalsCouponCodeInput,
	TotalsDiscountItem,
	TotalsFooterItem,
	TotalsShippingItem,
	TotalsTaxesItem,
} from '@kkart/base-components/cart-checkout';
import {
	COUPONS_ENABLED,
	DISPLAY_CART_PRICES_INCLUDING_TAX,
} from '@kkart/block-settings';
import { getCurrencyFromPriceResponse } from '@kkart/base-utils';
import { CartExpressPayment } from '@kkart/base-components/payment-methods';
import {
	useStoreCartCoupons,
	useStoreCart,
	useStoreNotices,
} from '@kkart/base-hooks';
import classnames from 'classnames';
import {
	Sidebar,
	SidebarLayout,
	Main,
} from '@kkart/base-components/sidebar-layout';
import Title from '@kkart/base-components/title';
import { getSetting } from '@kkart/settings';
import { useEffect } from '@wordpress/element';
import { decodeEntities } from '@wordpress/html-entities';

/**
 * Internal dependencies
 */
import CheckoutButton from '../checkout-button';
import CartLineItemsTitle from './cart-line-items-title';
import CartLineItemsTable from './cart-line-items-table';

import './style.scss';

/**
 * Component that renders the Cart block when user has something in cart aka "full".
 *
 * @param {Object} props Incoming props for the component.
 * @param {Object} props.attributes Incoming attributes for block.
 */
const Cart = ( { attributes } ) => {
	const {
		isShippingCalculatorEnabled,
		isShippingCostHidden,
		hasDarkControls,
	} = attributes;

	const {
		cartItems,
		cartTotals,
		cartIsLoading,
		cartItemsCount,
		cartItemErrors,
		cartNeedsPayment,
		cartNeedsShipping,
	} = useStoreCart();

	const {
		applyCoupon,
		removeCoupon,
		isApplyingCoupon,
		isRemovingCoupon,
		appliedCoupons,
	} = useStoreCartCoupons();

	const { addErrorNotice } = useStoreNotices();

	// Ensures any cart errors listed in the API response get shown.
	useEffect( () => {
		cartItemErrors.forEach( ( error ) => {
			addErrorNotice( decodeEntities( error.message ), {
				isDismissible: true,
				id: error.code,
			} );
		} );
	}, [ addErrorNotice, cartItemErrors ] );

	const totalsCurrency = getCurrencyFromPriceResponse( cartTotals );

	const cartClassName = classnames( 'kkart-block-cart', {
		'kkart-block-cart--is-loading': cartIsLoading,
		'has-dark-controls': hasDarkControls,
	} );

	return (
		<SidebarLayout className={ cartClassName }>
			<Main className="kkart-block-cart__main">
				<CartLineItemsTitle itemCount={ cartItemsCount } />
				<CartLineItemsTable
					lineItems={ cartItems }
					isLoading={ cartIsLoading }
				/>
			</Main>
			<Sidebar className="kkart-block-cart__sidebar">
				<Title headingLevel="2" className="kkart-block-cart__totals-title">
					{ __( 'Cart totals', 'kkart' ) }
				</Title>
				<SubtotalsItem
					currency={ totalsCurrency }
					values={ cartTotals }
				/>
				<TotalsFeesItem
					currency={ totalsCurrency }
					values={ cartTotals }
				/>
				<TotalsDiscountItem
					cartCoupons={ appliedCoupons }
					currency={ totalsCurrency }
					isRemovingCoupon={ isRemovingCoupon }
					removeCoupon={ removeCoupon }
					values={ cartTotals }
				/>
				{ cartNeedsShipping && (
					<TotalsShippingItem
						showCalculator={ isShippingCalculatorEnabled }
						showRatesWithoutAddress={ ! isShippingCostHidden }
						values={ cartTotals }
						currency={ totalsCurrency }
					/>
				) }
				{ ! DISPLAY_CART_PRICES_INCLUDING_TAX && (
					<TotalsTaxesItem
						currency={ totalsCurrency }
						values={ cartTotals }
					/>
				) }
				{ COUPONS_ENABLED && (
					<TotalsCouponCodeInput
						onSubmit={ applyCoupon }
						isLoading={ isApplyingCoupon }
					/>
				) }
				<TotalsFooterItem
					currency={ totalsCurrency }
					values={ cartTotals }
				/>
				<div className="kkart-block-cart__payment-options">
					{ cartNeedsPayment && <CartExpressPayment /> }
					<CheckoutButton
						link={ getSetting(
							'page-' + attributes?.checkoutPageId,
							false
						) }
					/>
				</div>
			</Sidebar>
		</SidebarLayout>
	);
};

Cart.propTypes = {
	attributes: PropTypes.object.isRequired,
};

export default Cart;
