<?php
// Load Rate Limits and initialize RateLimit class
require_once $abs_us_root . $us_url_root . 'users/includes/rate_limits.php';
require_once $abs_us_root . $us_url_root . 'users/classes/RateLimit.php';

$rateLimit = new RateLimit();
$secondWrite = Input::get('secondWrite');
if ($secondWrite == "true") {
    $cb = randomstring(4);
    Redirect::to($us_url_root . "users/admin.php?view=rate_limits&cb=$cb");
}

// Check if using default (insecure) rate limits
$using_default_rate_limits = false;
if (isset($rateLimits['login_attempt']['ip_max']) && $rateLimits['login_attempt']['ip_max'] >= 10000) {
    $using_default_rate_limits = true;
}

// Get proxy configuration status
$proxy_enabled = isset($settings->behind_reverse_proxy) && $settings->behind_reverse_proxy;
$proxy_config_count = $db->query("SELECT COUNT(*) as count FROM us_rate_limit_proxy_settings WHERE enabled = 1")->first()->count ?? 0;

// Handle form submissions
if (!empty($_POST)) {
    if (!Token::check($_POST['csrf'])) {
        usError("Token failed");
        Redirect::to($us_url_root . "users/admin.php?view=rate_limits");
    }

    // Handle proxy settings toggle
    if (isset($_POST['toggle_proxy'])) {
        $new_val = $settings->behind_reverse_proxy ? 0 : 1;
        $db->update('settings', 1, ['behind_reverse_proxy' => $new_val]);
        $message = $new_val ? "Configuration updated: UserSpice is now set as being behind a reverse proxy." : "Configuration updated: UserSpice is no longer set as being behind a reverse proxy.";
        usSuccess($message);
        logger($user->data()->id, 'RateLimit', "Toggled 'behind_reverse_proxy' setting to $new_val");
        Redirect::to($us_url_root . "users/admin.php?view=rate_limits");
    }

    // Handle proxy configuration
    if (isset($_POST['add_proxy'])) {
        $proxy_ip = Input::get('proxy_ip');
        $header_name = Input::get('header_name');
        $priority = (int)Input::get('priority');

        if ($header_name === 'custom') {
            $header_name = Input::get('custom_header_name');
        }

        if (empty(trim($header_name))) {
            usError("The HTTP header name cannot be empty.");
            Redirect::to($us_url_root . "users/admin.php?view=rate_limits");
        }

        if (!preg_match('/^[a-zA-Z0-9\-]+$/', $header_name)) {
            usError("Invalid HTTP header name format. Only letters, numbers, and hyphens are allowed.");
            Redirect::to($us_url_root . "users/admin.php?view=rate_limits");
        }

        if (filter_var($proxy_ip, FILTER_VALIDATE_IP) || preg_match('/^\d+\.\d+\.\d+\.\d+\/\d+$/', $proxy_ip)) {
            $db->insert('us_rate_limit_proxy_settings', [
                'proxy_ip' => $proxy_ip,
                'header_name' => $header_name,
                'priority' => $priority,
                'enabled' => 1
            ]);
            usSuccess("Trusted proxy source added successfully");
            logger($user->data()->id, 'RateLimit', "Added proxy config: $proxy_ip using header $header_name");
        } else {
            usError("Invalid IP address or CIDR format");
        }
        Redirect::to($us_url_root . "users/admin.php?view=rate_limits");
    }

    // Handle proxy deletion
    if (isset($_POST['delete_proxy'])) {
        $proxy_id = (int)Input::get('proxy_id');
        $db->delete('us_rate_limit_proxy_settings', ['id' => $proxy_id]);
        usSuccess("Proxy configuration removed");
        logger($user->data()->id, 'RateLimit', "Deleted proxy config ID: $proxy_id");
        Redirect::to($us_url_root . "users/admin.php?view=rate_limits");
    }

    // Handle rate limit updates
    if (isset($_POST['update_rate_limits'])) {
        $new_limits = [];
        foreach ($_POST['limits'] as $action => $limits) {
            $new_limits[$action] = [];
            foreach ($limits as $key => $value) {
                $new_limits[$action][$key] = max(0, (int)$value);
            }
        }

        // Write to usersc/includes/rate_limits.php
        $rate_limits_file = $abs_us_root . $us_url_root . 'usersc/includes/rate_limits.php';
        $usersc_dir = dirname($rate_limits_file);

        if (!is_dir($usersc_dir)) {
            mkdir($usersc_dir, 0755, true);
        }

        $content = "<?php\n";
        $content .= "/**\n";
        $content .= " * Custom Rate Limits Configuration\n";
        $content .= " * Generated by UserSpice Rate Limiting Dashboard\n";
        $content .= " * Last updated: " . date('Y-m-d H:i:s') . "\n";
        $content .= " */\n\n";

        foreach ($new_limits as $action => $limits) {
            foreach ($limits as $key => $value) {
                $content .= "\$rateLimits['$action']['$key'] = $value;\n";
            }
            $content .= "\n";
        }

        if (file_put_contents($rate_limits_file, $content)) {
            if (function_exists('opcache_invalidate')) {
                opcache_invalidate($rate_limits_file, true);
            }
            usSuccess("Rate limits updated successfully");

        } else {
            usError("Failed to write rate limits file. Check permissions on usersc/includes/");
        }
        Redirect::to($us_url_root . "users/admin.php?view=rate_limits&secondWrite=true");
    }

    // Handle cleanup
    if (isset($_POST['cleanup_logs'])) {
        $hours = max(1, (int)Input::get('cleanup_hours'));

        $count = $rateLimit->cleanup($hours);
        usSuccess("Cleaned up $count old rate limit records");
        logger($user->data()->id, 'RateLimit', "Cleaned up $count rate limit records older than $hours hours");
        Redirect::to($us_url_root . "users/admin.php?view=rate_limits");
    }
}

// Get current IP for testing
$current_ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
$real_ip = $current_ip; // Default fallback

// Try to get real IP if method is available
if (method_exists($rateLimit, 'getRealIP')) {
    try {
        $reflection = new ReflectionMethod($rateLimit, 'getRealIP');
        if ($reflection->isPublic()) {
            $real_ip = $rateLimit->getRealIP();
        }
    } catch (Exception $e) {
        // Method might be private, use fallback logic
        $real_ip = $current_ip;
        if ($proxy_enabled && !empty($proxy_configs)) {
            // Basic proxy detection fallback
            $headers = ['HTTP_X_FORWARDED_FOR', 'HTTP_X_REAL_IP', 'HTTP_CF_CONNECTING_IP'];
            foreach ($headers as $header) {
                if (!empty($_SERVER[$header])) {
                    $ips = explode(',', $_SERVER[$header]);
                    $client_ip = trim(end($ips));
                    if (filter_var($client_ip, FILTER_VALIDATE_IP)) {
                        $real_ip = $client_ip;
                        break;
                    }
                }
            }
        }
    }
}

// Get recent rate limit activity
$recent_activity = $db->query("
    SELECT action, success, COUNT(*) as count, MAX(attempt_time) as last_attempt
    FROM us_rate_limits 
    WHERE attempt_time > DATE_SUB(NOW(), INTERVAL 1 HOUR)
    GROUP BY action, success
    ORDER BY last_attempt DESC
    LIMIT 20
")->results();

// Get proxy configurations
$proxy_configs = $db->query("
    SELECT * FROM us_rate_limit_proxy_settings 
    WHERE enabled = 1 
    ORDER BY priority ASC
")->results();

// Calculate rate limiting health score
function calculateRateLimitHealth($rateLimits, $using_defaults, $proxy_enabled, $proxy_count)
{
    $score = 0;
    $max_score = 100;

    // Not using defaults (40 points)
    if (!$using_defaults) $score += 40;

    // Key actions have reasonable limits (30 points)
    $key_actions = ['login_attempt', 'password_reset_request', 'registration_attempt'];
    foreach ($key_actions as $action) {
        if (isset($rateLimits[$action]['ip_max']) && $rateLimits[$action]['ip_max'] < 1000) {
            $score += 10;
        }
    }

    // Proxy configuration (20 points)
    if ($proxy_enabled && $proxy_count > 0) {
        $score += 20;
    } elseif (!$proxy_enabled) {
        $score += 15; // Not behind proxy is also good
    }

    // Rate limiting enabled (10 points)
    if (class_exists('RateLimit')) $score += 10;

    return min($score, $max_score);
}

$rate_limit_health = calculateRateLimitHealth($rateLimits, $using_default_rate_limits, $proxy_enabled, $proxy_config_count);
$health_color = $rate_limit_health >= 80 ? 'success' : ($rate_limit_health >= 60 ? 'warning' : 'danger');
?>

<style>
    .rate-limit-card {
        transition: all 0.2s ease-in-out;
    }

    .rate-limit-card:hover {
        transform: translateY(-2px);
        box-shadow: 0 4px 8px rgba(0, 0, 0, 0.1);
    }

    .limit-input {
        width: 80px;
        font-family: 'Courier New', monospace;
    }

    .action-header {
        background: linear-gradient(45deg, #f8f9fa, #e9ecef);
        border-left: 4px solid #007bff;
    }

    .proxy-item {
        border-left: 3px solid #28a745;
        background: #f8f9fa;
    }

    .status-good {
        color: #28a745;
        font-weight: bold;
    }

    .status-warning {
        color: #ffc107;
        font-weight: bold;
    }

    .status-danger {
        color: #dc3545;
        font-weight: bold;
    }
</style>

<div class="row">
    <div class="col-12">
        <h2 class="mb-3">
            <i class="fas fa-traffic-light me-2"></i>Rate Limiting Control Center
            <span class="badge bg-<?= $health_color ?> ms-2"><?= $rate_limit_health ?>% Health Score</span>
        </h2>
        <p class="mb-4">Configure and monitor your rate limiting system to protect against brute-force attacks and abuse. Rate limiting tracks and restricts repeated actions from the same source.</p>
    </div>
</div>

<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-heartbeat me-2"></i>System Health Overview</h5>
            </div>
            <div class="card-body">
                <div class="row text-center">
                    <div class="col-6 col-md-3">
                        <div class="border-end">
                            <h4 class="mb-1 text-<?= $health_color ?>"><?= $rate_limit_health ?>%</h4>
                            <small class="text-muted">Health Score</small>
                        </div>
                    </div>
                    <div class="col-6 col-md-3">
                        <div class="border-end">
                            <h4 class="mb-1 <?= $using_default_rate_limits ? 'status-danger' : 'status-good' ?>">
                                <?= $using_default_rate_limits ? 'DEFAULT' : 'CUSTOM' ?>
                            </h4>
                            <small class="text-muted">Configuration</small>
                        </div>
                    </div>
                    <div class="col-6 col-md-3">
                        <div class="border-end">
                            <h4 class="mb-1 <?= $proxy_enabled ? 'status-good' : 'status-warning' ?>">
                                <?= $proxy_enabled ? 'YES' : 'NO' ?>
                            </h4>
                            <small class="text-muted">UserSpice is Behind Proxy</small>
                        </div>
                    </div>
                    <div class="col-6 col-md-3">
                        <h4 class="mb-1 text-info"><?= count($recent_activity) ?></h4>
                        <small class="text-muted">Recent Events</small>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<div class="row mb-4">
    <div class="col-12">
        <div class="card">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-bolt me-2"></i>Quick Actions</h5>
            </div>
            <div class="card-body">
                <div class="row g-3">
                    <div class="col-12 col-md-3">
                        <form method="post" class="h-100">
                            <input type="hidden" name="csrf" value="<?= Token::generate() ?>">
                            <button type="submit" name="toggle_proxy" class="btn btn-<?= $proxy_enabled ? 'success' : 'outline-secondary' ?> d-flex flex-column align-items-center justify-content-center h-100 w-100">
                                <i class="fas fa-server mb-2" style="font-size: 1.5rem;"></i>
                                <div>UserSpice is Behind Proxy</div>
                                <small><?= $proxy_enabled ? 'Enabled' : 'Disabled' ?></small>
                            </button>
                        </form>
                    </div>
                    <div class="col-12 col-md-3">
                        <button id="addProxyBtn" type="button" class="btn btn-outline-primary d-flex flex-column align-items-center justify-content-center h-100 w-100">
                            <i class="fas fa-plus mb-2" style="font-size: 1.5rem;"></i>
                            <div>Add Trusted Proxy</div>
                            <small>Configure Source IP</small>
                        </button>
                    </div>
                    <div class="col-12 col-md-3">
                        <button id="cleanupLogsBtn" type="button" class="btn btn-outline-warning d-flex flex-column align-items-center justify-content-center h-100 w-100">
                            <i class="fas fa-broom mb-2" style="font-size: 1.5rem;"></i>
                            <div>Cleanup Logs</div>
                            <small>Remove Old Records</small>
                        </button>
                    </div>
                    <div class="col-12 col-md-3">
                        <button type="button" class="btn btn-outline-info autoConfigBtn d-flex flex-column align-items-center justify-content-center h-100 w-100">
                            <i class="fas fa-magic mb-2" style="font-size: 1.5rem;"></i>
                            <div>Auto-Configure Limits</div>
                            <small>Set Recommended Values</small>
                        </button>
                    </div>

                </div>
            </div>
        </div>
    </div>
</div>

<div class="row">
    <div class="col-md-8">
        <div class="card mb-4">
            <div class="card-header d-flex justify-content-between align-items-center">
                <h5 class="mb-0"><i class="fas fa-sliders-h me-2"></i>Rate Limits Configuration</h5>
                <?php if ($using_default_rate_limits): ?>
                    <span class="badge bg-danger">Using Insecure Defaults</span>
                <?php else: ?>
                    <span class="badge bg-success">Custom Configuration</span>
                <?php endif; ?>
            </div>
            <div class="card-body">
                <?php if ($using_default_rate_limits): ?>
                    <div class="alert alert-danger mb-4">
                        <strong><i class="fas fa-exclamation-triangle me-2"></i>Security Warning</strong><br>
                        Your rate limits are set to extremely permissive default values that provide minimal protection.
                        Customize these immediately for production use.
                    </div>
                <?php endif; ?>

                <form method="post" id="rateLimitsForm">
                    <input type="hidden" name="csrf" value="<?= Token::generate() ?>">

                    <?php foreach ($rateLimits as $action => $limits): ?>
                        <div class="action-header p-3 mb-3 rounded">
                            <h6 class="mb-3 text-primary">
                                <i class="fas fa-cog me-2"></i><?= ucwords(str_replace('_', ' ', $action)) ?>
                                <small class="text-muted ms-2">(<?= $action ?>)</small>
                            </h6>

                            <div class="row g-3">
                                <?php if (isset($limits['ip_max'])): ?>
                                    <div class="col-md-6">
                                        <label class="form-label fw-bold">IP-Based Limits</label>
                                        <div class="input-group input-group-sm mb-2">
                                            <span class="input-group-text">Max Failures</span>
                                            <input type="number" name="limits[<?= $action ?>][ip_max]" value="<?= $limits['ip_max'] ?>" class="form-control limit-input" min="1" max="50000">
                                            <span class="input-group-text">attempts</span>
                                        </div>
                                        <div class="input-group input-group-sm">
                                            <span class="input-group-text">Window</span>
                                            <input type="number" name="limits[<?= $action ?>][ip_window]" value="<?= $limits['ip_window'] ?>" class="form-control limit-input" min="60" max="86400">
                                            <span class="input-group-text">seconds</span>
                                        </div>
                                    </div>
                                <?php endif; ?>

                                <?php if (isset($limits['user_max'])): ?>
                                    <div class="col-md-6">
                                        <label class="form-label fw-bold">User-Based Limits</label>
                                        <div class="input-group input-group-sm mb-2">
                                            <span class="input-group-text">Max Failures</span>
                                            <input type="number" name="limits[<?= $action ?>][user_max]" value="<?= $limits['user_max'] ?>" class="form-control limit-input" min="1" max="50000">
                                            <span class="input-group-text">attempts</span>
                                        </div>
                                        <div class="input-group input-group-sm">
                                            <span class="input-group-text">Window</span>
                                            <input type="number" name="limits[<?= $action ?>][user_window]" value="<?= $limits['user_window'] ?>" class="form-control limit-input" min="60" max="86400">
                                            <span class="input-group-text">seconds</span>
                                        </div>
                                    </div>
                                <?php endif; ?>

                                <?php if (isset($limits['email_max'])): ?>
                                    <div class="col-md-6">
                                        <label class="form-label fw-bold">Email-Based Limits</label>
                                        <div class="input-group input-group-sm mb-2">
                                            <span class="input-group-text">Max Attempts</span>
                                            <input type="number" name="limits[<?= $action ?>][email_max]" value="<?= $limits['email_max'] ?>" class="form-control limit-input" min="1" max="50000">
                                            <span class="input-group-text">attempts</span>
                                        </div>
                                        <div class="input-group input-group-sm">
                                            <span class="input-group-text">Window</span>
                                            <input type="number" name="limits[<?= $action ?>][email_window]" value="<?= $limits['email_window'] ?>" class="form-control limit-input" min="60" max="86400">
                                            <span class="input-group-text">seconds</span>
                                        </div>
                                    </div>
                                <?php endif; ?>

                                <?php if (isset($limits['token_max'])): ?>
                                    <div class="col-md-6">
                                        <label class="form-label fw-bold">Token-Based Limits</label>
                                        <div class="input-group input-group-sm mb-2">
                                            <span class="input-group-text">Max Attempts</span>
                                            <input type="number" name="limits[<?= $action ?>][token_max]" value="<?= $limits['token_max'] ?>" class="form-control limit-input" min="1" max="50000">
                                            <span class="input-group-text">attempts</span>
                                        </div>
                                        <div class="input-group input-group-sm">
                                            <span class="input-group-text">Window</span>
                                            <input type="number" name="limits[<?= $action ?>][token_window]" value="<?= $limits['token_window'] ?>" class="form-control limit-input" min="60" max="86400">
                                            <span class="input-group-text">seconds</span>
                                        </div>
                                    </div>
                                <?php endif; ?>

                                <?php if (isset($limits['credential_max'])): ?>
                                    <div class="col-md-6">
                                        <label class="form-label fw-bold">Credential-Based Limits</label>
                                        <div class="input-group input-group-sm mb-2">
                                            <span class="input-group-text">Max Attempts</span>
                                            <input type="number" name="limits[<?= $action ?>][credential_max]" value="<?= $limits['credential_max'] ?>" class="form-control limit-input" min="1" max="50000">
                                            <span class="input-group-text">attempts</span>
                                        </div>
                                        <div class="input-group input-group-sm">
                                            <span class="input-group-text">Window</span>
                                            <input type="number" name="limits[<?= $action ?>][credential_window]" value="<?= $limits['credential_window'] ?>" class="form-control limit-input" min="60" max="86400">
                                            <span class="input-group-text">seconds</span>
                                        </div>
                                    </div>
                                <?php endif; ?>

                                <?php if (isset($limits['total_max'])): ?>
                                    <div class="col-12">
                                        <div class="bg-light p-2 rounded">
                                            <label class="form-label fw-bold text-warning">
                                                <i class="fas fa-exclamation-triangle me-1"></i>Circuit Breaker (Total Attempts)
                                            </label>
                                            <div class="row">
                                                <div class="col-md-6">
                                                    <div class="input-group input-group-sm">
                                                        <span class="input-group-text">Max Total</span>
                                                        <input type="number" name="limits[<?= $action ?>][total_max]" value="<?= $limits['total_max'] ?>" class="form-control limit-input" min="1" max="50000">
                                                        <span class="input-group-text">attempts</span>
                                                    </div>
                                                </div>
                                                <div class="col-md-6">
                                                    <div class="input-group input-group-sm">
                                                        <span class="input-group-text">Window</span>
                                                        <input type="number" name="limits[<?= $action ?>][total_window]" value="<?= $limits['total_window'] ?>" class="form-control limit-input" min="60" max="86400">
                                                        <span class="input-group-text">seconds</span>
                                                    </div>
                                                </div>
                                            </div>
                                            <small class="text-muted">Blocks ALL attempts (successful + failed) for this action when exceeded</small>
                                        </div>
                                    </div>
                                <?php endif; ?>
                            </div>
                        </div>
                    <?php endforeach; ?>

                    <div class="d-grid gap-2 d-md-flex justify-content-md-end">
                        <button type="button" class="btn btn-info autoConfigBtn">
                            <i class="fas fa-magic me-2"></i>Auto-Configure Limits
                        </button>
                        <button type="submit" name="update_rate_limits" class="btn btn-primary">
                            <i class="fas fa-save me-2"></i>Save Configuration
                        </button>
                    </div>
                </form>
            </div>
        </div>

        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-graduation-cap me-2"></i>Understanding Rate Limits</h5>
            </div>
            <div class="card-body">
                <div class="row">
                    <div class="col-md-6">
                        <h6><i class="fas fa-bullseye me-2 text-primary"></i>Failure-Based Limits</h6>
                        <p class="small">These limits focus on <strong>failed</strong> actions, like incorrect password entries. They are perfect for stopping brute-force guessing attacks. When the user finally succeeds (e.g., logs in), their failure count for that action resets, preventing them from being locked out by their own mistakes.</p>

                        <h6><i class="fas fa-shield-alt me-2 text-warning"></i>Circuit Breaker (Total Limits)</h6>
                        <p class="small">This is a global safety net that tracks <strong>all</strong> attempts for an action, both successful and failed. If any single source sends too many requests in a short time, this limit will temporarily block them. It's designed to prevent system abuse and denial-of-service, even from a logged-in user.</p>
                    </div>
                    <div class="col-md-6">
                        <h6><i class="fas fa-balance-scale me-2 text-success"></i>Recommended Strategy</h6>
                        <ul class="small">
                            <li><strong>IP limits:</strong> Be more generous. A single IP could be a public WiFi, university, or office network shared by many legitimate users.</li>
                            <li><strong>User/Email limits:</strong> Be stricter. These protect individual accounts from targeted attacks.</li>
                            <li><strong>Time Windows:</strong> 5-15 minutes (300-900 seconds) is a common starting point. This is the period over which attempts are counted.</li>
                            <li><strong>Circuit Breaker:</strong> Set total limits higher than failure limits (e.g., 3-5x higher) to allow for normal mistakes but catch abusive behavior.</li>
                        </ul>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col-md-4">
        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-search-location me-2"></i>IP Detection Status</h5>
            </div>
            <div class="card-body">
                <div class="mb-3">
                    <strong>Server IP:</strong> <code><?= $current_ip ?></code>
                </div>
                <div class="mb-3">
                    <strong>Detected IP:</strong> <code class="<?= $real_ip !== $current_ip ? 'text-warning' : 'text-success' ?>"><?= $real_ip ?: 'Unable to detect' ?></code>
                </div>
                <?php if (in_array($current_ip, ['127.0.0.1', '::1']) || $current_ip === 'localhost' || !$real_ip): ?>
                    <div class="alert alert-info p-2 mb-3 small">
                        <i class="fas fa-info-circle me-1"></i>
                        <strong>Development Environment Detected:</strong> IP detection may not work as expected on localhost. This is normal and will function correctly in production with real client IPs.
                    </div>
                <?php elseif ($real_ip !== $current_ip): ?>
                    <div class="alert alert-info p-2 mb-3 small">
                        <i class="fas fa-info-circle me-1"></i>
                        Using proxy headers to detect real client IP.
                    </div>
                <?php endif; ?>

                <h6>Behind Reverse Proxy</h6>
                <div class="mb-2">
                    <strong>Status:</strong>
                    <span class="badge bg-<?= $proxy_enabled ? 'success' : 'secondary' ?>">
                        <?= $proxy_enabled ? 'Enabled' : 'Disabled' ?>
                    </span>
                </div>
                <div class="mb-3">
                    <strong>Trusted Sources:</strong>
                    <span class="badge bg-info"><?= $proxy_config_count ?> configured</span>
                </div>
                <small class="text-muted">Enable this if UserSpice sits behind a load balancer, CDN, or reverse proxy that forwards client IPs via headers.</small>

                <?php if ($proxy_enabled && empty($proxy_configs)): ?>
                    <div class="alert alert-warning p-2 small mt-2">
                        The 'Behind Proxy' setting is enabled, but no trusted proxy sources are configured. IP detection will fall back to the standard server IP.
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <?php if (!empty($proxy_configs)): ?>
            <div class="card mb-4">
                <div class="card-header">
                    <h5 class="mb-0"><i class="fas fa-server me-2"></i>Trusted Proxy Sources</h5>
                </div>
                <div class="card-body p-0">
                    <?php foreach ($proxy_configs as $conf): ?>
                        <div class="proxy-item p-3 border-bottom">
                            <div class="d-flex justify-content-between align-items-start">
                                <div>
                                    <strong><?= htmlspecialchars($conf->proxy_ip) ?></strong>
                                    <br><small class="text-muted">Header: <?= htmlspecialchars($conf->header_name) ?></small>
                                    <br><small class="text-muted">Priority: <?= $conf->priority ?></small>
                                </div>
                                <form method="post" class="d-inline">
                                    <input type="hidden" name="csrf" value="<?= Token::generate() ?>">
                                    <input type="hidden" name="proxy_id" value="<?= $conf->id ?>">
                                    <button type="submit" name="delete_proxy" class="btn btn-sm btn-outline-danger" onclick="return confirm('Remove this proxy configuration?')">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </form>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </div>
        <?php endif; ?>

        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-activity me-2"></i>Recent Activity (1 Hour)</h5>
            </div>
            <div class="card-body">
                <?php if (empty($recent_activity)): ?>
                    <div class="text-center text-muted p-3">
                        <i class="fas fa-check-circle fa-2x mb-2"></i>
                        <p class="mb-0">No recent rate limiting activity</p>
                    </div>
                <?php else: ?>
                    <div style="max-height: 250px; overflow-y: auto;">
                        <?php foreach ($recent_activity as $activity): ?>
                            <div class="d-flex justify-content-between align-items-center py-2 border-bottom">
                                <div>
                                    <strong><?= ucwords(str_replace('_', ' ', $activity->action)) ?></strong>
                                    <br><small class="text-muted"><?= date('H:i:s', strtotime($activity->last_attempt)) ?></small>
                                </div>
                                <div class="text-end">
                                    <span class="badge bg-<?= $activity->success ? 'success' : 'danger' ?>">
                                        <?= $activity->count ?> <?= $activity->success ? 'Success' : 'Failed' ?>
                                    </span>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
            </div>
        </div>

        <div class="card mb-4">
            <div class="card-header">
                <h5 class="mb-0"><i class="fas fa-lightbulb me-2"></i>Security Recommendations</h5>
            </div>
            <div class="card-body">
                <?php
                $recommendations = [];

                if ($using_default_rate_limits) {
                    $recommendations[] = [
                        'level' => 'danger',
                        'icon' => 'fas fa-exclamation-triangle',
                        'text' => 'Customize rate limits immediately - defaults are insecure'
                    ];
                }

                if (!$proxy_enabled && in_array($current_ip, ['127.0.0.1', '::1'])) {
                    $recommendations[] = [
                        'level' => 'warning',
                        'icon' => 'fas fa-server',
                        'text' => 'Consider enabling the \'Behind a Proxy\' setting if you are using a load balancer.'
                    ];
                }

                if (isset($rateLimits['login_attempt']['ip_max']) && $rateLimits['login_attempt']['ip_max'] > 100) {
                    $recommendations[] = [
                        'level' => 'warning',
                        'icon' => 'fas fa-shield-alt',
                        'text' => 'Login attempt IP limits seem high - consider lowering'
                    ];
                }

                $total_records = $db->query("SELECT COUNT(*) as count FROM us_rate_limits")->first()->count ?? 0;
                if ($total_records > 10000) {
                    $recommendations[] = [
                        'level' => 'info',
                        'icon' => 'fas fa-broom',
                        'text' => 'Consider cleaning up old rate limit logs'
                    ];
                }

                if (empty($recommendations)) {
                    $recommendations[] = [
                        'level' => 'success',
                        'icon' => 'fas fa-check-circle',
                        'text' => 'Rate limiting configuration looks good!'
                    ];
                }
                ?>

                <?php foreach ($recommendations as $rec): ?>
                    <div class="alert alert-<?= $rec['level'] ?> p-2 mb-2">
                        <i class="<?= $rec['icon'] ?> me-2"></i><?= $rec['text'] ?>
                    </div>
                <?php endforeach; ?>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="autoConfigModal" tabindex="-1" aria-labelledby="autoConfigModalLabel" aria-hidden="true">
    <div class="modal-dialog modal-lg">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="autoConfigModalLabel"><i class="fas fa-magic me-2"></i>Auto-Configure Rate Limit Profiles</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <div class="modal-body">
                <p>Select a profile that best matches your site's needs. This will automatically populate the form with recommended values. You can fine-tune them before saving.</p>

                <div class="list-group">
                    <label class="list-group-item list-group-item-action">
                        <div class="d-flex w-100 justify-content-between">
                            <h6 class="mb-1"><input type="radio" name="configProfile" value="high_security" class="me-2">High Security</h6>
                            <small><span class="badge bg-danger">Strict</span></small>
                        </div>
                        <p class="mb-1 small">Prioritizes security above all. Very strict limits, best for low-traffic sites or applications handling highly sensitive data. May be inconvenient for some users.</p>
                    </label>
                    <label class="list-group-item list-group-item-action active">
                        <div class="d-flex w-100 justify-content-between">
                            <h6 class="mb-1"><input type="radio" name="configProfile" value="balanced" class="me-2" checked>Balanced (Recommended)</h6>
                            <small><span class="badge bg-success">Recommended</span></small>
                        </div>
                        <p class="mb-1 small">A sensible balance between security and user convenience. Ideal for most websites, including forums, blogs, and standard business sites.</p>
                    </label>
                    <label class="list-group-item list-group-item-action">
                        <div class="d-flex w-100 justify-content-between">
                            <h6 class="mb-1"><input type="radio" name="configProfile" value="high_traffic" class="me-2">High Traffic / Convenience</h6>
                            <small><span class="badge bg-primary">Flexible</span></small>
                        </div>
                        <p class="mb-1 small">More lenient limits to prevent legitimate users on high-traffic sites (e.g., busy forums, e-commerce) from being accidentally locked out. Accepts a slightly higher security risk.</p>
                    </label>
                    <label class="list-group-item list-group-item-action">
                        <div class="d-flex w-100 justify-content-between">
                            <h6 class="mb-1"><input type="radio" name="configProfile" value="lax" class="me-2">Lax Policy</h6>
                            <small><span class="badge bg-warning text-dark">Not Recommended</span></small>
                        </div>
                        <p class="mb-1 small">Provides minimal protection. Use only for development, testing, or internal applications where abuse is not a concern. <strong>Do not use in production.</strong></p>
                    </label>
                </div>
            </div>
            <div class="modal-footer">
                <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                <button type="button" class="btn btn-primary" id="applyProfileBtn"><i class="fas fa-check me-2"></i>Apply Settings</button>
            </div>
        </div>
    </div>
</div>

<div class="modal fade" id="addProxyModal" tabindex="-1" aria-labelledby="addProxyModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="addProxyModalLabel">Add Trusted Proxy Source</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post">
                <div class="modal-body">
                    <input type="hidden" name="csrf" value="<?= Token::generate() ?>">

                    <div class="mb-3">
                        <label for="proxy_ip" class="form-label">Proxy IP Address or CIDR</label>
                        <input type="text" class="form-control" id="proxy_ip" name="proxy_ip" required
                            placeholder="192.168.1.1 or 10.0.0.0/8">
                        <div class="form-text">Single IP address or CIDR notation for IP ranges</div>
                    </div>

                    <div class="mb-3">
                        <label for="header_name" class="form-label">Header Name</label>
                        <select class="form-select" id="header_name" name="header_name" required onchange="toggleCustomHeader()">
                            <option value="X-Forwarded-For">X-Forwarded-For</option>
                            <option value="X-Real-IP">X-Real-IP</option>
                            <option value="CF-Connecting-IP">CF-Connecting-IP (Cloudflare)</option>
                            <option value="X-Client-IP">X-Client-IP</option>
                            <option value="X-Originating-IP">X-Originating-IP</option>
                            <option value="True-Client-IP">True-Client-IP (Akamai)</option>
                            <option value="X-Azure-ClientIP">X-Azure-ClientIP (Azure)</option>
                            <option value="X-Cluster-Client-IP">X-Cluster-Client-IP (GCP)</option>
                            <option value="custom">Custom Header...</option>
                        </select>
                        <div class="form-text">HTTP header used to detect the real client IP</div>
                    </div>

                    <div class="mb-3 d-none" id="customHeaderDiv">
                        <label for="custom_header_name" class="form-label">Custom Header Name</label>
                        <input type="text" class="form-control" id="custom_header_name" name="custom_header_name"
                            placeholder="X-Your-Custom-Header">
                        <div class="form-text">Enter the exact header name. Standard practice is Title-Case, e.g. X-Real-IP.</div>
                    </div>

                    <div class="mb-3">
                        <label for="priority" class="form-label">Priority</label>
                        <input type="number" class="form-control" id="priority" name="priority" value="0" min="0" max="100">
                        <div class="form-text">Lower numbers = higher priority (checked first)</div>
                    </div>

                    <div class="alert alert-warning p-2">
                        <strong>Security Note:</strong> Only add IP addresses you trust completely.
                        Malicious proxies can spoof client IPs and bypass rate limiting.
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" name="add_proxy" class="btn btn-primary">Add Trusted Proxy</button>
                </div>
            </form>
        </div>
    </div>
</div>

<div class="modal fade" id="cleanupModal" tabindex="-1" aria-labelledby="cleanupModalLabel" aria-hidden="true">
    <div class="modal-dialog">
        <div class="modal-content">
            <div class="modal-header">
                <h5 class="modal-title" id="cleanupModalLabel">Cleanup Rate Limit Logs</h5>
                <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
            </div>
            <form method="post">
                <div class="modal-body">
                    <input type="hidden" name="csrf" value="<?= Token::generate() ?>">

                    <div class="mb-3">
                        <label for="cleanup_hours" class="form-label">Remove records older than:</label>
                        <div class="input-group">
                            <input type="number" class="form-control" id="cleanup_hours" name="cleanup_hours"
                                value="24" min="1" max="720" required>
                            <span class="input-group-text">hours</span>
                        </div>
                    </div>

                    <?php
                    $total_records = $db->query("SELECT COUNT(*) as count FROM us_rate_limits")->first()->count ?? 0;
                    $old_records = $db->query("SELECT COUNT(*) as count FROM us_rate_limits WHERE attempt_time < DATE_SUB(NOW(), INTERVAL 24 HOUR)")->first()->count ?? 0;
                    ?>

                    <div class="alert alert-info p-2">
                        <strong>Current Status:</strong><br>
                        Total records: <?= $total_records ?><br>
                        Records older than 24 hours: <?= $old_records ?>
                    </div>

                    <div class="alert alert-warning p-2">
                        <strong>Note:</strong> This action cannot be undone. Only remove records you're sure you don't need for analysis.
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" name="cleanup_logs" class="btn btn-warning">
                        <i class="fas fa-broom me-2"></i>Cleanup Now
                    </button>
                </div>
            </form>
        </div>
    </div>
</div>

<script>
    // Define functions first to avoid reference errors
    function showAddProxyModal() {
        const modal = new bootstrap.Modal(document.getElementById('addProxyModal'));
        modal.show();
    }

    function showCleanupModal() {
        const modal = new bootstrap.Modal(document.getElementById('cleanupModal'));
        modal.show();
    }

    function showAutoConfigModal() {
        const modal = new bootstrap.Modal(document.getElementById('autoConfigModal'));
        modal.show();
    }


    function toggleCustomHeader() {
        const select = document.getElementById('header_name');
        const customDiv = document.getElementById('customHeaderDiv');
        const customInput = document.getElementById('custom_header_name');

        if (select.value === 'custom') {
            customDiv.classList.remove('d-none');
            customInput.required = true;
            customInput.focus();
        } else {
            customDiv.classList.add('d-none');
            customInput.required = false;
            customInput.value = '';
        }
    }

    function applyRateLimitProfile(profileName) {
        const profiles = {
            high_security: {
                'login_attempt': {
                    'ip_max': 20,
                    'ip_window': 900,
                    'user_max': 5,
                    'user_window': 900,
                    'total_max': 40,
                    'total_window': 900
                },
                'password_reset_request': {
                    'ip_max': 5,
                    'ip_window': 3600,
                    'email_max': 2,
                    'email_window': 3600,
                    'total_max': 10,
                    'total_window': 3600
                },
                'registration_attempt': {
                    'ip_max': 5,
                    'ip_window': 3600,
                    'total_max': 10,
                    'total_window': 3600
                },
                'passkey_verify': {
                    'ip_max': 20,
                    'ip_window': 900,
                    'user_max': 5,
                    'user_window': 900,
                    'credential_max': 3,
                    'credential_window': 900,
                    'total_max': 30,
                    'total_window': 900
                },
                'default': {
                    'ip_max': 20,
                    'ip_window': 600,
                    'user_max': 10,
                    'user_window': 600,
                    'total_max': 40,
                    'total_window': 600
                }
            },
            balanced: { // This is the "Recommended" set
                'login_attempt': {
                    'ip_max': 50,
                    'ip_window': 900,
                    'user_max': 10,
                    'user_window': 300,
                    'total_max': 100,
                    'total_window': 900
                },
                'password_reset_request': {
                    'ip_max': 20,
                    'ip_window': 3600,
                    'email_max': 5,
                    'email_window': 3600,
                    'total_max': 30,
                    'total_window': 3600
                },
                'registration_attempt': {
                    'ip_max': 10,
                    'ip_window': 3600,
                    'total_max': 15,
                    'total_window': 3600
                },
                'passkey_verify': {
                    'ip_max': 30,
                    'ip_window': 600,
                    'user_max': 10,
                    'user_window': 600,
                    'credential_max': 6,
                    'credential_window': 900,
                    'total_max': 50,
                    'total_window': 900
                },
                'default': {
                    'ip_max': 50,
                    'ip_window': 300,
                    'user_max': 25,
                    'user_window': 300,
                    'total_max': 100,
                    'total_window': 300
                }
            },
            high_traffic: {
                'login_attempt': {
                    'ip_max': 100,
                    'ip_window': 600,
                    'user_max': 15,
                    'user_window': 300,
                    'total_max': 200,
                    'total_window': 600
                },
                'password_reset_request': {
                    'ip_max': 50,
                    'ip_window': 1800,
                    'email_max': 10,
                    'email_window': 1800,
                    'total_max': 75,
                    'total_window': 1800
                },
                'registration_attempt': {
                    'ip_max': 30,
                    'ip_window': 3600,
                    'total_max': 60,
                    'total_window': 3600
                },
                'passkey_verify': {
                    'ip_max': 60,
                    'ip_window': 600,
                    'user_max': 15,
                    'user_window': 600,
                    'credential_max': 8,
                    'credential_window': 900,
                    'total_max': 100,
                    'total_window': 900
                },
                'default': {
                    'ip_max': 100,
                    'ip_window': 300,
                    'user_max': 50,
                    'user_window': 300,
                    'total_max': 200,
                    'total_window': 300
                }
            },
            lax: {
                'login_attempt': {
                    'ip_max': 500,
                    'ip_window': 300,
                    'user_max': 100,
                    'user_window': 300,
                    'total_max': 1000,
                    'total_window': 300
                },
                'password_reset_request': {
                    'ip_max': 100,
                    'ip_window': 600,
                    'email_max': 50,
                    'email_window': 600,
                    'total_max': 200,
                    'total_window': 600
                },
                'registration_attempt': {
                    'ip_max': 100,
                    'ip_window': 600,
                    'total_max': 200,
                    'total_window': 600
                },
                'passkey_verify': {
                    'ip_max': 200,
                    'ip_window': 300,
                    'user_max': 100,
                    'user_window': 300,
                    'credential_max': 50,
                    'credential_window': 300,
                    'total_max': 400,
                    'total_window': 300
                },
                'default': {
                    'ip_max': 500,
                    'ip_window': 300,
                    'user_max': 250,
                    'user_window': 300,
                    'total_max': 1000,
                    'total_window': 300
                }
            }
        };

        const selectedProfile = profiles[profileName];
        if (!selectedProfile) return;

        // Apply recommended values to form
        const allInputs = document.querySelectorAll('#rateLimitsForm input[type="number"]');
        allInputs.forEach(input => {
            const name = input.name;
            const match = name.match(/limits\[(.*?)\]\[(.*?)\]/);
            if (match) {
                const action = match[1];
                const key = match[2];

                let value;
                if (selectedProfile[action] && typeof selectedProfile[action][key] !== 'undefined') {
                    value = selectedProfile[action][key];
                } else if (selectedProfile['default'] && typeof selectedProfile['default'][key] !== 'undefined') {
                    // Fallback to default values for actions not explicitly defined in the profile
                    value = selectedProfile['default'][key];
                } else {
                    // If no value found, skip to not erase existing settings for unmentioned actions
                    return;
                }

                input.value = value;
                input.style.backgroundColor = '#d4edda'; // Highlight changed fields
                input.style.transition = 'background-color 0.5s ease';
                setTimeout(() => {
                    input.style.backgroundColor = '';
                }, 2500);
            }
        });

        const button = document.querySelector('button[name="update_rate_limits"]');
        if (button) {
            button.classList.add('btn-warning');
            button.innerHTML = '<i class="fas fa-exclamation-triangle me-2"></i>Unsaved Changes';
        }
    }


    // Event listeners after DOM is ready
    document.addEventListener('DOMContentLoaded', function() {
        // Attach listeners for quick action buttons
        document.getElementById('addProxyBtn')?.addEventListener('click', showAddProxyModal);
        document.getElementById('cleanupLogsBtn')?.addEventListener('click', showCleanupModal);
        document.querySelectorAll('.autoConfigBtn').forEach(btn => {
            btn.addEventListener('click', showAutoConfigModal);
        });

        // **MODIFIED SCRIPT**
        // This now applies the profile and submits the form
        document.getElementById('applyProfileBtn')?.addEventListener('click', function() {
            const selectedProfile = document.querySelector('input[name="configProfile"]:checked').value;
            applyRateLimitProfile(selectedProfile);

            // Submit the form to save the changes
            const rateLimitsForm = document.getElementById('rateLimitsForm');
            if (rateLimitsForm) {
                // Add the update_rate_limits input to ensure the server-side code recognizes the submission
                let submitInput = rateLimitsForm.querySelector('input[name="update_rate_limits"]');
                if (!submitInput) {
                    submitInput = document.createElement('input');
                    submitInput.type = 'hidden';
                    submitInput.name = 'update_rate_limits';
                    submitInput.value = '1';
                    rateLimitsForm.appendChild(submitInput);
                }
                rateLimitsForm.submit();
            }
        });

        // Auto-save indicator
        const rateLimitsForm = document.getElementById('rateLimitsForm');
        if (rateLimitsForm) {
            rateLimitsForm.addEventListener('input', function() {
                const button = document.querySelector('button[name="update_rate_limits"]');
                if (button && !button.classList.contains('btn-warning')) {
                    button.classList.add('btn-warning');
                    button.innerHTML = '<i class="fas fa-exclamation-triangle me-2"></i>Unsaved Changes';
                }
            });
        }

        // Convert seconds to human readable on input blur
        const windowInputs = document.querySelectorAll('input[name*="window"]');
        windowInputs.forEach(input => {
            input.addEventListener('blur', function() {
                const seconds = parseInt(this.value);
                if (!isNaN(seconds) && seconds >= 60) {
                    const minutes = Math.floor(seconds / 60);
                    const remainingSeconds = seconds % 60;
                    let display = `${minutes}m`;
                    if (remainingSeconds > 0) display += ` ${remainingSeconds}s`;

                    // Show tooltip with human readable time
                    this.title = `${display} (${seconds} seconds)`;
                } else {
                    this.title = `${seconds} seconds`;
                }
            });
        });

        // Activate radio button when clicking on the list group item
        const profileLabels = document.querySelectorAll('#autoConfigModal .list-group-item');
        profileLabels.forEach(label => {
            label.addEventListener('click', function(e) {
                // Uncheck all and remove active class
                profileLabels.forEach(l => {
                    l.classList.remove('active');
                    l.querySelector('input[type="radio"]').checked = false;
                });
                // Check the clicked one and add active class
                this.classList.add('active');
                this.querySelector('input[type="radio"]').checked = true;
            });
        });
    });
</script>